//////////////////////////////////////////////////////////////////////////////
//
//  GscApi.h
//
//  This is the header file for the General Standards Corp API implementation.
//
#ifndef GSCAPI_H
#define GSCAPI_H

#ifdef __cplusplus
extern "C" {
#endif

//
// Macro to export import the DLL routines
//
#ifndef GSCAPI_API

#ifdef PLX_LINUX
#define GSCAPI_API 
#else
#define GSCAPI_API __declspec(dllimport)
#endif

#endif

#ifdef PLX_LINUX
#define CALLBACK    // null windows function calling convention for linux.
#endif

// set deprecated extended attribute for c compilers
#ifdef PLX_LINUX

#define MSC_ATTRIB_DEPRECATED
#if (__GNUC__ > 3 || (__GNUC__ == 3 && __GNUC_MINOR__ >= 1)) 
#define GCC_ATTRIB_DEPRECATED __attribute__ ((deprecated)) 
#else 
#define GCC_ATTRIB_DEPRECATED 
#endif 

#else
#define GCC_ATTRIB_DEPRECATED 
#if _MSC_VER > 1200
#define MSC_ATTRIB_DEPRECATED __declspec(deprecated)
#else
#define MSC_ATTRIB_DEPRECATED
#endif

#endif
//
// This macro defines the api version number
//
#define GSC_API_VERSION	0x00010502		// 1.05.02

//
// Macro that determines the maximum number of boards allowed in a system
//
#define	MAXIMUM_NUMBER_OF_BOARDS		10

//
// Macro that defines the number of channels per board
//
#define NUMBER_OF_CHANNELS				4

//
// Macro that defines the maximum number of entries allowed in the DMA queue.
//
#define MAXIMUM_DMA_QUEUE_ITEMS		   100

//
// Macro that defines the maximum clock input to the Zilog Chip.
//
#define MAXIMUM_USC_CLOCK			   20000000

//
// Structure to hold info about the SIO4 boards in the system
//
typedef struct _GSC_DEVICES_STRUCT
{
	int		busNumber;
	int		slotNumber;
	int		vendorId;
	int		deviceId;
	char	serialNumber[25];
} GSC_DEVICES_STRUCT;

//
// Structure to hold information about a specific board in the system
//
typedef struct _BOARD_INFO
{
	char	apiVersion[20];
	char	driverVersion[20];
	char	fpgaVersion[20];
	char	boardType[50];
} BOARD_INFO, *PBOARD_INFO;


//
// Structure to hold the current configuration of an Analog channel
//
typedef struct _GSC_ANALOG_CONFIG
{
	int inputMode;						      // Inputs - Normal, High, Low
	int inputType;						      // Differential, single ended
	int inputRange;						      // +-2.5, +-5.0, Bipolar, Unipolar, etc
	int dataFormat;						      // Offset Binary or 2's complement
	int externalSync;					      // Enabled/Disabled
	int dataWidth;					        // 16 bits - 24 bits
    int dataPacking;                    // 2 samples per 32 bits in input data fifo
	int scanMarkingMode;				    // Enabled/Disabled
	int scanMarkingValue;
	int sampleClockSource;				  // Internal, External, etc.
	int masterClockFrequency;			  // Frequency of board oscillator

} GSC_ANALOG_CONFIG, *PGSC_ANALOG_CONFIG;


//
// Structure to hold the current configuration of an ASYNC channel
//
typedef struct _GSC_ASYNC_CONFIG
{
	// Channel Configuration Variables
	int	bitRate;						        // Baud rate for the channel.
	int encoding;						        // Encoding - NRZ, BiPhase, etc.
	int protocol;						        // Bus Protocol - RS485, RS232, V.35, etc.
	int termination;					      // Termination Resistors enabled/disabled
	int parity;							        // Parity mode - None, Even, Odd, etc.
	int stopBits;						        // Stop bits - 0, 1, 1.5, 2

	// Transmitter Configuration Variables
	int txStatus;						        // Transmitter Enabled/Disabled
	int	txCharacterLength;				  // Bits per Tx character
	int txClockSource;					    // Clock source for the transmitter

	// Receiver Configuration Variables
	int rxStatus;						        // Receiver Enabled/Disabled
	int	rxCharacterLength;				  // Bits per Rx character
	int rxClockSource;					    // Clock source for the receiver

	// Pin Configuration Variables
	int interfaceMode;					    // DTE or DCE interface
	int	txDataPinMode;					    // Auto (system use) or GPIO
	int	rxDataPinMode;					    // Auto (system use) or GPIO
	int	txClockPinMode;					    // Auto (system use) or GPIO
	int	rxClockPinMode;					    // Auto (system use) or GPIO
	int	ctsPinMode;						      // Auto (system use) or GPIO
	int	rtsPinMode;						      // Auto (system use) or GPIO
	int loopbackMode;					      // None, internal, or external loop back

	//FIFO features
	int fifoTimetagByte;				    // add 24-bit timetag for each byte in rx fifo
	int fifoParityErrorByte;			  // add 8-bit parity error for each byte in rx fifo

} GSC_ASYNC_CONFIG, *PGSC_ASYNC_CONFIG;

//
// Structure to hold the current configuration of an HDLC channel
//
typedef struct _GSC_HDLC_CONFIG
{
	// Channel Configuration Variables
	int	bitRate;						// Baud rate for the channel.
	int encoding;						// Encoding - NRZ, BiPhase, etc.
	int protocol;						// Bus Protocol - RS485, RS232, V.35, etc.
	int termination;					// Termination Resistors enabled/disabled
	int parity;							// Parity mode - None, Even, Odd, etc.
	int crcMode;						// CRC Type - Disabled, CCITT, etc.
	int	crcInitialValue;				// Initial CRC - All 1 or 0

	// Transmitter Configuration Variables
	int txStatus;						// Transmitter Enabled/Disabled
	int	txCharacterLength;				// Bits per Tx character

	int	txUnderRun;						// What to do on a Tx underrun
	int txPreamble;						// Length of Preamble
	int	txPreamblePattern;				// Type of Preamble
	int txSharedZero;					// Share 0s in adjacent flags?
	int txClockSource;					// Clock source for the transmitter
	int txIdleCondition;				// What to transmit when the line is idle

	// Receiver Configuration Variables
	int rxStatus;						// Receiver Enabled/Disabled
	int	rxCharacterLength;				// Bits per Rx character
	int rxAddrSearchMode;				// Rx address search mode
	int	rxAddress;						// Address to search for
	int rxClockSource;					// Clock source for the receiver

	// Pin Configuration Variables
	int interfaceMode;					// DTE or DCE interface
	int	txDataPinMode;					// Auto (system use) or GPIO
	int	rxDataPinMode;					// Auto (system use) or GPIO
	int	txClockPinMode;					// Auto (system use) or GPIO
	int	rxClockPinMode;					// Auto (system use) or GPIO
	int	ctsPinMode;						// Auto (system use) or GPIO
	int	rtsPinMode;						// Auto (system use) or GPIO
	int loopbackMode;					// None, internal, or external loop back

	// Misc Configuration Variables
	int	packetFraming;					// Internal use only, leave enabled

	//FIFO features
	int fifoTimetagByte;				// add 24-bit timetag for each byte in rx fifo
	int fifoParityErrorByte;		// add 8-bit parity error for each byte in rx fifo

} GSC_HDLC_CONFIG, *PGSC_HDLC_CONFIG;

//
// Structure to hold the current configuration of an SYNC channel
//
typedef struct _GSC_SYNC_CONFIG
{
	// Channel Configuration Variables
	int	bitRate;						// Baud rate for the channel.
	int encoding;						// Encoding - NRZ, NRZB
	int protocol;						// Bus Protocol - RS485, RS232, V.35, etc.
	int termination;					// Termination Resistors enabled/disabled

	// Transmitter Configuration Variables
	int txStatus;						// Transmitter Enabled/Disabled
	int	txCharacterLength;				// Bits per Tx character
	int	txGapLength;					// Bits between Tx characters
	int txClockSource;					// Clock source for the transmitter
	int txClockEdge;					// Clock edge for the transmitter
	int txEnvPolarity;					// Envelope polarity for the transmitter
	int txIdleCondition;				// What to transmit when the line is idle
	int txClockIdleCondition;			// What to do with the clock when the line is idle
	int	txMsbLsb;						// Bit order for transmitter

	// Receiver Configuration Variables
	int rxStatus;						// Receiver Enabled/Disabled
	int rxClockSource;					// Clock source for the receiver
	int rxClockEdge;					// Clock edge for the receiver
	int rxEnvPolarity;					// Envelope polarity for the receiver
	int	rxMsbLsb;						// Bit order for receiver

	// Pin Configuration Variables
	int interfaceMode;					// DTE or DCE interface
	int	txDataPinMode;					// Auto (system use) or GPIO
	int	rxDataPinMode;					// Auto (system use) or GPIO
	int	txClockPinMode;					// Auto (system use) or GPIO
	int	rxClockPinMode;					// Auto (system use) or GPIO
	int	txEnvPinMode;					// Auto (system use) or GPIO
	int	rxEnvPinMode;					// Auto (system use) or GPIO
	int loopbackMode;					// None, internal, or external loop back

	// Misc Configuration Variables
	int	packetFraming;					// Internal use only, leave disabled

} GSC_SYNC_CONFIG, *PGSC_SYNC_CONFIG;

//
// Structure to hold the current configuration of a BiSync channel
//
typedef struct _GSC_BISYNC_CONFIG
{
	// Channel Configuration Variables
	int	bitRate;						// Baud rate for the channel.
	int encoding;						// Encoding - NRZ, BiPhase, etc.
	int protocol;						// Bus Protocol - RS485, RS232, V.35, etc.
	int termination;					// Termination Resistors enabled/disabled
	int parity;							// Parity mode - None, Even, Odd, etc.
	int crcMode;						// CRC Type - Disabled, CCITT, etc.
	int	crcInitialValue;				// Initial CRC - All 1 or 0

	// Transmitter Configuration Variables
	int txStatus;						// Transmitter Enabled/Disabled
	int	txCharacterLength;				// Bits per Tx character
	int txClockSource;					// Clock source for the transmitter
	int txIdleCondition;				// What to transmit when the line is idle
	int	txSyncWord;						// Two character sync pattern
	int	txUnderRun;						// What to do on a Tx underrun
	int txPreamble;						// Enable/disable preamble before opening sync
	int	txPreambleLength;				// Preamble length - 8,16,32,64 bits
	int txPreamblePattern;				// Preamble pattern - all zeros, all ones, etc.
	int txShortSync;					// Length of sync character - 8 bits or same as txCharacterLength

	// Receiver Configuration Variables
	int rxStatus;						// Receiver Enabled/Disabled
	int rxClockSource;					// Clock source for the receiver
	int	rxCharacterLength;				// Bits per Rx character
	int	rxSyncWord;						// Two character sync pattern
	int rxSyncStrip;					// Sync character stripping enable/disable
	int rxShortSync;					// Length of sync character - 8 bits or same as rxCharacterLength

	// Pin Configuration Variables
	int interfaceMode;					// DTE or DCE interface
	int	txDataPinMode;					// Auto (system use) or GPIO
	int	rxDataPinMode;					// Auto (system use) or GPIO
	int	txClockPinMode;					// Auto (system use) or GPIO
	int	rxClockPinMode;					// Auto (system use) or GPIO
	int	ctsPinMode;						// Auto (system use) or GPIO
	int	rtsPinMode;						// Auto (system use) or GPIO
	int loopbackMode;					// None, internal, or external loop back

	// Misc Configuration Variables
	int	packetFraming;					// Internal use only, leave disabled
} GSC_BISYNC_CONFIG, *PGSC_BISYNC_CONFIG;

//
// Structure to hold the current configuration of a BiSync16 channel.
// This mode is available only on SIO4 -BISYNC16 boards.
//
typedef struct _GSC_BISYNC16_CONFIG
{
	// Channel Configuration Variables
	int	bitRate;						// Baud rate for the channel.
	int encoding;						// Encoding - NRZ, BiPhase, etc.
	int protocol;						// Bus Protocol - RS485, RS232, V.35, etc.
	int termination;					// Termination Resistors enabled/disabled

	// Transmitter Configuration Variables
	int txStatus;						// Transmitter Enabled/Disabled
	int txIdleCondition;				// What to transmit when the line is idle
	int	txSyncWord;						// Two character sync pattern
	int txBitOrder;
	int	txByteOrder;

	// Receiver Configuration Variables
	int rxStatus;						// Receiver Enabled/Disabled
	int	rxSyncWord;						// Two character sync pattern
	int	maxRxCount;

	// Pin Configuration Variables
	int interfaceMode;					// DTE or DCE interface
	int	txDataPinMode;					// Auto (system use) or GPIO
	int	rxDataPinMode;					// Auto (system use) or GPIO
	int	txClockPinMode;					// Auto (system use) or GPIO
	int	rxClockPinMode;					// Auto (system use) or GPIO
	int	ctsPinMode;						// Auto (system use) or GPIO
	int	rtsPinMode;						// Auto (system use) or GPIO
	int loopbackMode;					// None, internal, or external loop back

} GSC_BISYNC16_CONFIG, *PGSC_BISYNC16_CONFIG;

//
// Structure to hold the current configuration of a RASYNC channel
//
typedef struct _GSC_RASYNC_CONFIG
{
	// Channel Configuration Variables
	int	bitRate;						// Baud rate for the channel.
	int encoding;						// Encoding - NRZ, BiPhase, etc.
	int protocol;						// Bus Protocol - RS485, RS232, V.35, etc.
	int termination;					// Termination Resistors enabled/disabled
	int parity;							// Parity mode - None, Even, Odd, etc.
	int stopBits;						// Stop bits - 0, 1, 2, etc.
	int	activeCircuit;					// Auto, A, B, C, or D

	// Transmitter Configuration Variables
	int txStatus;						// Transmitter Enabled/Disabled
	int	txGapLength;					// Bits between Tx characters

	// Receiver Configuration Variables
	int rxStatus;						// Receiver Enabled/Disabled

	int rxSaveDataOnError;

	// Pin Configuration Variables
	int interfaceMode;					// DTE or DCE interface
	int loopbackMode;					// None, internal, or external loop back

	// Misc Configuration Variables
	int	packetFraming;					// Internal use only, leave disabled
} GSC_RASYNC_CONFIG, *PGSC_RASYNC_CONFIG;


//
// Typedef for the callback functions
//
#ifdef __linux__
typedef void GSC_CB_FUNCTION(int boardNumber, int channel, int interrupt);
#else
typedef void _stdcall GSC_CB_FUNCTION(int boardNumber, int channel, int interrupt);
#endif
//
// Tokens for the Analog Card Registers
//
enum GSC_ANALOG_REGISTERS
{
	GSC_ANALOG_BCR	  = 0x8000,		// Board Control Register
	GSC_ANALOG_ICR	  = 0x8004,		// Interrupt Control
	GSC_ANALOG_IDB	  = 0x8008,		// Input Data Buffer
	GSC_ANALOG_IBC	  = 0x800c,		// Input Buffer Control
	GSC_ANALOG_RAG	  = 0x8010,		// Rate A Generator
	GSC_ANALOG_RBG	  = 0x8014,		// Rate B Generator
	GSC_ANALOG_BS	    = 0x8018,		// Buffer Size
	GSC_ANALOG_BURST  = 0x801c,		// Burst Size
	GSC_ANALOG_SSC	  = 0x8020,		// Scan and Sync Control
	GSC_ANALOG_ACA	  = 0x8024,		// Active Channel Assignment
	GSC_ANALOG_FWR	  = 0x8028,		// Firmware revision.
	GSC_ANALOG_AV	    = 0x802c,		// Autocal Values
	GSC_ANALOG_ARWR	  = 0x8030,		// Auxillary Read/Write Register
	GSC_ANALOG_ASIO   = 0x8034,		// Auxillary Sync I/O Control
	GSC_ANALOG_SMFV	  = 0x8038,		// Scan Marker First Value
	GSC_ANALOG_SMSV	  = 0x803c,		// Scan Marker Second Value
                                //
                                // 24sdi12-specific registers
  GSC_ANALOG_DIV    = 0x8040,   // Rate divisor register.
  GSC_ANALOG_RAR    = 0x8041    // Rate assignment register.
};

//
// Tokens for the SIO4 internal register addresses
//
enum GSC_ADDRESSES
{
	// Local Registers
	FW_REVISION_REG    = 0x0000,			// Firmware Revision Register
	BOARD_CONTROL_REG  = 0x0004,			// Board Control Register
	BOARD_STATUS_REG   = 0x0008,			// Board Status Register
	CLOCK_CONTROL_REG  = 0x000c,			// Clock Control Register
	TX_ALMOST_BASE_REG = 0x0010,			// Base value for the Tx Almost registers
	RX_ALMOST_BASE_REG = 0x0014,			// Base value for the Rx Almost registers
	DATA_FIFO_BASE_REG = 0x0018,			// Base value for the Tx/Rx data FIFOs
	CONTROL_STATUS_BASE_REG = 0x001c,		// Base value for the Control/Status registers
	SYNC_CHARACTER_BASE_REG = 0x0050,		// Base value for the Sync Byte Registers
	INTERRUPT_CONTROL_REG   = 0x0060,		// Interrupt Control Register
	INTERRUPT_STATUS_REG    = 0x0064,		// Interrupt Status/Clear Register
	INTERRUPT_EDGE_LEVEL_REG= 0x0068,		// Interrupt Edge/Level Register (RO)
	INTERRUPT_HI_LO_REG		= 0x006c,		// Interrupt High/Low, Rising/Falling register
	MAX_RX_COUNT_BASE_REG	= 0x0070,		// (-BiSync16 only) Base Value for Max Rx Count registers
	PIN_SOURCE_BASE_REG		= 0x0080,		// Base value for the Pin Source Registers
	PIN_STATUS_BASE_REG		= 0x0090,		// Base value for the Pin Status Registers
	POSC_RAM_ADDRESS_REG	= 0x00a0,		// Programmable Osc address register
	POSC_RAM_DATA_REG		= 0x00a4,		// Programmable Osc data register
	POSC_CONTROL_STATUS_REG	= 0x00a8,		// Programmable Osc control/status register
	POSC_RAM2_DATA_REG		= 0x00ac,		// Programmable Osc data register
	TX_COUNT_BASE_REG		= 0x00b0,
	RX_COUNT_BASE_REG		= 0x00c0,
	FIFO_COUNT_BASE_REG		= 0x00d0,		// Base value for the FIFO Count Registers
	FIFO_SIZE_BASE_REG		= 0x00e0,		// Base value for the FIFO Size Registers
	FEATURES_REG			= 0x00fc,		// Features Register

	// USC/Channel Registers
	USC_CCAR = 0x0000,	// Channel Command/Address Register
	USC_CMR  = 0x0002,	// Channel Mode Register
	USC_CCSR = 0x0004,	// Channel Command/Status Register
	USC_CCR  = 0x0006,	// Channel Control Register
	USC_TMDR = 0x000c,	// Test Mode Data Register
	USC_TMCR = 0x000e,	// Test Mode Control Register
	USC_CMCR = 0x0010,	// Clock Mode Control Register
	USC_HCR  = 0x0012,	// Hardware Configuration Register
	USC_IVR	 = 0x0014,	// Interrupt Vector Register
	USC_IOCR = 0x0016,	// I/O Control Register
	USC_ICR	 = 0x0018,	// Interrupt Control Register
	USC_DCCR = 0x001a,	// Daisy Chain Control Register
	USC_MISR = 0x001c,	// Misc Interrupt Status Register
	USC_SICR = 0x001e,	// Status Interrupt Control Register
	USC_RDR	 = 0x0020,	// Receive Data Register (RO)
	USC_RMR	 = 0x0022,	// Receive Mode Register
	USC_RCSR = 0x0024,	// Receive Command Status Register
	USC_RICR = 0x0026,	// Receive Interrupt Control Register
	USC_RSR	 = 0x0028,	// Receive Sync Register
	USC_RCLR = 0x002a,	// Receive Count Limit Register
	USC_RCCR = 0x002c,	// Receive Character Count Register
	USC_TC0R = 0x002e,	// Time Constant 0 Register
	USC_TDR	 = 0x0030,	// Transmit Data Register (WO)
	USC_TMR	 = 0x0032,	// Transmit Mode Register
	USC_TCSR = 0x0034,	// Transmit Command Status Register
	USC_TICR = 0x0036,	// Transmit Interrupt Control Register
	USC_TSR	 = 0x0038,	// Transmit Sync Register
	USC_TCLR = 0x003a,	// Transmit Count Limit Register
	USC_TCCR = 0x003c,	// Transmit Character Count Register
	USC_TC1R = 0x003e,	// Time Constant 1 Register
};

//
// Tokens for the various interrupt sources
//
enum GSC_INTR_TOKENS
{
	GSC_INTR_SYNC_DETECT			= 0x0001,
	GSC_INTR_USC					= 0x0002,
	GSC_INTR_TX_FIFO_EMPTY			= 0x0004,
	GSC_INTR_TX_FIFO_FULL			= 0x0008,
	GSC_INTR_TX_FIFO_ALMOST_EMPTY	= 0x0010,
	GSC_INTR_RX_FIFO_EMPTY			= 0x0020,
	GSC_INTR_RX_FIFO_FULL			= 0x0040,
	GSC_INTR_RX_FIFO_ALMOST_FULL	= 0x0080,
	GSC_INTR_TX_TRANSFER_COMPLETE	= 0x0100,
	GSC_INTR_RX_TRANSFER_COMPLETE	= 0x0200,
	GSC_INTR_RX_ENVELOPE			= GSC_INTR_SYNC_DETECT,		// -Sync card definition
  GSC_INTR_ANA_AUTOCALCOMPLETED = 0x0400,
  GSC_INTR_ANA_THRESHOLDLOWTOHIGH = 0x0800,
  GSC_INTR_ANA_THRESHOLDHIGHTOLOW = 0x1000,
  GSC_INTR_ANA_OVERUNDERFLOW = 0x2000,
  GSC_INTR_ANA_CHANNELSREADY = 0x4000,
};


//
// Misc tokens 
//
enum GSC_TOKENS
{
	GSC_MODE_UNINITIALIZED,
	GSC_MODE_ASYNC,
	GSC_MODE_HDLC,
	GSC_MODE_EXT_SYNC,
	GSC_MODE_SYNC,				// -Sync Boards Only
	GSC_MODE_SYNC_ENV,			// -Sync Boards Only
	GSC_MODE_ISO,
	GSC_MODE_ASYNC_CV,
	GSC_MODE_MONOSYNC,
	GSC_MODE_BISYNC,
	GSC_MODE_BISYNC16,			// -BiSync16 Boards Only
	GSC_MODE_TRANS_BISYNC,
	GSC_MODE_NBIF,
	GSC_MODE_802_3,
	GSC_MODE_RASYNC,
	GSC_MODE_LAST_MODE,

	GSC_LOOP_NONE,
	GSC_LOOP_INTERNAL,
	GSC_LOOP_EXTERNAL,

	GSC_ABORT,
	GSC_EXTENDED_ABORT,
	GSC_FLAG,
	GSC_CRC_FLAG,
	GSC_ADDR_ONLY,
	GSC_ADDR_CONTROL1,
	GSC_ADDR_CONTROL2,
	GSC_EXT_ADDR_CONTROL1,
	GSC_EXT_ADDR_CONTROL2,
	GSC_EXT_ADDR_EXT_CONTROL1,
	GSC_EXT_ADDR_EXT_CONTROL2,
	GSC_SYN1,
	GSC_SYN0_SYN1,
	GSC_CRC_SYN1,
	GSC_CRC_SYN0_SYN1,
	GSC_PREAMBLE_8BITS,
	GSC_PREAMBLE_16BITS,
	GSC_PREAMBLE_32BITS,
	GSC_PREAMBLE_64BITS,
	GSC_PREAMBLE_ALL_0,
	GSC_PREAMBLE_ALL_1,
	GSC_PREAMBLE_ALL_1_0,
	GSC_PREAMBLE_ALL_0_1,
	GSC_ENABLED,
	GSC_DISABLED,
	GSC_ENABLED_WITHAUTOENABLE,
	GSC_DATA_TRANSMIT,
	GSC_DATA_RECEIVE,

	GSC_IDLE_FLAGS,
	GSC_IDLE_ALTERNATING_01,
	GSC_IDLE_ALL_0,
	GSC_IDLE_ALL_1,
	GSC_IDLE_ALTERNATING_MARK_SPACE,
	GSC_IDLE_ALL_SPACE,
	GSC_IDLE_ALL_MARK,
	GSC_IDLE_ACTIVE,
	GSC_IDLE_NOT_ACTIVE,

	GSC_PIN_DTE,
	GSC_PIN_DCE,
	GSC_PIN_AUTO,
	GSC_PIN_GPIO,
	GSC_PIN_RX_CLOCK,				// Keep these enums in order
	GSC_PIN_RX_DATA,				//
	GSC_PIN_CTS,					//
	GSC_PIN_RX_ENV = GSC_PIN_CTS,	//
	GSC_PIN_DCD,					//
	GSC_PIN_TX_CLOCK,				//
	GSC_PIN_TX_DATA,				//
	GSC_PIN_RTS,					//
	GSC_PIN_TX_ENV = GSC_PIN_RTS,	//,
	GSC_PIN_AUXCLK,					// Keep these enums in order

	GSC_ENCODING_NRZ,
	GSC_ENCODING_NRZB,
	GSC_ENCODING_NRZI_MARK,
	GSC_ENCODING_NRZI_SPACE,
	GSC_ENCODING_BIPHASE_MARK,
	GSC_ENCODING_BIPHASE_SPACE,
	GSC_ENCODING_BIPHASE_LEVEL,
	GSC_ENCODING_DIFF_BIPHASE_LEVEL,

	GSC_PROTOCOL_RS422_RS485,
	GSC_PROTOCOL_RS423,
	GSC_PROTOCOL_RS232,
	GSC_PROTOCOL_RS530_1,
	GSC_PROTOCOL_RS530_2,
	GSC_PROTOCOL_V35_1,
	GSC_PROTOCOL_V35_2,
	GSC_PROTOCOL_RS422_RS423_1,
	GSC_PROTOCOL_RS422_RS423_2,

	GSC_TERMINATION_ENABLED,
	GSC_TERMINATION_DISABLED,

	GSC_PARITY_NONE,
	GSC_PARITY_EVEN,
	GSC_PARITY_ODD,
	GSC_PARITY_MARK,
	GSC_PARITY_SPACE,

	GSC_STOP_BITS_0,
	GSC_STOP_BITS_1,
	GSC_STOP_BITS_1_5,
	GSC_STOP_BITS_2,

	GSC_CRC_NONE,
	GSC_CRC_16,
	GSC_CRC_32,
	GSC_CRC_CCITT,
	GSC_CRC_INIT_0,
	GSC_CRC_INIT_1,

	GSC_RISING_EDGE,
	GSC_FALLING_EDGE,
	GSC_HIGH_TRUE,
	GSC_LOW_TRUE,

	GSC_CLOCK_INTERNAL,
	GSC_CLOCK_EXTERNAL,

	GSC_LSB_FIRST,
	GSC_MSB_FIRST,

	GSC_CIRCUIT_AUTO,
	GSC_CIRCUIT_A,
	GSC_CIRCUIT_B,
	GSC_CIRCUIT_C,
	GSC_CIRCUIT_D,

	// Analog Board Ranges
	GSC_ANA_BIP_2_5,			// BiPolar, -2.5 to +2.5
	GSC_ANA_BIP_5_0,			// BiPolar, -5.0 to +5.0
	GSC_ANA_BIP_10_0,			// BiPolar, -10.0 to +10.0
	GSC_ANA_UNI_5_0,			// UniPolar, 0.0 to +5.0
	GSC_ANA_UNI_10_0,			// UniPolar, 0.0 to +10.0

	// Analog Board Input Types
	GSC_ANA_SINGLE_ENDED,
	GSC_ANA_PSEUDO_DIFF,
	GSC_ANA_DIFFERENTIAL,

	// Analog Channel Inputs
	GSC_ANA_NORMAL,
	GSC_ANA_VREF,
	GSC_ANA_ZERO,

	// Analog Data Formats
	GSC_ANA_OFFSET_BINARY,
	GSC_ANA_TWOS_COMPLEMENT,

	// Analog Clock Sources
	GSC_ANA_CLOCK_INTERNAL,
	GSC_ANA_CLOCK_SECONDARY,
	GSC_ANA_CLOCK_EXTERNAL,
	GSC_ANA_CLOCK_MANUAL,

  // Analog buffer data width
  GSC_ANA_DATAWIDTH_16,
  GSC_ANA_DATAWIDTH_18,
  GSC_ANA_DATAWIDTH_20,
  GSC_ANA_DATAWIDTH_24,
  GSC_ANA_DATAWIDTH_32,

  // Analog external sync
  GSC_ANA_SYNCTARGET,
  GSC_ANA_SYNCINITIATOR,

  // Analog DMA demand mode
  GSC_ANA_DMA_AUTONOMOUS,
  GSC_ANA_DMA_THRESHOLD
};

//
// Analog auto calibration status
//
#define   GSC_ANA_AUTOCAL_INIT  0x01
#define   GSC_ANA_AUTOCAL_PASS  0x02


//
// Protocol options supported by GscSIOGetOption() and GscSIOSetOption()
//
enum GSC_OPTION_NAME
{
	// common options
	GSC_SIO_DATASIZE,
	GSC_SIO_GAPSIZE,
	GSC_SIO_ENCODING,
	GSC_SIO_PROTOCOL,
	GSC_SIO_PARITY,
	GSC_SIO_STOPBITS,
	GSC_SIO_DTEDCE,
	GSC_SIO_TXALMOST,
	GSC_SIO_RXALMOST,
	GSC_SIO_LOOPBACK,
	GSC_SIO_TXDATAPINMODE,
	GSC_SIO_RXDATAPINMODE,
	GSC_SIO_TXCLOCKPINMODE,
	GSC_SIO_RXCLOCKPINMODE,
	GSC_SIO_CTSPINMODE,
	GSC_SIO_RTSPINMODE,
	GSC_SIO_CLOCKSOURCE,
	GSC_SIO_RECEIVER,
	GSC_SIO_TRANSMITTER,
	GSC_SIO_TXUNDERRUN,
	GSC_SIO_TXPREAMBLE,
	GSC_SIO_TXIDLECONDITION,

	// HDLC - specific options
	GSC_SIO_CRCMODE,
	GSC_SIO_TXSHAREDZERO,
	GSC_SIO_RXADDRSEARCH,

	// Sync - specific options
	GSC_SIO_MSBLSBORDER,
	GSC_SIO_CLOCKEDGE,
	GSC_SIO_TXCLOCKIDLECONDITION,
	GSC_SIO_ENVELOPEPOLARITY,

	// Bisync/Bisync16 options
	GSC_SIO_SYNCWORD,
	
	// Bisync - specific options
	GSC_SIO_TXPREAMBLELEN,
	GSC_SIO_TXPREAMBLEPATTERN,
	GSC_SIO_TXSHORTSYNC,
	GSC_SIO_RXSYNCSTRIP,
	GSC_SIO_RXSHORTSYNC,

	// Bisync16 - specific options
	GSC_SIO_ORDERING,
	GSC_SIO_MAXRXCOUNT,

  // FIFO features
  GSC_SIO_FIFO_TIMETAG_BYTE,
  GSC_SIO_FIFO_PARITYERROR_BYTE,

  // Analog Card Options
	GSC_ANA_INPUT_MODE,
	GSC_ANA_INPUT_TYPE,
	GSC_ANA_INPUT_RANGE,
	GSC_ANA_DATA_FORMAT,
	GSC_ANA_EXTERNAL_SYNC,
  GSC_ANA_DATA_PACKING,
	GSC_ANA_SCAN_MARKING,
	GSC_ANA_SCAN_MARKING_VALUE,
	GSC_ANA_CLOCK_SOURCE,
	GSC_ANA_MASTER_CLOCK,
  GSC_ANA_DATA_WIDTH,
  GSC_ANA_SAMPLE_RATE,
  GSC_ANA_NUMACTIVECHANNELS,
  GSC_ANA_MAXCHANNELS,
  GSC_ANA_SINGLEACTIVECHANNEL,
  GSC_ANA_RATEGEN,
  GSC_ANA_INPUTBUFFER,
  GSC_ANA_CLEARBUFFER_ON_SYNC,
  GSC_ANA_DMA_DEMANDMODE
};

//
// System Level Routines
//
GSCAPI_API int GscFindBoards( int *boardCount, GSC_DEVICES_STRUCT *results);
GSCAPI_API int GscGetErrorString( int errorCode, char *errorString);

//
// Board Level Routines
//
GSCAPI_API int GscOpen( int boardNumber, int headerVersion);
GSCAPI_API int GscClose( int boardNumber);
GSCAPI_API int GscGetInfo( int boardNumber, PBOARD_INFO info);
GSCAPI_API int GscGetVersions( int boardNumber, int *libVersion, int *driverVersion, int *fpgaVersion);
GSCAPI_API int GscLocalRegisterRead( int boardNumber, int reg, int *result);
GSCAPI_API int GscLocalRegisterWrite(int boardNumber, int reg, int value);

GSCAPI_API int GscGetNumberOfChannels( int boardNumber, int *channels);

//
// Channel Level Routines
//
GSCAPI_API int GscGetRxFifoSize( int boardNumber, int channel, int *size);
GSCAPI_API int GscGetTxFifoSize( int boardNumber, int channel, int *size);

GSCAPI_API int GscSio4SetOption( int boardNumber, int channelNumber, enum GSC_OPTION_NAME option, int value );
GSCAPI_API int GscSio4GetOption( int boardNumber, int channelNumber, enum GSC_OPTION_NAME option, int value[] );
GSCAPI_API int GscSio4ChannelReset(int boardNumber, int channel);
GSCAPI_API int GscSio4ChannelResetRxFifo(int boardNumber, int channel);
GSCAPI_API int GscSio4ChannelResetTxFifo(int boardNumber, int channel);
GSCAPI_API int GscSio4ChannelRegisterRead(int boardNumber, int channel, int reg, int *value);
GSCAPI_API int GscSio4ChannelRegisterWrite(int boardNumber, int channel, int reg, int value);
GSCAPI_API int GscSio4GetLastError( int boardNumber, int channel, int errorCode, char *errorString, char *detailString);
GSCAPI_API int GscSio4ChannelGetMode(int boardNumber, int channel, int *mode, int *bitRate);
GSCAPI_API int GscSio4ChannelSetMode(int boardNumber, int channel, int mode, int bitRate);
GSCAPI_API int GscSio4ChannelGetTxAlmost(int boardNumber, int channel, int *almostValue);
GSCAPI_API int GscSio4ChannelSetTxAlmost(int boardNumber, int channel, int almostValue);
GSCAPI_API int GscSio4ChannelGetRxAlmost(int boardNumber, int channel, int *almostValue);
GSCAPI_API int GscSio4ChannelSetRxAlmost(int boardNumber, int channel, int almostValue);
GSCAPI_API int GscSio4ChannelSetClock( int boardNumber, int channel, int frequency);
GSCAPI_API int GscSio4ChannelCheckForData(int boardNumber, int channel, char *buffer, int *count);
GSCAPI_API int GscSio4ChannelReceivePacket(int boardNumber, int channel, char *buffer, int *count, int *transferStatus);
GSCAPI_API int GscSio4ChannelReceiveData(int boardNumber, int channel, char *buffer, int count, int *id);
GSCAPI_API int GscSio4ChannelReceiveDataAndWait(int boardNumber, int channel, char *buffer, int count, int timeout, int *bytesTransferred);
GSCAPI_API int GscSio4ChannelTransmitData(int boardNumber, int channel, char *buffer, int count, int *id);
GSCAPI_API int GscSio4ChannelTransmitDataAndWait(int boardNumber, int channel, char *buffer, int count, int timeout, int *bytesTransferred);
GSCAPI_API int GscSio4ChannelFifoCounts( int boardNumber, int channel, int *counts);
GSCAPI_API int GscSio4ChannelFifoSizes( int boardNumber, int channel, int *sizes);
GSCAPI_API int GscSio4ChannelQueryTransfer(int boardNumber, int channel, int *stat, int id);
GSCAPI_API int GscSio4ChannelWaitForTransfer(int boardNumber, int channel, int timeout, int id, int *bytesTransferred);
GSCAPI_API int GscSio4ChannelWaitForTransfer_A(int boardNumber, int channel, int timeout, int id, int *bytesTransferred);
GSCAPI_API int GscSio4ChannelFlushTransfer( int boardNumber, int channel, int id);
GSCAPI_API int GscSio4ChannelRemoveTransfer( int boardNumber, int channel, int id, int *bytesTransferred);
GSCAPI_API int GscSio4ChannelReceiverControl( int boardNumber, int channel, int rxMode);
GSCAPI_API int GscSio4ChannelTransmitterControl( int boardNumber, int channel, int txMode);
GSCAPI_API int GscSio4ChannelRegisterInterrupt( int boardNumber, int channel, int interrupt, int type, GSC_CB_FUNCTION *function);

GSCAPI_API int GscSio4ChannelGetPinMode( int boardNumber, int channel, int pinName, int *mode);
GSCAPI_API int GscSio4ChannelSetPinMode( int boardNumber, int channel, int pinName, int mode);
GSCAPI_API int GscSio4ChannelGetPinValue( int boardNumber, int channel, int pinName, int *value);
GSCAPI_API int GscSio4ChannelSetPinValue( int boardNumber, int channel, int pinName, int value);

//
// Deprecated channel level routines
//
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetDataSize(int boardNumber, int channel, int *dataSize) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetDataSize(int boardNumber, int channel, int dataSize) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetGapSize(int boardNumber, int channel, int *gapSize) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetGapSize(int boardNumber, int channel, int gapSize) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetMsbLsbOrder(int boardNumber, int channel, int *txOrder, int *rxOrder) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetMsbLsbOrder(int boardNumber, int channel, int txOrder, int rxOrder) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetEncoding(int boardNumber, int channel, int *encoding) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetEncoding(int boardNumber, int channel, int encoding) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetProtocol(int boardNumber, int channel, int *protocol, int *termination) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetProtocol(int boardNumber, int channel, int protocol, int termination) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetParity(int boardNumber, int channel, int *parity) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetParity(int boardNumber, int channel, int parity) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetStopBits(int boardNumber, int channel, int *stopBits) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetStopBits(int boardNumber, int channel, int stopBits) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetDteDce(int boardNumber, int channel, int *mode) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetDteDce(int boardNumber, int channel, int mode) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetLoopBack( int boardNumber, int channel, int *loopMode) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetLoopBack( int boardNumber, int channel, int loopMode) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetClockSource( int boardNumber, int channel, int txSource, int rxSource) GCC_ATTRIB_DEPRECATED;

//
// Deprecated system Level Routines
//
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4FindBoards( int *boardCount, GSC_DEVICES_STRUCT *results) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4GetErrorString( int errorCode, char *errorString) GCC_ATTRIB_DEPRECATED;

//
// Deprecated board Level Routines
//
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4Open( int boardNumber, int headerVersion) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4Close( int boardNumber) GCC_ATTRIB_DEPRECATED; 
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4GetInfo( int boardNumber, PBOARD_INFO info) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4GetVersions( int boardNumber, int *libVersion, int *driverVersion, int *fpgaVersion) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4LocalRegisterRead( int boardNumber, int reg, int *result) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4LocalRegisterWrite(int boardNumber, int reg, int value) GCC_ATTRIB_DEPRECATED;

//
// Async Specific Commands
//
GSCAPI_API int GscSio4AsyncGetDefaults( PGSC_ASYNC_CONFIG config );
GSCAPI_API int GscSio4AsyncGetConfig( int boardNumber, int channel, PGSC_ASYNC_CONFIG config );
GSCAPI_API int GscSio4AsyncSetConfig( int boardNumber, int channel, GSC_ASYNC_CONFIG config );

//
// RAsync Specific Commands
//
GSCAPI_API int GscSio4RAsyncGetDefaults( PGSC_RASYNC_CONFIG config );
GSCAPI_API int GscSio4RAsyncGetConfig( int boardNumber, int channel, PGSC_RASYNC_CONFIG config );
GSCAPI_API int GscSio4RAsyncSetConfig( int boardNumber, int channel, GSC_RASYNC_CONFIG config );

//
// HDLC mode-specific API functions
//
GSCAPI_API int GscSio4HdlcGetDefaults( PGSC_HDLC_CONFIG config);
GSCAPI_API int GscSio4HdlcGetConfig(int boardNumber, int channel, PGSC_HDLC_CONFIG config);
GSCAPI_API int GscSio4HdlcSetConfig(int boardNumber, int channel, GSC_HDLC_CONFIG config);

// Deprecated HDLC mode-specific API functions
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4HdlcSetCrcMode(int boardNumber, int channel, int crcMode, int initialValue) GCC_ATTRIB_DEPRECATED;

//
// Sync Specific Commands
//
GSCAPI_API int GscSio4SyncGetDefaults( PGSC_SYNC_CONFIG config );
GSCAPI_API int GscSio4SyncGetConfig( int boardNumber, int channel, PGSC_SYNC_CONFIG config );
GSCAPI_API int GscSio4SyncSetConfig( int boardNumber, int channel, GSC_SYNC_CONFIG config );

//
// BiSync mode-specific API functions
//
GSCAPI_API int GscSio4BiSyncGetDefaults( PGSC_BISYNC_CONFIG config );
GSCAPI_API int GscSio4BiSyncGetConfig(int boardNumber, int channel, PGSC_BISYNC_CONFIG config);
GSCAPI_API int GscSio4BiSyncSetConfig(int boardNumber, int channel, GSC_BISYNC_CONFIG config);

// Deprecated BiSync mode-specific API functions
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetBiSyncPattern( int boardNumber, int channel, int *TxSync, int *RxSync) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetBiSyncPattern( int boardNumber, int channel, int TxSync, int RxSync) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetBiSyncTxUnderrun( int boardNumber, int channel, int *txUnderrun) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetBiSyncTxUnderrun( int boardNumber, int channel, int txUnderrun) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetBiSyncTxPreamble( int boardNumber, int channel, int *txPreamble) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetBiSyncTxPreamble( int boardNumber, int channel, int txPreamble) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetBiSyncTxShortSync( int boardNumber, int channel, int *txShortSync) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetBiSyncTxShortSync( int boardNumber, int channel, int txShortSync) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetBiSyncRxSyncStrip( int boardNumber, int channel, int *rxSyncStrip) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetBiSyncRxSyncStrip( int boardNumber, int channel, int rxSyncStrip) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetBiSyncRxShortSync( int boardNumber, int channel, int *rxShortSync) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetBiSyncRxShortSync( int boardNumber, int channel, int rxShortSync) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetBiSyncTxPreambleLength( int boardNumber, int channel, int *txPreambleLength) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetBiSyncTxPreambleLength( int boardNumber, int channel, int txPreambleLength) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelGetBiSyncTxPreamblePattern( int boardNumber, int channel, int *txPreamblePattern) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4ChannelSetBiSyncTxPreamblePattern( int boardNumber, int channel, int txPreamblePattern) GCC_ATTRIB_DEPRECATED;

//
// BiSync16 mode-specific functions
//
GSCAPI_API int GscSio4BiSync16GetDefaults( PGSC_BISYNC16_CONFIG config );
GSCAPI_API int GscSio4BiSync16GetConfig(int boardNumber, int channel, PGSC_BISYNC16_CONFIG config);
GSCAPI_API int GscSio4BiSync16SetConfig(int boardNumber, int channel, GSC_BISYNC16_CONFIG config);
GSCAPI_API int GscSio4BiSync16GetTxCounts( int boardNumber, int channel, int *remaining, int *initial);
GSCAPI_API int GscSio4BiSync16GetRxCounts( int boardNumber, int channel, int *remaining, int *initial);
GSCAPI_API int GscSio4BiSync16EnterHuntMode( int boardNumber, int channel);
GSCAPI_API int GscSio4BiSync16AbortTx( int boardNumber, int channel);
GSCAPI_API int GscSio4BiSync16Pause( int boardNumber, int channel);
GSCAPI_API int GscSio4BiSync16Resume( int boardNumber, int channel);

// Deprecated BiSync16 mode-specific functions
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4BiSync16GetSyncWord( int boardNumber, int channel, int *syncWord, int *idleWord) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4BiSync16SetSyncWord( int boardNumber, int channel, int syncWord, int idleWord) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4BiSync16GetOrdering( int boardNumber, int channel, int *byteOrder, int *bitOrder) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4BiSync16SetOrdering( int boardNumber, int channel, int byteOrder, int bitOrder) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4BiSync16GetMaxRxCount( int boardNumber, int channel, int *maxRxCount) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4BiSync16SetMaxRxCount( int boardNumber, int channel, int maxRxCount) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4BiSync16GetReceiver( int boardNumber, int channel, int *mode) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4BiSync16SetReceiver( int boardNumber, int channel, int mode) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4BiSync16GetTransmitter( int boardNumber, int channel, int *mode) GCC_ATTRIB_DEPRECATED;
GSCAPI_API MSC_ATTRIB_DEPRECATED int GscSio4BiSync16SetTransmitter( int boardNumber, int channel, int mode) GCC_ATTRIB_DEPRECATED;

//
// Analog Card specific API functions
//
GSCAPI_API int GscAnaGetDefaults( int boardNumber, PGSC_ANALOG_CONFIG config );
GSCAPI_API int GscAnaGetConfig(int boardNumber, PGSC_ANALOG_CONFIG config);
GSCAPI_API int GscAnaSetConfig(int boardNumber, GSC_ANALOG_CONFIG config);
GSCAPI_API int GscAnaGetOption( int boardNumber, enum GSC_OPTION_NAME optionName, int optionValue[]);
GSCAPI_API int GscAnaSetOption( int boardNumber, enum GSC_OPTION_NAME optionName, int optionValue);
GSCAPI_API int GscAnaScanInputs( int boardNumber, int samples, double samplesPerSecond, int firstChannel, int lastChannel, char *buffer);
GSCAPI_API int GscAnaAutoCal( int boardNumber, int waitForCompletion );
GSCAPI_API int GscAnaRegisterInterrupt( int boardNumber, int interrupt, GSC_CB_FUNCTION *function);
GSCAPI_API int GscAnaReceiveScan( int boardNumber, char *buffer, int *count, int *transferStatus);
GSCAPI_API int GscAnaLocalRegisterRead( int BoardNo, int reg, int *result);
GSCAPI_API int GscAnaLocalRegisterWrite(int boardNumber, int reg, int value);
GSCAPI_API int GscAnaInitializeBoard( int BoardNo );
GSCAPI_API int GscAnaGetAutoCalStatus( int BoardNo, int *sts );
GSCAPI_API int GscAnaClearInputBuffer( int BoardNo );
GSCAPI_API int GscAnaGetInputBufferCount( int BoardNo, int *BuffSize );
GSCAPI_API int GscAnaSetInputBufferThreshold( int BoardNo, int threshold );
GSCAPI_API int GscAnaSoftwareSync( int BoardNo );

//GSCAPI_API int GscSio4ChannelGetRxAlmostEmpty(int boardNumber, int channel, int *almostValue);
//GSCAPI_API int GscSio4ChannelGetRxAlmostFull(int boardNumber, int channel, int *almostValue);
//GSCAPI_API int GscSio4ChannelGetTxAlmostEmpty(int boardNumber, int channel, int *almostValue);
//GSCAPI_API int GscSio4ChannelGetTxAlmostFull(int boardNumber, int channel, int *almostValue);

#ifdef __cplusplus
}
#endif

#endif
