// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI6LN4AO/utils/query.c $
// $Rev: 53777 $
// $Date: 2023-09-28 12:51:48 -0500 (Thu, 28 Sep 2023) $

// 24DSI6LN4AO: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	dsi6ln4ao_query
*
*	Purpose:
*
*		Provide a visual wrapper for the DSI6LN4AO_IOCTL_QUERY service.
*
*	Arguments:
*
*		fd		Use this handle to access the device.
*
*		index	The index of the device to access. Ignore if < 0.
*				This is for display purposes only.
*
*		verbose	Work verbosely?
*
*		set		This is the query option to access.
*
*		get		The results are reported here. This may be NULL.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int dsi6ln4ao_query(int fd, int index, int verbose, s32 set, s32* get)
{
	char		buf[128]	= { 0 };
	int			errs		= 0;
	const char*	ptr			= buf;
	s32			query		= set;
	int			ret;
	s32			tmp;

	switch (query)
	{
		default:

			errs++;
			ptr	= buf;
			sprintf(buf, "Query Error (#%ld)", (long) query);
			break;

		case DSI6LN4AO_QUERY_AI_CHAN_MAX:		ptr	= "AI Channels Max";		break;
		case DSI6LN4AO_QUERY_AI_CHAN_QTY:		ptr	= "AI Channel Qty";			break;
		case DSI6LN4AO_QUERY_AI_FIFO_SIZE:		ptr	= "AI FIFO Size";			break;
		case DSI6LN4AO_QUERY_AI_FILTER:			ptr	= "Ai Filter";				break;
		case DSI6LN4AO_QUERY_AI_FREF:			ptr	= "AI Master Clock";		break;
		case DSI6LN4AO_QUERY_AI_FSAMP_MAX:		ptr	= "AI Fsamp Maximum";		break;
		case DSI6LN4AO_QUERY_AI_FSAMP_MIN:		ptr	= "AI Fsamp Min";			break;
		case DSI6LN4AO_QUERY_AI_NDIV_MAX:		ptr	= "AI Ndiv Maximum";		break;
		case DSI6LN4AO_QUERY_AI_NDIV_MIN:		ptr	= "AI Ndiv Minimum";		break;
		case DSI6LN4AO_QUERY_AI_NREF_MAX:		ptr	= "AI Nref Maximum";		break;
		case DSI6LN4AO_QUERY_AI_NREF_MAX_OPT:	ptr	= "AI Nref Max Optimal";	break;
		case DSI6LN4AO_QUERY_AI_NREF_MIN:		ptr	= "AI Nref Minimum";		break;
		case DSI6LN4AO_QUERY_AI_NVCO_MAX:		ptr	= "AI Nvco Maximum";		break;
		case DSI6LN4AO_QUERY_AI_NVCO_MAX_OPT:	ptr	= "AI Nvco Max Optimal";	break;
		case DSI6LN4AO_QUERY_AI_NVCO_MIN:		ptr	= "AI Nvco Minimum";		break;
		case DSI6LN4AO_QUERY_AI_RANGE:			ptr	= "AI Voltage Range";		break;
		case DSI6LN4AO_QUERY_AO_CHAN_MAX:		ptr	= "AO Channels Max";		break;
		case DSI6LN4AO_QUERY_AO_CHAN_QTY:		ptr	= "AO Channel Qty";			break;
		case DSI6LN4AO_QUERY_AO_FCLK_MAX:		ptr	= "AO Fclk Maximum";		break;
		case DSI6LN4AO_QUERY_AO_FCLK_MIN:		ptr	= "AO Fclk Minimum";		break;
		case DSI6LN4AO_QUERY_AO_FREF:			ptr	= "AO Master Clock";		break;
		case DSI6LN4AO_QUERY_AO_NRATE_MAX:		ptr	= "AO Nrate Maximum";		break;
		case DSI6LN4AO_QUERY_AO_NRATE_MIN:		ptr	= "AO Nrate Minimum";		break;
		case DSI6LN4AO_QUERY_COUNT:				ptr	= "Query Options";			break;
		case DSI6LN4AO_QUERY_DEVICE_TYPE:		ptr	= "Device Type";			break;
		case DSI6LN4AO_QUERY_INIT_MS:			ptr	= "Init. Duration";			break;
		case DSI6LN4AO_QUERY_TEMPERATURE:		ptr	= "Temperature Range";		break;
	}

	if (verbose)
		gsc_label_index(ptr, index);

	ret		= dsi6ln4ao_ioctl(fd, DSI6LN4AO_IOCTL_QUERY, &set);
	errs	+= ret ? 1 : 0;

	switch (query)
	{
		default:

			errs++;
			sprintf(buf, "%ld", (long) set);
			break;

		case DSI6LN4AO_QUERY_AI_CHAN_MAX:
		case DSI6LN4AO_QUERY_AI_CHAN_QTY:
		case DSI6LN4AO_QUERY_AI_NDIV_MAX:
		case DSI6LN4AO_QUERY_AI_NDIV_MIN:
		case DSI6LN4AO_QUERY_AI_NREF_MAX:
		case DSI6LN4AO_QUERY_AI_NREF_MAX_OPT:
		case DSI6LN4AO_QUERY_AI_NREF_MIN:
		case DSI6LN4AO_QUERY_AI_NVCO_MAX:
		case DSI6LN4AO_QUERY_AI_NVCO_MAX_OPT:
		case DSI6LN4AO_QUERY_AI_NVCO_MIN:
		case DSI6LN4AO_QUERY_AO_CHAN_MAX:
		case DSI6LN4AO_QUERY_AO_CHAN_QTY:
		case DSI6LN4AO_QUERY_AO_NRATE_MAX:
		case DSI6LN4AO_QUERY_AO_NRATE_MIN:
		case DSI6LN4AO_QUERY_COUNT:

			gsc_label_long_comma_buf(set, buf);
			break;

		case DSI6LN4AO_QUERY_AI_FREF:
		case DSI6LN4AO_QUERY_AO_FCLK_MAX:
		case DSI6LN4AO_QUERY_AO_FCLK_MIN:
		case DSI6LN4AO_QUERY_AO_FREF:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " Hz");
			break;

		case DSI6LN4AO_QUERY_INIT_MS:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " ms");
			break;

		case DSI6LN4AO_QUERY_AI_FSAMP_MAX:
		case DSI6LN4AO_QUERY_AI_FSAMP_MIN:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " S/S");
			break;

		case DSI6LN4AO_QUERY_AI_FIFO_SIZE:

			ptr	= "";
			tmp	= set;

			if ((tmp) && ((tmp % 1024) == 0))
			{
				ptr	= "K";
				tmp	/= 1024;
			}

			gsc_label_long_comma_buf(tmp, buf);
			strcat(buf, ptr);
			strcat(buf, " Words");
			break;

		case DSI6LN4AO_QUERY_DEVICE_TYPE:

			if (set == GSC_DEV_TYPE_24DSI6LN4AO)
			{
				strcpy(buf, "24DSI6LN4AO");
			}
			else
			{
				errs	= 1;
				sprintf(buf, "INVALID: %ld", (long) set);
			}

			break;

		case DSI6LN4AO_QUERY_AI_FILTER:

			switch (set)
			{
				default:

					errs++;
					sprintf(buf, "Invalid option: %ld", (long) set);
					break;

				case DSI6LN4AO_AI_FILTER_270KHZ:

					strcpy(buf, "270KHz");
					break;

				case DSI6LN4AO_AI_FILTER_CUSTOM:

					strcpy(buf, "Custom");
					break;
			}

			break;

		case DSI6LN4AO_QUERY_AI_RANGE:

			switch (set)
			{
				default:

					errs++;
					sprintf(buf, "Invalid option: %ld", (long) set);
					break;

				case DSI6LN4AO_AI_RANGE_2_5:

					strcpy(buf, "+-2.5V");
					break;

				case DSI6LN4AO_AI_RANGE_5:

					strcpy(buf, "+-5V");
					break;

				case DSI6LN4AO_AI_RANGE_10:

					strcpy(buf, "+-10V");
					break;
			}

			break;

		case DSI6LN4AO_QUERY_TEMPERATURE:

			switch (set)
			{
				default:

					errs++;
					sprintf(buf, "Invalid option: %ld", (long) set);
					break;

				case DSI6LN4AO_TEMPERATURE_COM:

					strcpy(buf, "Commercial");
					break;

				case DSI6LN4AO_TEMPERATURE_EXT:

					strcpy(buf, "Extended");
					break;
			}

			break;
	}

	if (verbose)
	{
		if (errs)
			printf("FAIL <---  (%s)\n", buf);
		else
			printf("%s\n", buf);
	}

	if (get)
		get[0]	= set;

	return(errs);
}


