// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI6LN4AO/utils/fsamp_ai.c $
// $Rev: 51458 $
// $Date: 2022-08-04 11:13:02 -0500 (Thu, 04 Aug 2022) $

// 24DSI6LN4AO: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	dsi6ln4ao_fsamp_ai_compute
*
*	Purpose:
*
*		Determine the best configuration for the requested sample rate.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fsamp	The desired sample rate.
*
*		optimal	Are we to look for the best overall of the best optimal?
*
*		nvco	The Nvco result is reported here - can be NULL.
*
*		nref	The Nref result is reported here - can be NULL.
*
*		ndiv	The Ndiv result is reported here - can be NULL.
*
*		rate	The resulting sample rate is reported here - can be NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int dsi6ln4ao_fsamp_ai_compute(
	int		fd,
	int		index,
	int		verbose,
	int		fsamp,
	int		optimal,
	s32*	nvco,
	s32*	nref,
	s32*	ndiv,
	float*	rate)
{
	double	delta;
	double	delta_best	= 999999999;
	double	div;
	int		errs		= 0;
	int		init		= 0;
	s32		fref;

	double	fsamp_best	= fsamp;
	double	fsamp_got;
	s32		fsamp_max	= 250000;
	s32		fsamp_min	= 2000;

	s32		ndiv_best	= 0;
	s32		ndiv_ctr;
	s32		ndiv_max	= 25;
	s32		ndiv_min	= 0;

	s32		nref_best	= 0;
	s32		nref_ctr;
	s32		nref_max	= 1000;
	s32		nref_min	= 0;

	s32		nvco_best	= 0;
	s32		nvco_ctr;
	s32		nvco_max	= 1000;
	s32		nvco_min	= 0;

	if (verbose)
	{
		gsc_label_index("Sample Rate", index);
		gsc_label_long_comma(fsamp);
		printf(" S/S %s", optimal ? "Optimal" : "Overall");
	}

	errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AI_FREF, &fref);
	errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AI_FSAMP_MAX, &fsamp_max);
	errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AI_FSAMP_MIN, &fsamp_min);
	errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AI_NDIV_MIN, &ndiv_min);
	errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AI_NREF_MIN, &nref_min);
	errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AI_NVCO_MIN, &nvco_min);
	errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AI_NDIV_MAX, &ndiv_max);

	if (optimal)
	{
		errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AI_NREF_MAX_OPT, &nref_max);
		errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AI_NVCO_MAX_OPT, &nvco_max);
	}
	else
	{
		errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AI_NREF_MAX, &nref_max);
		errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AI_NVCO_MAX, &nvco_max);
	}

	if (errs)
	{
		if (verbose)
			printf(" FAIL <---\n");
	}
	else
	{
		if (fsamp < fsamp_min)
			fsamp	= fsamp_min;

		if (fsamp > fsamp_max)
			fsamp	= fsamp_max;

		for (nvco_ctr = nvco_min; nvco_ctr <= nvco_max; nvco_ctr++)
		{
			for (nref_ctr = nref_min; nref_ctr <= nref_max; nref_ctr++)
			{
				for (ndiv_ctr = ndiv_min; ndiv_ctr <= ndiv_max; ndiv_ctr++)
				{
					div			= ndiv_ctr ? ndiv_ctr : 0.5;
					fsamp_got	= (double) fref * nvco_ctr / nref_ctr / div / 512;
					delta		= fsamp_got - fsamp;

					if (delta < 0)
						delta	= -delta;

					if ((delta < delta_best) || (init == 0))
					{
						init		= 1;
						fsamp_best	= fsamp_got;
						delta_best	= delta;
						nvco_best	= nvco_ctr;
						nref_best	= nref_ctr;
						ndiv_best	= ndiv_ctr;
					}
				}
			}
		}

		if (verbose)
		{
			printf(" (");
			printf("Fref ");
			gsc_label_long_comma(fref);
			printf(", Nvco ");
			gsc_label_long_comma(nvco_best);
			printf(", Nref ");
			gsc_label_long_comma(nref_best);
			printf(", Ndiv ");
			gsc_label_long_comma(ndiv_best);
			printf(", Fsamp ");
			gsc_label_float_comma(fsamp_best, 0, 3);
			printf(")\n");
		}

		if (nvco)
			nvco[0]	= nvco_best;

		if (nref)
			nref[0]	= nref_best;

		if (ndiv)
			ndiv[0]	= ndiv_best;

		if (rate)
			rate[0]	= (float) fsamp_best;
	}

	return(errs);
}



//*****************************************************************************
int dsi6ln4ao_fsamp_ai_report(int fd, int index, int verbose, int chan, s32* fsamp)
{
	char	buf[64];
	double	div;
	int		errs	= 0;
	s32		fref;
	s32		last;
	s32		max;
	s32		ndiv;
	s32		nref;
	s32		nvco;
	s32		qty;
	double	rate	= 0;

	if (verbose)
	{
		sprintf(buf, "Channel %d", chan);
		gsc_label(buf);
	}

	errs	+= dsi6ln4ao_query			(fd, -1, 0, DSI6LN4AO_QUERY_AI_CHAN_MAX, &max);
	errs	+= dsi6ln4ao_query			(fd, -1, 0, DSI6LN4AO_QUERY_AI_CHAN_QTY, &qty);
	errs	+= dsi6ln4ao_query			(fd, -1, 0, DSI6LN4AO_QUERY_AI_FREF, &fref);
	errs	+= dsi6ln4ao_ai_chan_last	(fd, -1, 0, -1, &last);
	errs	+= dsi6ln4ao_ai_nvco		(fd, -1, 0, -1, &nvco);
	errs	+= dsi6ln4ao_ai_nref		(fd, -1, 0, -1, &nref);
	errs	+= dsi6ln4ao_ai_ndiv		(fd, -1, 0, -1, &ndiv);

	if (errs)
	{
		if (verbose)
			printf("FAIL <---\n");
	}
	else if ((chan < 0) || (chan >= max))
	{
		errs++;

		if (verbose)
			printf("FAIL <---  (Invalid channel: %d)\n", chan);
	}
	else if (chan >= qty)
	{
		errs++;

		if (verbose)
			printf("0 S/S  (Channel not installed.\n");
	}
	else if (chan > last)
	{
		if (verbose)
			printf("0 S/S  (Channel not active.\n");
	}
	else
	{
		div		= ndiv ? ndiv : 0.5;
		rate	= (double) fref * nvco / nref / div / 512;

		if (verbose)
		{
			gsc_label_long_comma((long long) (rate + 0.5));
			printf(" S/S (Fref ");
			gsc_label_long_comma(fref);
			printf(", Nvco ");
			gsc_label_long_comma(nvco);
			printf(", Nref ");
			gsc_label_long_comma(nref);
			printf(", Ndiv ");
			gsc_label_long_comma(ndiv);
			printf(" Fsamp ");
			gsc_label_float_comma(rate, 0, 3);
			printf(")\n");
		}
	}

	if (fsamp)
		fsamp[0]	= rate;

	return(errs);
}



//*****************************************************************************
int dsi6ln4ao_fsamp_ai_report_all(int fd, int index, int verbose, s32* fsamp)
{
	int	errs;
	int	i;
	s32	qty		= 0;
	s32	tmp;
	s32	total	= 0;

	if (verbose)
		gsc_label_index("Sample Rates", index);

	errs	= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AI_CHAN_QTY, &qty);

	if (errs)
	{
		if (verbose)
			printf("FAIL <---\n");
	}
	else
	{
		if (verbose)
			printf("\n");

		gsc_label_level_inc();

		for (i = 0; i < qty; i++)
		{
			tmp		= 0;
			errs	+= dsi6ln4ao_fsamp_ai_report(fd, index, verbose, i, &tmp);
			total	+= tmp;
		}

		if (verbose)
		{
			gsc_label("Overall");
			gsc_label_long_comma(total);
			printf(" S/S\n");
		}

		gsc_label_level_dec();
	}

	if (fsamp)
		fsamp[0]	= total;

	return(errs);
}


