// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI6LN4AO/utils/fclk_ao.c $
// $Rev: 51458 $
// $Date: 2022-08-04 11:13:02 -0500 (Thu, 04 Aug 2022) $

// 24DSI6LN4AO: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	dsi6ln4ao_fclk_ao_compute
*
*	Purpose:
*
*		Determine the best configuration for the requested clock rate.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fclk	The desired clock rate.
*
*		nrate	The Nrate result is reported here - can be NULL.
*
*		rate	The resulting clock rate is reported here - can be NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int dsi6ln4ao_fclk_ao_compute(
	int		fd,
	int		index,
	int		verbose,
	int		fclk,
	s32*	nrate,
	float*	rate)
{
	double	delta;
	double	delta_best	= 999999999;
	int		errs		= 0;
	int		init		= 0;
	s32		fref;

	double	fclk_best	= fclk;
	double	fclk_got;
	s32		fclk_max	= 250000;
	s32		fclk_min	= 2000;

	s32		nrate_best	= 0;
	s32		nrate_ctr;
	s32		nrate_max	= 1000;
	s32		nrate_min	= 0;

	if (verbose)
	{
		gsc_label_index("Clock Rate", index);
		gsc_label_long_comma(fclk);
		printf(" Hz");
	}

	errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AO_FREF, &fref);
	errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AO_FCLK_MAX, &fclk_max);
	errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AO_FCLK_MIN, &fclk_min);
	errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AO_NRATE_MIN, &nrate_min);
	errs	+= dsi6ln4ao_query(fd, -1, 0, DSI6LN4AO_QUERY_AO_NRATE_MAX, &nrate_max);

	if (errs)
	{
		if (verbose)
			printf(" FAIL <---\n");
	}
	else
	{
		if (fclk < fclk_min)
			fclk	= fclk_min;

		if (fclk > fclk_max)
			fclk	= fclk_max;

		if (fclk > 0)
			nrate_ctr	= fref / fclk;
		else
			nrate_ctr	= nrate_max;

		if (nrate_ctr > nrate_max)
			nrate_ctr	= nrate_max;

		if (nrate_max > (nrate_ctr + 3))
			nrate_max	= nrate_ctr + 3;

		if (nrate_min < (nrate_ctr - 3))
			nrate_min	= nrate_ctr - 3;

		for (nrate_ctr = nrate_min; nrate_ctr <= nrate_max; nrate_ctr++)
		{
			fclk_got	= (double) fref / nrate_ctr;
			delta		= fclk_got - fclk;

			if (delta < 0)
				delta	= -delta;

			if ((delta < delta_best) || (init == 0))
			{
				init		= 1;
				fclk_best	= fclk_got;
				delta_best	= delta;
				nrate_best	= nrate_ctr;
			}
		}

		if (verbose)
		{
			printf(" (");
			printf("Fref ");
			gsc_label_long_comma(fref);
			printf(", Nrate ");
			gsc_label_long_comma(nrate_best);
			printf(", Fclk ");
			gsc_label_float_comma(fclk_best, 0, 3);
			printf(" Hz)\n");
		}

		if (nrate)
			nrate[0]	= nrate_best;

		if (rate)
			rate[0]	= (float) fclk_best;
	}

	return(errs);
}



//*****************************************************************************
int dsi6ln4ao_fclk_ao_report(int fd, int index, int verbose, s32* fclk)
{
	int		errs	= 0;
	s32		fref;
	s32		nrate;
	s32		qty;
	double	rate	= 0;

	if (verbose)
		gsc_label("AO Clock Rate");

	errs	+= dsi6ln4ao_query			(fd, -1, 0, DSI6LN4AO_QUERY_AO_CHAN_QTY, &qty);
	errs	+= dsi6ln4ao_query			(fd, -1, 0, DSI6LN4AO_QUERY_AO_FREF, &fref);
	errs	+= dsi6ln4ao_ao_nrate		(fd, -1, 0, -1, &nrate);

	if (errs)
	{
		if (verbose)
			printf("FAIL <---\n");
	}
	else if (qty <= 0)
	{
		errs++;

		if (verbose)
			printf("0 Hz  (No channels installed.\n");
	}
	else
	{
		rate	= (double) fref / nrate;

		if (verbose)
		{
			gsc_label_long_comma((long long) (rate + 0.5));
			gsc_label("Hz (Fref ");
			gsc_label_long_comma(fref);
			gsc_label(", Nrate ");
			gsc_label_long_comma(nrate);
			printf(")\n");
		}

		gsc_label_level_dec();
	}

	if (fclk)
		fclk[0]	= rate;

	return(errs);
}


