// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI6LN4AO/utils/config_ao.c $
// $Rev: 53777 $
// $Date: 2023-09-28 12:51:48 -0500 (Thu, 28 Sep 2023) $

// 24DSI6LN4AO: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	dsi6ln4ao_config_ao
*
*	Purpose:
*
*		Configure the basic services.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fclk	The desired output clock frequency.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int	dsi6ln4ao_config_ao(int fd, int index, int verbose, int fclk)
{
	int	errs	= 0;
	s32	nrate	= 0;
	s32	qty		= 0;
	s32	zero	= 0x8000;

	if (verbose)
	{
		gsc_label_index("AO Configuration", index);
		printf("\n");
		gsc_label_level_inc();
	}

	errs	+= dsi6ln4ao_query(fd, index, verbose, DSI6LN4AO_QUERY_AO_CHAN_QTY, &qty);

	if (qty <= 0)
	{
		if (verbose)
		{
			gsc_label(NULL);
			printf("No output channels to configure.\n");
		}
	}
	else
	{
		errs	+= dsi6ln4ao_initialize			(fd, index, verbose);

		// Settings must be applied after the initialization call!

		errs	+= dsi6ln4ao_ao_clk_mode		(fd, index, verbose, DSI6LN4AO_AO_CLK_MODE_SYNC,	NULL);
		errs	+= dsi6ln4ao_ao_clk_src			(fd, index, verbose, DSI6LN4AO_AO_CLK_SRC_INT,		NULL);
		errs	+= dsi6ln4ao_data_format		(fd, index, verbose, DSI6LN4AO_DATA_FORMAT_OFF_BIN,	NULL);
		errs	+= dsi6ln4ao_ao_range			(fd, index, verbose, DSI6LN4AO_AO_RANGE_10V,		NULL);

		errs	+= dsi6ln4ao_fclk_ao_compute	(fd, index, verbose, fclk, &nrate,					NULL);
		errs	+= dsi6ln4ao_ao_nrate			(fd, index, verbose, nrate,							NULL);
		errs	+= dsi6ln4ao_fclk_ao_report		(fd, index, verbose,								NULL);

		errs	+= dsi6ln4ao_ao_ch_0_write		(fd, index, verbose, zero,							NULL);
		errs	+= dsi6ln4ao_ao_ch_1_write		(fd, index, verbose, zero,							NULL);
		errs	+= dsi6ln4ao_ao_ch_2_write		(fd, index, verbose, zero,							NULL);
		errs	+= dsi6ln4ao_ao_ch_3_write		(fd, index, verbose, zero,							NULL);
		errs	+= dsi6ln4ao_ao_enable			(fd, index, verbose, DSI6LN4AO_AO_ENABLE_YES,		NULL);

		errs	+= dsi6ln4ao_fsamp_ai_report_all(fd, index, verbose,								NULL);

		// Settings affecting the sample rate or voltage range must generally
		// be applied before autocalibration.
		// Refer to the board user manual for clarification.

		// The 24DSI6LN4AO doesn't support autocalibration capabilities so no
		// such call is being made. If the feature was supported, the call
		// would be placced here.

		// Interrupt selections should be applied after autocalibration as the
		// driver overwrites the current selection to detect the
		// Autocalibration Done interrupt. If an application intends to detect
		// the Autocalibration Done interrupt, then the interrupt selection may
		// be done beforehand. Given that this board doesn't conatin any
		// Autocalibration capabilities, no such effort is attempted. In
		// addition, the interrupt selection call can appear anywhere after the
		// initialization call.
		errs	+= dsi6ln4ao_irq_sel			(fd, index, verbose, DSI6LN4AO_IRQ_INIT_DONE,		NULL);

		// This board doesn't contain any output buffers, so no AO Buffer
		// Clear operation is supported or called for.
	}

	if (verbose)
		gsc_label_level_dec();

	return(errs);
}


