// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI6LN4AO/samples/savedata/perform.c $
// $Rev: 53788 $
// $Date: 2023-09-28 12:56:15 -0500 (Thu, 28 Sep 2023) $

// 24DSI6LN4AO: Sample Application: source file

#include "main.h"



// macros *********************************************************************

#define	_1M					(1024L * 1024L)
#define	FILE_NAME			"data.txt"
#define	TAG_DECODE(v)		GSC_FIELD_DECODE((v),26,24)
#define	TAG_MASK(v)			((v) & 0x7000000)
#define	VALUE_MASK(v)		((v) & 0x0FFFFFF)



// variables ******************************************************************

static	u32	_buffer[_1M];



//*****************************************************************************
static int _channels(const args_t* args, s32* chans)
{
	int	errs;

	errs	= dsi6ln4ao_query(args->fd, -1, 1, DSI6LN4AO_QUERY_AI_CHAN_QTY, chans);

	if (errs)
	{
		chans[0]	= 4;	// We'll assume all channels are installed.
		gsc_label(NULL);
		printf("We'll assume %d channels are present.\n", (int) chans[0]);
	}

	return(errs);
}



//*****************************************************************************
static int _read_data(const args_t* args)
{
	char	buf[80];
	int		errs	= 0;
	long	get		= sizeof(_buffer);
	int		got;
	int		repeat;

	for (repeat = 0; (errs == 0) && (repeat < args->repeat);)
	{
		repeat++;
		strcpy(buf, "Reading");

		if (args->repeat > 1)
		{
			strcat(buf, " #");
			gsc_label_long_comma_buf(repeat, buf + strlen(buf));
		}

		gsc_label(buf);

		got	= dsi6ln4ao_read(args->fd, _buffer, get);

		if (got < 0)
		{
			errs	= 1;
			printf("FAIL <---  (read error: %d)\n", got);
			break;
		}

		if (got != get)
		{
			errs	= 1;
			printf(	"FAIL <---  (got %ld bytes, requested %ld)\n",
					(long) got,
					(long) get);
			break;
		}

		printf("PASS  (");
		gsc_label_long_comma(get / 4);
		printf(" samples)\n");
	}

	return(errs);
}



//*****************************************************************************
static int _save_eol(const char* name, FILE* file)
{
	int	errs	= 0;
	int	i;

	i	= fprintf(file, "\r\n");

	if (i != 2)
	{
		printf("FAIL <---  (fprintf() failure to %s)\n", name);
		errs	= 1;
	}

	return(errs);
}



//*****************************************************************************
static int _save_value(const args_t* args, const char* name, FILE* file, u32 value)
{
	char	buf1[40];
	char	buf2[80];
	int		errs	= 0;
	int		i;
	int		len;

	if (args->chan_tag == CHAN_TAG_EXCLUDE)
		value	= VALUE_MASK(value);
	else if (args->chan_tag == CHAN_TAG_ONLY)
		value	= TAG_MASK(value);

	if (args->decimal)
		sprintf(buf1, "%ld", (long) value);
	else
		sprintf(buf1, "%08lX", (long) value);

	sprintf(buf2, "  %8s", buf1);
	len	= strlen(buf2);
	i	= fprintf(file, "%8s", buf2);

	if (i != len)
	{
		printf("FAIL <---  (fprintf() failure to %s)\n", name);
		errs	= 1;
	}

	return(errs);
}



//*****************************************************************************
static int _validate_data(int chans, u32 last, long index, u32 data)
{
	int	chan_tag;
	int	errs		= 0;
	int	expect;
	int	last_tag;

	if (data & 0xF8000000)
	{
		errs	= 1;
		printf(" FAIL <---  (Upper bits are invalid: [%ld] = 0x%08lX)\n", index, (long) data);
	}

	chan_tag	= TAG_DECODE(data);

	if (chan_tag >= chans)
	{
		errs	= 1;
		printf(" FAIL <---  (Invalid Channel Tag: [%ld] = 0x%08lX)\n", index, (long) data);
	}

	if (last == 0xFFFFFFFF)
	{
		if (chan_tag != 0)
		{
			errs	= 1;
			printf(" FAIL <---  (First tag not zero: [%ld] = 0x%08lX)\n", index, (long) data);
		}
	}
	else
	{
		last_tag	= TAG_DECODE(last);
		expect		= (last_tag + 1) % chans;

		if (chan_tag != expect)
		{
			errs	= 1;
			printf(	" FAIL <---  (Lost Synchronozation: [%ld] = 0x%08lX, [%ld] = 0x%08lX)\n",
					index - 1,
					(long) last,
					index,
					(long) data);
		}
	}

	return(errs);
}



//*****************************************************************************
static int _save_data(const args_t* args, int chans, int errs)
{
	char		buf[64];
	FILE*		file;
	long		index;
	u32			last	= 0xFFFFFFFF;
	const char*	name	= FILE_NAME;
	long		samples	= sizeof(_buffer) / 4;
	int			tag_got;

	strcpy(buf, "Saving");

	if (args->force_save)
	{
		errs	= 0;
		strcat(buf, " (Forced)");
	}

	gsc_label(buf);

	for (;;)
	{
		if ((errs) && (args->force_save == 0))
		{
			printf("SKIPPED  (errors)\n");
			errs	= 0;
			break;
		}

		file	= fopen(name, "w+b");

		if (file == NULL)
		{
			printf("FAIL <---  (unable to create %s)\n", name);
			errs	= 1;
			break;
		}

		for (index = 0; index < samples; index++)
		{
			tag_got	= TAG_DECODE(_buffer[index]);

			if ((index > 0) && (tag_got == 0))
				errs	+= _save_eol(name, file);

			errs	+= _save_value(args, name, file, _buffer[index]);
			errs	+= _validate_data(chans, last, index, _buffer[index]);
			last	= _buffer[index];
		}

		fclose(file);

		if (errs == 0)
			printf("PASS  (%s)\n", name);

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _config(const args_t* args)
{
	int	errs	= 0;

	errs	+= dsi6ln4ao_config_ai	(args->fd, -1, 1, args->fsamp);
	errs	+= dsi6ln4ao_ai_mode	(args->fd, -1, 1, args->ai_mode, NULL);
	errs	+= dsi6ln4ao_rx_io_mode	(args->fd, -1, 1, args->io_mode, NULL);

	errs	+= dsi6ln4ao_rx_io_overflow		(args->fd, -1, 1, DSI6LN4AO_IO_OVERFLOW_IGNORE, NULL);
	errs	+= dsi6ln4ao_rx_io_underflow	(args->fd, -1, 1, DSI6LN4AO_IO_UNDERFLOW_IGNORE, NULL);
	errs	+= dsi6ln4ao_ai_buf_over		(args->fd, -1, 1, DSI6LN4AO_AI_BUF_OVER_CLEAR, NULL);
	errs	+= dsi6ln4ao_ai_buf_under		(args->fd, -1, 1, DSI6LN4AO_AI_BUF_UNDER_CLEAR, NULL);
	errs	+= dsi6ln4ao_rx_io_timeout		(args->fd, -1, 1, 30, NULL);

	return(errs);
}



//*****************************************************************************
int perform_tests(const args_t* args)
{
	s32	chans	= 6;
	int	errs	= 0;

	gsc_label("Capture & Save");
	printf("\n");
	gsc_label_level_inc();

	errs	+= _config(args);
	errs	+= _channels(args, &chans);
	errs	+= dsi6ln4ao_ai_buf_clear	(args->fd, -1, 1);
	errs	+= _read_data(args);
	errs	+= dsi6ln4ao_ai_buf_over	(args->fd, -1, 1, -1, NULL);
	errs	+= dsi6ln4ao_ai_buf_under	(args->fd, -1, 1, -1, NULL);
	errs	+= _save_data				(args, chans, errs);

	gsc_label_level_dec();
	return(errs);
}



