// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI6LN4AO/samples/aout/perform.c $
// $Rev: 53778 $
// $Date: 2023-09-28 12:53:21 -0500 (Thu, 28 Sep 2023) $

// 24DSI6LN4AO: Sample Application: source file

#include "main.h"



// macros *********************************************************************

#define	PIE					(3.1415962)
#define	CYCLES_PER_PATTERN	1
#define	PATTERN_SAMPLES		100
#define	CYCLE_SAMPLES		(PATTERN_SAMPLES / CYCLES_PER_PATTERN)
#define	FILE_NAME			"output.txt"



// variables ******************************************************************

static	u32	_chan_0[PATTERN_SAMPLES];
static	u32	_chan_1[PATTERN_SAMPLES];
static	u32	_chan_2[PATTERN_SAMPLES];
static	u32	_chan_3[PATTERN_SAMPLES];



//*****************************************************************************
static int _save_sample(const args_t* args, FILE* file, int index)
{
	char		buf[128]	= "";
	int			errs		= 0;
	int			len;
	int			ret;

	if (args->chan_0)
	{
		if (args->decimal)
			sprintf(buf + strlen(buf), "%5ld", (long) _chan_0[index]);
		else
			sprintf(buf + strlen(buf), "0x%04lX", (long) _chan_0[index]);
	}

	if (args->chan_1)
	{
		if (args->decimal)
			sprintf(buf + strlen(buf), "  %5ld", (long) _chan_1[index]);
		else
			sprintf(buf + strlen(buf), "  0x%04lX", (long) _chan_1[index]);
	}

	if (args->chan_2)
	{
		if (args->decimal)
			sprintf(buf + strlen(buf), "  %5ld", (long) _chan_2[index]);
		else
			sprintf(buf + strlen(buf), "  0x%04lX", (long) _chan_2[index]);
	}

	if (args->chan_3)
	{
		if (args->decimal)
			sprintf(buf + strlen(buf), "  %5ld", (long) _chan_3[index]);
		else
			sprintf(buf + strlen(buf), "  0x%04lX", (long) _chan_3[index]);
	}

	strcat(buf, "\n");
	len	= strlen(buf);
	ret	= fprintf(file, "%s", buf);

	if (ret != len)
	{
		errs++;
		printf("FAIL <---  (File write error.)\n");
	}

	return(errs);
}



//*****************************************************************************
static int _save_data(const args_t* args)
{
	int		errs	= 0;
	FILE*	file	= NULL;
	int		i;

	gsc_label("Saving Data");

	for (;;)	// A convenience loop.
	{
		file	= fopen(FILE_NAME, "w+b");

		if (file == NULL)
		{
			printf("FAIL <---  (unable to create %s)\n", FILE_NAME);
			errs	= 1;
			break;
		}

		for (i = 0; (errs == 0) && (i < PATTERN_SAMPLES); i++)
			errs	+= _save_sample(args, file, i);

		if (errs == 0)
			printf("PASS  (%d intervals)\n", PATTERN_SAMPLES);

		fclose(file);
		break;
	}

	return(errs);
}



//*****************************************************************************
static int _write_sample(const args_t* args, int index)
{
	int		errs	= 0;
	size_t	ms_begin;
	size_t	ms_now;
	int		ret;

	for (;;)	// A convenience loop.
	{
		ms_begin	= gsc_time_delta_ms();

		if (args->chan_0)
		{
			ret	= dsi6ln4ao_ioctl(args->fd, DSI6LN4AO_IOCTL_AO_CH_0_WRITE, &_chan_0[index]);

			if (ret)
			{
				errs	= 1;
				printf("FAIL <---  (Writing value #%d to AO Chnnel 0. Status = %d.)\n", index, ret);
				break;
			}
		}

		if (args->chan_1)
		{
			ret	= dsi6ln4ao_ioctl(args->fd, DSI6LN4AO_IOCTL_AO_CH_1_WRITE, &_chan_1[index]);

			if (ret)
			{
				errs	= 1;
				printf("FAIL <---  (Writing value #%d to A1 Chnnel 0. Status = %d.)\n", index, ret);
				break;
			}
		}

		if (args->chan_2)
		{
			ret	= dsi6ln4ao_ioctl(args->fd, DSI6LN4AO_IOCTL_AO_CH_2_WRITE, &_chan_2[index]);

			if (ret)
			{
				errs	= 1;
				printf("FAIL <---  (Writing value #%d to AO Chnnel 2. Status = %d.)\n", index, ret);
				break;
			}
		}

		if (args->chan_3)
		{
			ret	= dsi6ln4ao_ioctl(args->fd, DSI6LN4AO_IOCTL_AO_CH_3_WRITE, &_chan_3[index]);

			if (ret)
			{
				errs	= 1;
				printf("FAIL <---  (Writing value #%d to AO Chnnel 3. Status = %d.)\n", index, ret);
				break;
			}
		}

		if (args->synchronous == 0)
			break;

		for (;;)
		{
			ms_now	= gsc_time_delta_ms();

			if (ms_now != ms_begin)
				break;	// We've rolled over to a new ms interval;
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _write_data(const args_t* args, int errs)
{
	time_t	end;
	int		i;
	time_t	now;

	gsc_label("Writing Data");

	for (;;)	// A convenience loop.
	{
		if (errs)
		{
			errs	= 0;
			printf("SKIPPED  (due to errors)\n");
			break;
		}

		now	= time(NULL);
		end	= now + args->seconds;

		for (;;)
		{
			for (i = 0; (errs == 0) && (i < PATTERN_SAMPLES); i++)
				errs	+= _write_sample(args, i);

			now	= time(NULL);

			if (now > end)
				break;
		}

		if (errs == 0)
			printf("PASS  (~%ld seconds @ ~1ms/sample)\n", (long) args->seconds);

		break;
	}

	return(errs);
}



//*****************************************************************************
static void _init_data(void)
{
	u32*	dest;
	int		i;
	int		samp;

	// The data is encoded in OFFSET BINARY format.
	gsc_label("Initializing Data");

	// Channel 0: sawtooth wave
	memset(_chan_0, 0, sizeof(_chan_0));
	dest	= _chan_0;

	for (i = 0, samp = 0; i < PATTERN_SAMPLES; i++)
	{
		*dest++	= (u16) (0xFFFF - (0xFFFF * samp / (CYCLE_SAMPLES - 1)));
		samp	= (samp + 1) % CYCLE_SAMPLES;
	}

	// Channel 1: ramp wave
	memset(_chan_1, 0, sizeof(_chan_1));
	dest	= _chan_1;

	for (i = 0, samp = 0; i < PATTERN_SAMPLES; i++)
	{
		*dest++	= (u16) (0xFFFF * samp / (CYCLE_SAMPLES - 1));
		samp	= (samp + 1) % CYCLE_SAMPLES;
	}

	// Channel 2: square wave
	memset(_chan_2, 0, sizeof(_chan_2));
	dest	= _chan_2;

	for (i = 0, samp = 0; i < PATTERN_SAMPLES; i++)
	{
		*dest++	= (samp < (CYCLE_SAMPLES / 2)) ? 0xFFFF : 0;
		samp	= (samp + 1) % CYCLE_SAMPLES;
	}

	// Channel 3: Sine wave
	memset(_chan_3, 0, sizeof(_chan_3));
	dest	= _chan_3;

	for (i = 0, samp = 0; i < PATTERN_SAMPLES; i++)
	{
		*dest++	= ((long) 0x7FFF) * (1 + sin((2 * PIE / CYCLE_SAMPLES) * samp));
		samp	= (samp + 1) % CYCLE_SAMPLES;
	}

	printf("DONE  (pattern length: %ld samples)\n", (long) CYCLE_SAMPLES);
}



//*****************************************************************************
int perform_tests(const args_t* args)
{
	s32			clock	= args->synchronous ? DSI6LN4AO_AO_CLK_MODE_SYNC : DSI6LN4AO_AO_CLK_MODE_ASYNC;
	const char*	comma	= "";
	int			errs	= 0;
	s32			nrate;

	gsc_label("Analog Output");
	printf("\n");
	gsc_label_level_inc();

	gsc_label("Channels");

	if (args->chan_0)
	{
		printf("#0");
		comma	= ", ";
	}

	if (args->chan_1)
	{
		printf("%s#1", comma);
		comma	= ", ";
	}

	if (args->chan_2)
	{
		printf("%s#2", comma);
		comma	= ", ";
	}

	if (args->chan_3)
		printf("%s#3", comma);

	printf("\n");

	errs	+= dsi6ln4ao_initialize		(args->fd, -1, 1);
	errs	+= dsi6ln4ao_fclk_ao_compute(args->fd, -1, 1, 1000, &nrate, NULL);
	errs	+= dsi6ln4ao_ao_nrate		(args->fd, -1, 1, nrate, NULL);
	errs	+= dsi6ln4ao_ao_clk_mode	(args->fd, -1, 1, clock, NULL);
	errs	+= dsi6ln4ao_data_format	(args->fd, -1, 1, DSI6LN4AO_DATA_FORMAT_OFF_BIN, NULL);
	errs	+= dsi6ln4ao_ao_enable		(args->fd, -1, 1, DSI6LN4AO_AO_ENABLE_YES, NULL);

	_init_data();

	if (args->save_txt)
		errs	+= _save_data(args);

	errs	+= _write_data(args, errs);
	gsc_label_level_dec();
	return(errs);
}


