// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI6LN4AO/driver/read.c $
// $Rev: 53791 $
// $Date: 2023-09-28 12:58:13 -0500 (Thu, 28 Sep 2023) $

// 24DSI6LN4AO: Device Driver: source file

#include "main.h"



//*****************************************************************************
static void _dev_io_sw_init(dev_data_t* dev, dev_io_t* io)
{
	io->dma_threshold		= DSI6LN4AO_RX_IO_DMA_THRESH_DEF;
	io->io_mode				= DSI6LN4AO_IO_MODE_DEFAULT;
	io->overflow_check		= DSI6LN4AO_IO_OVERFLOW_DEFAULT;
	io->pio_threshold		= DSI6LN4AO_RX_IO_PIO_THRESH_DEF;
	io->timeout_s			= DSI6LN4AO_IO_TIMEOUT_DEFAULT;
	io->underflow_check		= DSI6LN4AO_IO_UNDERFLOW_DEFAULT;
}



//*****************************************************************************
static void _dev_io_close(dev_data_t* dev, dev_io_t* io)
{
	io->dma_channel	= NULL;
}



//*****************************************************************************
static void _dev_io_open(dev_data_t* dev, dev_io_t* io)
{
	_dev_io_sw_init(dev, io);
}



//*****************************************************************************
static int _dev_io_startup(dev_data_t* dev, dev_io_t* io)
{
	u32	reg;
	int	ret	= 0;

	if ((io->overflow_check) || (io->underflow_check))
	{
		reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ibcr_32);

		if ((io->overflow_check) && (reg & D24))
			ret	= -EIO;

		if ((io->underflow_check) && (reg & D25))
			ret	= -EIO;
	}

	return(ret);
}



//*****************************************************************************
static long _dev_pio_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	u32	available;
	u32	reg;

	reg			= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ibsr_32);
	available	= reg & 0x7FFFF;
	bytes		= available * 4;
	return(bytes);
}



//*****************************************************************************
static long _dev_pio_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	long	qty;

	qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	return(qty);
}



//*****************************************************************************
static long _dev_bmdma_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	s32	available;

	available	= _dev_pio_available(dev, io, bytes);

	if (available >= (s32) bytes)
		bytes	= available;
	else if (available < (io->dma_threshold * 4))
		bytes	= 0;
	else
		bytes	= available;

	return(bytes);
}



//*****************************************************************************
static long _dev_bmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	long			qty;
	long			samples	= bytes / 4;
	gsc_dma_setup_t	setup;

	if (samples < io->pio_threshold)
	{
		qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	}
	else
	{
		memset(&setup, 0, sizeof(gsc_dma_setup_t));
		setup.alt		= dev;
		setup.dev		= dev;
		setup.io		= io;
		setup.mem		= mem;
		setup.st_end	= st_end;
		setup.bytes		= bytes;
		setup.ability	= GSC_DMA_CAP_BMDMA_READ;

		setup.mode		= GSC_DMA_MODE_BLOCK_DMA
						| GSC_DMA_MODE_SIZE_32_BITS
						| GSC_DMA_MODE_INPUT_ENABLE
						| GSC_DMA_MODE_BURSTING_LOCAL
						| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
						| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
						| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		setup.dpr		= GSC_DMA_DPR_BOARD_TO_HOST
						| GSC_DMA_DPR_END_OF_CHAIN
						| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty	= gsc_dma_perform(&setup);
	}

	return(qty);
}



//*****************************************************************************
static long _dev_dmdma_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	return(bytes);
}



//*****************************************************************************
static long _dev_dmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	long			qty;
	long			samples	= bytes / 4;
	gsc_dma_setup_t	setup;

	if (samples < io->pio_threshold)
	{
		qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	}
	else
	{
		memset(&setup, 0, sizeof(gsc_dma_setup_t));
		setup.alt		= dev;
		setup.dev		= dev;
		setup.io		= io;
		setup.mem		= mem;
		setup.st_end	= st_end;
		setup.bytes		= bytes;
		setup.ability	= GSC_DMA_CAP_DMDMA_READ;

		setup.mode		= GSC_DMA_MODE_DM_DMA
						| GSC_DMA_MODE_SIZE_32_BITS
						| GSC_DMA_MODE_INPUT_ENABLE
						| GSC_DMA_MODE_BURSTING_LOCAL
						| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
						| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
						| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		setup.dpr		= GSC_DMA_DPR_BOARD_TO_HOST
						| GSC_DMA_DPR_END_OF_CHAIN
						| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty	= gsc_dma_perform(&setup);
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	dev_read_create
*
*	Purpose:
*
*		Perform a one-tine setup of the Read Input streaming I/O structure.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*		io		The I/O structure for this stream.
*
*	Returned:
*
*		None.
*
******************************************************************************/

int dev_read_create(dev_data_t* dev, dev_io_t* io)
{
	int	ret;

	io->bytes_per_sample	= 4;
	io->io_reg_offset		= GSC_REG_OFFSET(DSI6LN4AO_GSC_IDBR);
	io->io_reg_vaddr		= dev->vaddr.gsc_idbr_32;

	io->dev_io_sw_init		= _dev_io_sw_init;
	io->dev_io_close		= _dev_io_close;
	io->dev_io_open			= _dev_io_open;
	io->dev_io_startup		= _dev_io_startup;
	io->dev_pio_available	= _dev_pio_available;
	io->dev_bmdma_available	= _dev_bmdma_available;
	io->dev_dmdma_available	= _dev_dmdma_available;
	io->dev_pio_xfer		= _dev_pio_xfer;
	io->dev_bmdma_xfer		= _dev_bmdma_xfer;
	io->dev_dmdma_xfer		= _dev_dmdma_xfer;

	io->wait.abort			= DSI6LN4AO_WAIT_IO_RX_ABORT;
	io->wait.done			= DSI6LN4AO_WAIT_IO_RX_DONE;
	io->wait.error			= DSI6LN4AO_WAIT_IO_RX_ERROR;
	io->wait.timeout		= DSI6LN4AO_WAIT_IO_RX_TIMEOUT;

	ret	= gsc_io_create(dev, io, dev->cache.ai_fifo_size * 4);
	return(ret);
}


