// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI6LN4AO/driver/main.h $
// $Rev: 53791 $
// $Date: 2023-09-28 12:58:13 -0500 (Thu, 28 Sep 2023) $

// 24DSI6LN4AO: Device Driver: header file

#ifndef	__MAIN_H__
#define	__MAIN_H__

#include "24dsi6ln4ao.h"

#define	DEV_BAR_SHOW						0
#define	DEV_METRICS_SHOW					0
#define	DEV_PCI_ID_SHOW						0
#define	DEV_SUPPORTS_IRQ
#define	DEV_SUPPORTS_READ
#define	DEV_SUPPORTS_WAIT
#define	GSC_ALT_DEV_GET(a)					(a)
#define	GSC_ALT_STRUCT_GET(i,d)				(d)
#define	GSC_ALT_STRUCT_T					dev_data_t
#define	GSC_DEVS_PER_BOARD					1

typedef struct _dev_data_t					dev_data_t;

#include "gsc_main.h"



// macros *********************************************************************

#define	DEV_MODEL							"24DSI6LN4AO"	// Upper case form of the below.
#define	DEV_NAME							"24dsi6ln4ao"	// MUST AGREE WITH DSI6LN4AO_BASE_NAME

#define	DEV_VERSION							"1.0"		// FOR DEVICE SPECIFIC CODE ONLY!
// 1.0	Initial release.

// I/O services
#define	DEV_IO_STREAM_QTY					(DEV_IO_RX_STREAM_QTY + DEV_IO_TX_STREAM_QTY)
#define	DEV_IO_RX_STREAM_QTY				1
#define	DEV_IO_TX_STREAM_QTY				0

#define	GSC_READ_PIO_WORK_32_BIT

// WAIT services
#define	DEV_WAIT_GSC_ALL					DSI6LN4AO_WAIT_GSC_ALL
#define	DEV_WAIT_ALT_ALL					DSI6LN4AO_WAIT_ALT_ALL
#define	DEV_WAIT_IO_ALL						DSI6LN4AO_WAIT_IO_ALL



// data types *****************************************************************

struct _dev_io_t
{
	// Initialized by open and IOCTL services (initialize and/or service specific).

	s32					dma_threshold;		// Say bytes = zero if samples < this.
	s32					io_mode;			// PIO, DMA, DMDMA
	s32					overflow_check;		// Check overflow when performing I/O
	s32					pio_threshold;		// Use PIO if samples <= this.
	s32					timeout_s;			// I/O timeout in seconds.
	s32					underflow_check;	// Check underflow when performing I/O

	// Initialized by I/O service at time of use.

	int					abort;				// Also set via IOCTL service.
	gsc_dma_ch_t*		dma_channel;		// Use this channel for DMA.
	int					non_blocking;		// Is this non-blocking I/O?

	// Initialized by device specific I/O create code.

	int					bytes_per_sample;	// Sample size in bytes.
	u32					io_reg_offset;		// Offset of device's I/O FIFO.
	VADDR_T				io_reg_vaddr;		// Address of device's I/O FIFO.

	void				(*dev_io_sw_init)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	void				(*dev_io_close)			(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	void				(*dev_io_open)			(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	int					(*dev_io_startup)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	long				(*dev_pio_available)	(GSC_ALT_STRUCT_T* alt, dev_io_t* io, size_t count);
	long				(*dev_bmdma_available)	(GSC_ALT_STRUCT_T* alt, dev_io_t* io, size_t count);
	long				(*dev_dmdma_available)	(GSC_ALT_STRUCT_T* alt, dev_io_t* io, size_t count);
	long				(*dev_pio_xfer)			(GSC_ALT_STRUCT_T* alt, dev_io_t* io, const os_mem_t* mem, size_t count, os_time_tick_t st_end);
	long				(*dev_bmdma_xfer)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io, const os_mem_t* mem, size_t count, os_time_tick_t st_end);
	long				(*dev_dmdma_xfer)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io, const os_mem_t* mem, size_t count, os_time_tick_t st_end);

	struct
	{
		u32				abort;		// ...WAIT_IO_XXX_ABORT
		u32				done;		// ...WAIT_IO_XXX_DONE
		u32				error;		// ...WAIT_IO_XXX_ERROR
		u32				timeout;	// ...WAIT_IO_XXX_TIMEOUT
	} wait;

	// Initialize by GSC commone create code.

	os_mem_t			mem;				// I/O buffer.
	os_sem_t			sem;				// Only one Tx or Rx at a time.
};

struct _dev_data_t
{
	// Put an integral data type at beginning of structure.
	int					board_index;	// Index of the device being accessed.
	gsc_dev_type_t		board_type;		// Corresponds to basic device type.
	const char*			model;			// Base model number in upper case.
	int					share;			// Were we opened in shared mode?
	int					users;			// Number of currently active open() requests.

	os_pci_t			pci;			// The kernel PCI device descriptor.
	os_data_t			os;				// OS specific data.
	os_spinlock_t		spinlock;		// Control ISR access.
	os_sem_t			sem;			// Control thread access.

	gsc_bar_t			bar;			// device register mappings
	gsc_dma_t			dma;			// For DMA based I/O.
	gsc_irq_t			irq;			// For interrupts.
	gsc_wait_node_t*	wait_list;

	struct
	{					// This is for streaming I/O
		dev_io_t		rx;				// For read operations.
		dev_io_t*		io_streams[DEV_IO_STREAM_QTY];
	} io;

	struct
	{
		VADDR_T			plx_intcsr_32;	// Interrupt Control/Status Register
		VADDR_T			plx_dmaarb_32;	// DMA Arbitration Register
		VADDR_T			plx_dmathr_32;	// DMA Threshold Register

		VADDR_T			gsc_bctlr_32;	// 0x00 Board Control Register
		VADDR_T			gsc_diopr_32;	// 0x04 Digital I/O Port Register
		VADDR_T			gsc_oc0r_32;	// 0x08 Output Ch 0 Register
		VADDR_T			gsc_oc1r_32;	// 0x0C Output Ch 1 Register
		VADDR_T			gsc_oc2r_32;	// 0x10 Output Ch 2 Register
		VADDR_T			gsc_oc3r_32;	// 0x14 Output Ch 3 Register
		VADDR_T			gsc_idbr_32;	// 0x18 Input Data Buffer Register
		VADDR_T			gsc_irar_32;	// 0x1C Input Rate Assignment Register
		VADDR_T			gsc_irdr_32;	// 0x20 Input Rate Divisor Register
		VADDR_T			gsc_ibcr_32;	// 0x24 Input Buffer Control Register
		VADDR_T			gsc_ibsr_32;	// 0x28 Input Buffer Size Register
		VADDR_T			gsc_bcfgr_32;	// 0x34 Board Configuration Register
		VADDR_T			gsc_ordr_32;	// 0x38 Output Rate Divisor Register

	} vaddr;

	struct
	{
		u32				gsc_bcfgr_32;		// Board Configuration Register

		s32				ai_chan_max;		// maximum possible AI channels
		s32				ai_chan_qty;		// current number of AI channels
		s32				ai_fifo_size;		// size in 32-bit values
		s32				ai_filter;			// from BCFGR
		s32				ai_fref;			// AI Fref frequency
		s32				ai_fsamp_max;		// maximum AI Fsamp
		s32				ai_fsamp_min;		// minimum AI Faamp
		s32				ai_ndiv_max;		// maximum AI Ndiv
		s32				ai_ndiv_min;		// minimum AI Ndiv
		s32				ai_nref_max;		// maximum AI Nref
		s32				ai_nref_max_opt;	// maximum optimal AI Nref
		s32				ai_nref_min;		// minimum AI Nref
		s32				ai_nvco_max;		// maximum AI Nvco
		s32				ai_nvco_max_opt;	// maximum optimal AI Nvco
		s32				ai_nvco_min;		// minimum AI Nvco
		s32				ai_range;			// AI voltage range

		s32				ao_chan_max;		// maximum possible AO channels
		s32				ao_chan_qty;		// current number of AO channels
		s32				ao_fclk_max;		// maximum AO clocking rate
		s32				ao_fclk_min;		// minimum AO clocking rate
		s32				ao_fref;			// AO Fref frequency
		s32				ao_nrate_max;		// maximum AO Nrate
		s32				ao_nrate_min;		// minimum AO Nrate

		s32				board_type;			// board type number

		s32				initialize_ms;		// initialization time in ms

		s32				temperature;		// temperature order option
	} cache;
};



// prototypes *****************************************************************

void		dev_io_close(dev_data_t* dev);
int			dev_io_create(dev_data_t* dev);
void		dev_io_destroy(dev_data_t* dev);
int			dev_io_open(dev_data_t* dev);
dev_io_t*	dev_io_read_select(dev_data_t* dev, size_t bytes);
int			dev_irq_create(dev_data_t* dev);
void		dev_irq_destroy(dev_data_t* dev);
int			dev_read_create(dev_data_t* dev, dev_io_t* io);

int			initialize_ioctl(dev_data_t* dev, void* arg);



#endif
