// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI6LN4AO/driver/ioctl.c $
// $Rev: 53791 $
// $Date: 2023-09-28 12:58:13 -0500 (Thu, 28 Sep 2023) $

// 24DSI6LN4AO: Device Driver: source file

#include "main.h"



// macros *********************************************************************

#define	INIT_START				D31		// BCTLR

#define	IRQ_INIT_DONE			0x0000	// BCTLR
#define	IRQ_MASK				0x0700	// BCTLR
#define	IRQ_ACTIVE				0x0800	// BCTLR



//*****************************************************************************
static void _bctlr_mod(dev_data_t* dev, int lock, u32 value, u32 mask)
{
	// BCTLR must be modified by this routine to prevent the interrupt clear
	// bit from being cleared unintentionally.

	value	|= IRQ_ACTIVE;	// Make sure we never clear this bit.

	if (lock)
		os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, value, mask);
	else
		os_reg_mem_mx_u32(NULL, dev->vaddr.gsc_bctlr_32, value, mask);
}



//*****************************************************************************
static int _query(dev_data_t* dev, s32* arg)
{
	int		ret	= 0;

	switch (arg[0])
	{
		default:	arg[0]	= DSI6LN4AO_IOCTL_QUERY_ERROR;
					ret		= -EINVAL;
					break;

		case DSI6LN4AO_QUERY_AI_CHAN_MAX:		arg[0]	= dev->cache.ai_chan_max;		break;
		case DSI6LN4AO_QUERY_AI_CHAN_QTY:		arg[0]	= dev->cache.ai_chan_qty;		break;
		case DSI6LN4AO_QUERY_AI_FIFO_SIZE:		arg[0]	= dev->cache.ai_fifo_size;		break;
		case DSI6LN4AO_QUERY_AI_FILTER:			arg[0]	= dev->cache.ai_filter;			break;
		case DSI6LN4AO_QUERY_AI_FREF:			arg[0]	= dev->cache.ai_fref;			break;
		case DSI6LN4AO_QUERY_AI_FSAMP_MAX:		arg[0]	= dev->cache.ai_fsamp_max;		break;
		case DSI6LN4AO_QUERY_AI_FSAMP_MIN:		arg[0]	= dev->cache.ai_fsamp_min;		break;
		case DSI6LN4AO_QUERY_AI_NDIV_MAX:		arg[0]	= dev->cache.ai_ndiv_max;		break;
		case DSI6LN4AO_QUERY_AI_NDIV_MIN:		arg[0]	= dev->cache.ai_ndiv_min;		break;
		case DSI6LN4AO_QUERY_AI_NREF_MAX:		arg[0]	= dev->cache.ai_nref_max;		break;
		case DSI6LN4AO_QUERY_AI_NREF_MAX_OPT:	arg[0]	= dev->cache.ai_nref_max_opt;	break;
		case DSI6LN4AO_QUERY_AI_NREF_MIN:		arg[0]	= dev->cache.ai_nref_min;		break;
		case DSI6LN4AO_QUERY_AI_NVCO_MAX:		arg[0]	= dev->cache.ai_nvco_max;		break;
		case DSI6LN4AO_QUERY_AI_NVCO_MAX_OPT:	arg[0]	= dev->cache.ai_nvco_max_opt;	break;
		case DSI6LN4AO_QUERY_AI_NVCO_MIN:		arg[0]	= dev->cache.ai_nvco_min;		break;
		case DSI6LN4AO_QUERY_AI_RANGE:			arg[0]	= dev->cache.ai_range;			break;
		case DSI6LN4AO_QUERY_AO_CHAN_MAX:		arg[0]	= dev->cache.ao_chan_max;		break;
		case DSI6LN4AO_QUERY_AO_CHAN_QTY:		arg[0]	= dev->cache.ao_chan_qty;		break;
		case DSI6LN4AO_QUERY_AO_FCLK_MAX:		arg[0]	= dev->cache.ao_fclk_max;		break;
		case DSI6LN4AO_QUERY_AO_FCLK_MIN:		arg[0]	= dev->cache.ao_fclk_min;		break;
		case DSI6LN4AO_QUERY_AO_FREF:			arg[0]	= dev->cache.ao_fref;			break;
		case DSI6LN4AO_QUERY_AO_NRATE_MAX:		arg[0]	= dev->cache.ao_nrate_max;		break;
		case DSI6LN4AO_QUERY_AO_NRATE_MIN:		arg[0]	= dev->cache.ao_nrate_min;		break;
		case DSI6LN4AO_QUERY_COUNT:				arg[0]	= DSI6LN4AO_QUERY_LAST;			break;
		case DSI6LN4AO_QUERY_DEVICE_TYPE:		arg[0]	= dev->cache.board_type;		break;
		case DSI6LN4AO_QUERY_INIT_MS:			arg[0]	= dev->cache.initialize_ms;		break;
		case DSI6LN4AO_QUERY_TEMPERATURE:		arg[0]	= dev->cache.temperature;		break;
	}

	return(ret);
}



//*****************************************************************************
static int _init_start(dev_data_t* dev, void* arg)
{
	// Initiate initialization.
	_bctlr_mod(dev, 0, INIT_START, INIT_START);
	return(0);
}



//*****************************************************************************
int initialize_ioctl(dev_data_t* dev, void* arg)
{
	int			i;
	int			mask;
	long		ms			= dev->cache.initialize_ms + 5000;
	long		ms_total	= ms;
	u32			reg;
	int			ret			= 0;
	os_sem_t	sem;
	int			tmp;
	VADDR_T		va			= dev->vaddr.gsc_bctlr_32;
	int			value;
	gsc_wait_t	wt;

	if ((dev->irq.opened) && (gsc_global.driver_unloading == 0))
	{
		ms_total	*= 2;

		// Safely select the Initialize Done interrupt.
		mask	= IRQ_MASK | IRQ_ACTIVE;
		value	= IRQ_INIT_DONE;
		_bctlr_mod(dev, 1, value, mask);

		// Wait for the local interrupt.
		os_sem_create(&sem);	// dummy, required for wait operations.
		memset(&wt, 0, sizeof(wt));
		wt.flags		= GSC_WAIT_FLAG_INTERNAL;
		wt.gsc			= DSI6LN4AO_WAIT_GSC_INIT_DONE;
		wt.timeout_ms	= ms;
		ret				= gsc_wait_event(dev, &wt, _init_start, NULL, &sem);
		os_sem_destroy(&sem);

		if (wt.flags & GSC_WAIT_FLAG_TIMEOUT)
		{
			ret	= ret ? ret : -ETIMEDOUT;
			printf(	"%s: INITILIZE DONE IRQ TIMED OUT AFTER %ld ms.\n",
					dev->model,
					ms);
		}
	}
	else
	{
		_init_start(dev, NULL);
	}

	// Manually wait for completion as the IRQ wait may have ended early.
	tmp	= gsc_poll_u32(dev, ms, va, INIT_START, 0);

	if (tmp)
	{
		ret	= ret ? ret : tmp;
		printf(	"%s: INITIALIZATION DID NOT COMPLETE WITHIN %ld ms.\n",
				dev->model,
				ms_total);
	}

	// Manually check for completion as the IRQ wait may have ended early.
	reg	= os_reg_mem_rx_u32(dev, va);

	if (reg & INIT_START)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: INITILIZE STILL ACTIVE AFTER %ld ms.\n",
				dev->model,
				ms_total);
	}

	// Initialize the software settings.

	for (i = 0; i < DEV_IO_STREAM_QTY; i++)
	{
		if (dev->io.io_streams[i])
		{
			if (dev->io.io_streams[i]->dev_io_sw_init)
			{
				(dev->io.io_streams[i]->dev_io_sw_init)(dev, dev->io.io_streams[i]);
			}
		}
	}

	return(ret);
}



/*****************************************************************************/
static int _ai_buf_clear(dev_data_t* dev, void* arg)
{
	#define	CLEAR	D19		// IBCR
	#define	OVER	D24		// IBCR
	#define	UNDER	D25		// IBCR

	int		i;
	u32		reg;
	int		ret	= 0;
	VADDR_T	va	= dev->vaddr.gsc_ibcr_32;

	// Clear the buffer.
	os_reg_mem_mx_u32(dev, va, CLEAR, CLEAR);

	// Wait for the bit to clear.

	for (i = 0;; i++)
	{
		reg	= os_reg_mem_rx_u32(dev, va);

		if ((reg & CLEAR) == 0)
			break;

		if (i >= 250)
		{
			ret	= -EINVAL;
			printf(	"%s: The analog input buffer took too long to clear.\n",
					DEV_NAME);
			break;
		}
	}

	// Clear the Overflow status bit.
	os_reg_mem_mx_u32(dev, va, 0, OVER);

	// Clear the Underflow status bit.
	os_reg_mem_mx_u32(dev, va, 0, UNDER);

	return(ret);
}



/*****************************************************************************/
static int _ai_buf_enable(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI6LN4AO_AI_BUF_ENABLE_NO,
		DSI6LN4AO_AI_BUF_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_ibcr_32, 18, 18);
	return(ret);
}



/*****************************************************************************/
static int _ai_buf_level(dev_data_t* dev, s32* arg)
{
	u32	reg;

	reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ibsr_32);
	arg[0]	= reg & 0x7FFFF;
	return(0);
}



/*****************************************************************************/
static int _ai_buf_over(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI6LN4AO_AI_BUF_OVER_NO,
		DSI6LN4AO_AI_BUF_OVER_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_ibcr_32, 24, 24);
	return(ret);
}



/*****************************************************************************/
static int _ai_buf_status(dev_data_t* dev, s32* arg)
{
	u32	reg;

	reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);

	if (reg & D23)
		arg[0]	= DSI6LN4AO_AI_BUF_STATUS_EMPTY;
	else if (reg & D24)
		arg[0]	= DSI6LN4AO_AI_BUF_STATUS_FULL;
	else if (reg & D25)
		arg[0]	= DSI6LN4AO_AI_BUF_STATUS_AE;
	else
		arg[0]	= DSI6LN4AO_AI_BUF_STATUS_AF;

	return(0);
}



/*****************************************************************************/
static int _ai_buf_under(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI6LN4AO_AI_BUF_UNDER_NO,
		DSI6LN4AO_AI_BUF_UNDER_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_ibcr_32, 25, 25);
	return(ret);
}



/*****************************************************************************/
static int _ai_chan_last(dev_data_t* dev, s32* arg)
{
	u32	mask;
	u32	reg;
	int	ret		= 0;
	u32	value;

	switch (arg[0])
	{
		case 4:
		case 5:

			if (dev->cache.ai_chan_qty == 4)
			{
				ret	= -EINVAL;
				break;
			}

			FALLTHROUGH;	/* fall through */

		case 0:
		case 1:
		case 2:
		case 3:

			mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, 18, 16);
			value	= GSC_FIELD_ENCODE(arg[0], 18, 16);
			_bctlr_mod(dev, 1, value, mask);

			FALLTHROUGH;	/* fall through */

		case -1:

			reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);
			arg[0]	= GSC_FIELD_DECODE(reg, 18, 16);
			break;

		default:

			ret	= -EINVAL;
			break;
	}

	return(ret);
}



/*****************************************************************************/
static int _ai_coupling(dev_data_t* dev, s32* arg)
{
	u32	mask;
	u32	reg;
	int	ret		= 0;
	u32	value;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case DSI6LN4AO_AI_COUPLING_AC:
		case DSI6LN4AO_AI_COUPLING_DC:

			mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, 22, 22);
			value	= GSC_FIELD_ENCODE(arg[0], 22, 22);
			_bctlr_mod(dev, 1, value, mask);

			FALLTHROUGH;	/* fall through */

		case -1:

			reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);
			arg[0]	= GSC_FIELD_DECODE(reg, 22, 22);
			break;
	}

	return(ret);
}



/*****************************************************************************/
static int _ai_data_width(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI6LN4AO_AI_DATA_WIDTH_16,
		DSI6LN4AO_AI_DATA_WIDTH_18,
		DSI6LN4AO_AI_DATA_WIDTH_20,
		DSI6LN4AO_AI_DATA_WIDTH_24,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_ibcr_32, 21, 20);
	return(ret);
}



/*****************************************************************************/
static int _ai_mode(dev_data_t* dev, s32* arg)
{
	u32	mask;
	u32	reg;
	int	ret		= 0;
	u32	value;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case DSI6LN4AO_AI_MODE_SE:
		case DSI6LN4AO_AI_MODE_DIFF:

			mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, 0, 0);
			value	= GSC_FIELD_ENCODE(arg[0], 0, 0);
			_bctlr_mod(dev, 1, value, mask);

			FALLTHROUGH;	/* fall through */

		case -1:

			reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);
			arg[0]	= GSC_FIELD_DECODE(reg, 0, 0);
			break;
	}

	return(ret);
}



/*****************************************************************************/
static int _ai_ndiv(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(
			dev,
			arg,
			dev->cache.ai_ndiv_min,
			dev->cache.ai_ndiv_max,
			dev->vaddr.gsc_irdr_32,
			5,
			0);
	return(ret);
}



/*****************************************************************************/
static int _ai_nref(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(
			dev,
			arg,
			dev->cache.ai_nref_min,
			dev->cache.ai_nref_max,
			dev->vaddr.gsc_irar_32,
			25,
			16);
	return(ret);
}



/*****************************************************************************/
static int _ai_nvco(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(
			dev,
			arg,
			dev->cache.ai_nvco_min,
			dev->cache.ai_nvco_max,
			dev->vaddr.gsc_irar_32,
			9,
			0);
	return(ret);
}



/*****************************************************************************/
static int _ai_ready(dev_data_t* dev, s32* arg)
{
	u32	reg;
	int	ret	= 0;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case DSI6LN4AO_AI_READY_WAIT:

			ret	= gsc_poll_u32(dev, 500, dev->vaddr.gsc_bctlr_32, D20, 0);

			FALLTHROUGH;	/* fall through */

		case -1:

			reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);
			arg[0]	= GSC_FIELD_DECODE(reg, 20, 20);
			break;
	}

	return(ret);
}



/*****************************************************************************/
static int _ai_sampling(dev_data_t* dev, s32* arg)
{
	u32	mask;
	u32	reg;
	int	ret		= 0;
	u32	value;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case DSI6LN4AO_AI_SAMPLING_BURST:
		case DSI6LN4AO_AI_SAMPLING_CONT:

			mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, 1, 1);
			value	= GSC_FIELD_ENCODE(arg[0], 1, 1);
			_bctlr_mod(dev, 1, value, mask);

			FALLTHROUGH;	/* fall through */

		case -1:

			reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);
			arg[0]	= GSC_FIELD_DECODE(reg, 1, 1);
			break;
	}

	return(ret);
}



/*****************************************************************************/
static int _ai_sync(dev_data_t* dev, s32* arg)
{
	_bctlr_mod(dev, 1, D7, D7);
	return(0);
}



/*****************************************************************************/
static int _ai_thresh(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0x3FFFF, dev->vaddr.gsc_ibcr_32, 17, 0);
	return(ret);
}



/*****************************************************************************/
static int _ai_thresh_sts(dev_data_t* dev, s32* arg)
{
	u32	reg;

	reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);
	arg[0]	= GSC_FIELD_DECODE(reg, 25, 25);
	return(0);
}



/*****************************************************************************/
static int _ai_trigger(dev_data_t* dev, void* arg)
{
	_bctlr_mod(dev, 1, D21, D21);
	return(0);
}



/*****************************************************************************/
static int _ao_ch_0_write(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFF, dev->vaddr.gsc_oc0r_32, 15, 0);
	return(ret);
}



/*****************************************************************************/
static int _ao_ch_1_write(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFF, dev->vaddr.gsc_oc1r_32, 15, 0);
	return(ret);
}



/*****************************************************************************/
static int _ao_ch_2_write(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFF, dev->vaddr.gsc_oc2r_32, 15, 0);
	return(ret);
}



/*****************************************************************************/
static int _ao_ch_3_write(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFF, dev->vaddr.gsc_oc3r_32, 15, 0);
	return(ret);
}



/*****************************************************************************/
static int _ao_clk_mode(dev_data_t* dev, s32* arg)
{
	u32	mask;
	u32	reg;
	int	ret		= 0;
	u32	value;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case DSI6LN4AO_AO_CLK_MODE_ASYNC:
		case DSI6LN4AO_AO_CLK_MODE_SYNC:

			mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, 29, 29);
			value	= GSC_FIELD_ENCODE(arg[0], 29, 29);
			_bctlr_mod(dev, 1, value, mask);

			FALLTHROUGH;	/* fall through */

		case -1:

			reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);
			arg[0]	= GSC_FIELD_DECODE(reg, 29, 29);
			break;
	}

	return(ret);
}



/*****************************************************************************/
static int _ao_clk_src(dev_data_t* dev, s32* arg)
{
	u32	mask;
	u32	reg;
	int	ret		= 0;
	u32	value;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case DSI6LN4AO_AO_CLK_SRC_EXT:
		case DSI6LN4AO_AO_CLK_SRC_INT:

			mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, 26, 26);
			value	= GSC_FIELD_ENCODE(arg[0], 26, 26);
			_bctlr_mod(dev, 1, value, mask);

			FALLTHROUGH;	/* fall through */

		case -1:

			reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);
			arg[0]	= GSC_FIELD_DECODE(reg, 26, 26);
			break;
	}

	return(ret);
}



/*****************************************************************************/
static int _ao_enable(dev_data_t* dev, s32* arg)
{
	u32	mask;
	u32	reg;
	int	ret		= 0;
	u32	value;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case DSI6LN4AO_AO_ENABLE_NO:
		case DSI6LN4AO_AO_ENABLE_YES:

			mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, 30, 30);
			value	= GSC_FIELD_ENCODE(arg[0], 30, 30);
			_bctlr_mod(dev, 1, value, mask);

			FALLTHROUGH;	/* fall through */

		case -1:

			reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);
			arg[0]	= GSC_FIELD_DECODE(reg, 30, 30);
			break;
	}

	return(ret);
}



/*****************************************************************************/
static int _ao_nrate(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(
			dev,
			arg,
			dev->cache.ao_nrate_min,
			dev->cache.ao_nrate_max,
			dev->vaddr.gsc_ordr_32,
			23,
			0);
	return(ret);
}



/*****************************************************************************/
static int _ao_range(dev_data_t* dev, s32* arg)
{
	u32	mask;
	u32	reg;
	int	ret		= 0;
	u32	value;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case DSI6LN4AO_AO_RANGE_2_5V:
		case DSI6LN4AO_AO_RANGE_5V:
		case DSI6LN4AO_AO_RANGE_10V:

			mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, 3, 2);
			value	= GSC_FIELD_ENCODE(arg[0], 3, 2);
			_bctlr_mod(dev, 1, value, mask);

			FALLTHROUGH;	/* fall through */

		case -1:

			reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);
			arg[0]	= GSC_FIELD_DECODE(reg, 3, 2);
			break;
	}

	return(ret);
}



/*****************************************************************************/
static int _ao_ready(dev_data_t* dev, s32* arg)
{
	u32	reg;
	int	ret	= 0;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case DSI6LN4AO_AO_READY_WAIT:

			ret	= gsc_poll_u32(dev, 500, dev->vaddr.gsc_bctlr_32, D27, 1);

			FALLTHROUGH;	/* fall through */

		case -1:

			reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);
			arg[0]	= GSC_FIELD_DECODE(reg, 27, 27);
			break;
	}

	return(ret);
}



/*****************************************************************************/
static int _ao_trigger(dev_data_t* dev, void* arg)
{
	_bctlr_mod(dev, 1, D28, D28);
	return(0);
}



//*****************************************************************************
static int _data_format(dev_data_t* dev, s32* arg)
{
	u32	mask;
	u32	reg;
	int	ret		= 0;
	u32	value;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case DSI6LN4AO_DATA_FORMAT_2S_COMP:
		case DSI6LN4AO_DATA_FORMAT_OFF_BIN:

			mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, 5, 5);
			value	= GSC_FIELD_ENCODE(arg[0], 5, 5);
			_bctlr_mod(dev, 1, value, mask);

			FALLTHROUGH;	/* fall through */

		case -1:

			reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);
			arg[0]	= GSC_FIELD_DECODE(reg, 5, 5);
			break;
	}

	return(ret);
}



/*****************************************************************************/
static int _gpio_dir(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI6LN4AO_GPIO_DIR_INPUT,
		DSI6LN4AO_GPIO_DIR_OUTPUT,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_diopr_32, 10, 10);
	return(ret);
}



/*****************************************************************************/
static int _gpio_rx(dev_data_t* dev, s32* arg)
{
	u32	reg;

	reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_diopr_32);
	arg[0]	= reg & 0x3FF;
	return(0);
}



/*****************************************************************************/
static int _gpio_tx(dev_data_t* dev, s32* arg)
{
	int	ret	= 0;

	for (;;)	// A convenience loop.
	{
		if (arg[0] == -1)
		{
			_gpio_rx(dev, arg);
			break;
		}

		if (arg[0] & 0xFFFFFC00)	// The DEDICATED INPUT bit is ignored.
		{
			ret	= -EINVAL;
			break;
		}

		os_reg_mem_mx_u32(dev, dev->vaddr.gsc_diopr_32, arg[0], 0x3FF);
		break;
	}

	return(ret);
}



/*****************************************************************************/
static int _irq_sel(dev_data_t* dev, s32* arg)
{
	u32	mask;
	u32	reg;
	int	ret		= 0;
	u32	value;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case DSI6LN4AO_IRQ_INIT_DONE:
		case DSI6LN4AO_IRQ_AI_READY_L2H:
		case DSI6LN4AO_IRQ_AI_BUF_THR_L2H:
		case DSI6LN4AO_IRQ_AI_BUF_THR_H2L:
		case DSI6LN4AO_IRQ_AI_BUF_ERROR:
		case DSI6LN4AO_IRQ_AO_CLOCK_DONE:

			mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, 10, 8);
			value	= GSC_FIELD_ENCODE(arg[0], 10, 8);
			_bctlr_mod(dev, 1, value, mask);

			FALLTHROUGH;	/* fall through */

		case -1:

			reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);
			arg[0]	= GSC_FIELD_DECODE(reg, 10, 8);
			break;
	}

	return(ret);
}



//*****************************************************************************
static int _rx_io_abort(dev_data_t* dev, s32* arg)
{
	arg[0]	= gsc_read_abort_active_xfer(dev, &dev->io.rx);
	return(0);
}



//*****************************************************************************
static int _rx_io_dma_thresh(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret		= gsc_s32_range_var(arg, 0, S32_MAX, &dev->io.rx.dma_threshold);
	return(ret);
}



/*****************************************************************************/
static int _rx_io_mode(dev_data_t* dev, s32* arg)
{
	static const s32	list[]	=
	{
		GSC_IO_MODE_PIO,
		GSC_IO_MODE_BMDMA,
		GSC_IO_MODE_DMDMA,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.rx.io_mode);
	return(ret);
}



/*****************************************************************************/
static int _rx_io_overflow(dev_data_t* dev, s32* arg)
{
	static const s32	list[]	=
	{
		DSI6LN4AO_IO_OVERFLOW_IGNORE,
		DSI6LN4AO_IO_OVERFLOW_CHECK,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.rx.overflow_check);
	return(ret);
}



//*****************************************************************************
static int _rx_io_pio_thresh(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret		= gsc_s32_range_var(arg, 0, S32_MAX, &dev->io.rx.pio_threshold);
	return(ret);
}



/*****************************************************************************/
static int _rx_io_timeout(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_var(
			arg,
			DSI6LN4AO_IO_TIMEOUT_MIN,
			DSI6LN4AO_IO_TIMEOUT_INFINITE,
			&dev->io.rx.timeout_s);
	return(ret);
}



/*****************************************************************************/
static int _rx_io_underflow(dev_data_t* dev, s32* arg)
{
	static const s32	list[]	=
	{
		DSI6LN4AO_IO_UNDERFLOW_IGNORE,
		DSI6LN4AO_IO_UNDERFLOW_CHECK,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.rx.underflow_check);
	return(ret);
}



// variables ******************************************************************

const gsc_ioctl_t	dev_ioctl_list[]	=
{
	{ DSI6LN4AO_IOCTL_REG_READ,			(void*) gsc_reg_read_ioctl		},
	{ DSI6LN4AO_IOCTL_REG_WRITE,		(void*) gsc_reg_write_ioctl		},
	{ DSI6LN4AO_IOCTL_REG_MOD,			(void*) gsc_reg_mod_ioctl		},
	{ DSI6LN4AO_IOCTL_QUERY,			(void*) _query					},
	{ DSI6LN4AO_IOCTL_INITIALIZE,		(void*) initialize_ioctl		},
	{ DSI6LN4AO_IOCTL_AI_BUF_CLEAR,		(void*) _ai_buf_clear			},
	{ DSI6LN4AO_IOCTL_AI_BUF_ENABLE,	(void*) _ai_buf_enable			},
	{ DSI6LN4AO_IOCTL_AI_BUF_LEVEL,		(void*) _ai_buf_level			},
	{ DSI6LN4AO_IOCTL_AI_BUF_OVER,		(void*) _ai_buf_over			},
	{ DSI6LN4AO_IOCTL_AI_BUF_STATUS,	(void*) _ai_buf_status			},
	{ DSI6LN4AO_IOCTL_AI_BUF_UNDER,		(void*) _ai_buf_under			},
	{ DSI6LN4AO_IOCTL_AI_CHAN_LAST,		(void*) _ai_chan_last			},
	{ DSI6LN4AO_IOCTL_AI_COUPLING,		(void*) _ai_coupling			},
	{ DSI6LN4AO_IOCTL_AI_DATA_WIDTH,	(void*) _ai_data_width			},
	{ DSI6LN4AO_IOCTL_AI_MODE,			(void*) _ai_mode				},
	{ DSI6LN4AO_IOCTL_AI_NDIV,			(void*) _ai_ndiv				},
	{ DSI6LN4AO_IOCTL_AI_NREF,			(void*) _ai_nref				},
	{ DSI6LN4AO_IOCTL_AI_NVCO,			(void*) _ai_nvco				},
	{ DSI6LN4AO_IOCTL_AI_READY,			(void*) _ai_ready				},
	{ DSI6LN4AO_IOCTL_AI_SAMPLING,		(void*) _ai_sampling			},
	{ DSI6LN4AO_IOCTL_AI_SYNC,			(void*) _ai_sync				},
	{ DSI6LN4AO_IOCTL_AI_THRESH,		(void*) _ai_thresh				},
	{ DSI6LN4AO_IOCTL_AI_THRESH_STS,	(void*) _ai_thresh_sts			},
	{ DSI6LN4AO_IOCTL_AI_TRIGGER,		(void*) _ai_trigger				},
	{ DSI6LN4AO_IOCTL_AO_CH_0_WRITE,	(void*) _ao_ch_0_write			},
	{ DSI6LN4AO_IOCTL_AO_CH_1_WRITE,	(void*) _ao_ch_1_write			},
	{ DSI6LN4AO_IOCTL_AO_CH_2_WRITE,	(void*) _ao_ch_2_write			},
	{ DSI6LN4AO_IOCTL_AO_CH_3_WRITE,	(void*) _ao_ch_3_write			},
	{ DSI6LN4AO_IOCTL_AO_CLK_MODE,		(void*) _ao_clk_mode			},
	{ DSI6LN4AO_IOCTL_AO_CLK_SRC,		(void*) _ao_clk_src				},
	{ DSI6LN4AO_IOCTL_AO_ENABLE,		(void*) _ao_enable				},
	{ DSI6LN4AO_IOCTL_AO_NRATE,			(void*) _ao_nrate				},
	{ DSI6LN4AO_IOCTL_AO_RANGE,			(void*) _ao_range				},
	{ DSI6LN4AO_IOCTL_AO_READY,			(void*) _ao_ready				},
	{ DSI6LN4AO_IOCTL_AO_TRIGGER,		(void*) _ao_trigger				},
	{ DSI6LN4AO_IOCTL_DATA_FORMAT,		(void*) _data_format			},
	{ DSI6LN4AO_IOCTL_GPIO_DIR,			(void*) _gpio_dir				},
	{ DSI6LN4AO_IOCTL_GPIO_RX,			(void*) _gpio_rx				},
	{ DSI6LN4AO_IOCTL_GPIO_TX,			(void*) _gpio_tx				},
	{ DSI6LN4AO_IOCTL_IRQ_SEL,			(void*) _irq_sel				},
	{ DSI6LN4AO_IOCTL_RX_IO_ABORT,		(void*) _rx_io_abort			},
	{ DSI6LN4AO_IOCTL_RX_IO_DMA_THRESH,	(void*) _rx_io_dma_thresh		},
	{ DSI6LN4AO_IOCTL_RX_IO_MODE,		(void*) _rx_io_mode				},
	{ DSI6LN4AO_IOCTL_RX_IO_OVERFLOW,	(void*) _rx_io_overflow			},
	{ DSI6LN4AO_IOCTL_RX_IO_PIO_THRESH,	(void*) _rx_io_pio_thresh		},
	{ DSI6LN4AO_IOCTL_RX_IO_TIMEOUT,	(void*) _rx_io_timeout			},
	{ DSI6LN4AO_IOCTL_RX_IO_UNDERFLOW,	(void*) _rx_io_underflow		},
	{ DSI6LN4AO_IOCTL_WAIT_CANCEL,		(void*) gsc_wait_cancel_ioctl	},
	{ DSI6LN4AO_IOCTL_WAIT_EVENT,		(void*) gsc_wait_event_ioctl	},
	{ DSI6LN4AO_IOCTL_WAIT_STATUS,		(void*) gsc_wait_status_ioctl	},
	{ -1, NULL }
};


