// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI6LN4AO/driver/device.c $
// $Rev: 53791 $
// $Date: 2023-09-28 12:58:13 -0500 (Thu, 28 Sep 2023) $

// 24DSI6LN4AO: Device Driver: source file

#include "main.h"



// variables ******************************************************************

const gsc_dev_id_t	dev_id_list[]	=
{
	// model			Vendor	Device	SubVen	SubDev	type
	{ "24DSI6LN4AO",	0x10B5, 0x9056, 0x10B5, 0x3608,	GSC_DEV_TYPE_24DSI6LN4AO	},
	{ NULL }
};



//*****************************************************************************
static void _channels_compute(dev_data_t* dev)
{
	if (dev->cache.gsc_bcfgr_32 & D16)
		dev->cache.ai_chan_qty	= 6;
	else
		dev->cache.ai_chan_qty	= 4;

	if (dev->cache.gsc_bcfgr_32 & D17)
		dev->cache.ao_chan_qty	= 4;
	else
		dev->cache.ao_chan_qty	= 0;

	dev->cache.ai_chan_max	= 6;
	dev->cache.ao_chan_max	= 4;
}



//*****************************************************************************
static void _filter_compute(dev_data_t* dev)
{
	if (dev->cache.gsc_bcfgr_32 & D20)
		dev->cache.ai_filter	= DSI6LN4AO_AI_FILTER_CUSTOM;
	else
		dev->cache.ai_filter	= DSI6LN4AO_AI_FILTER_270KHZ;
}



//*****************************************************************************
static void _range_compute(dev_data_t* dev)
{
	s32	range	= GSC_FIELD_DECODE(dev->cache.gsc_bcfgr_32, 19, 18);

	switch (range)
	{
		default:

			printf("%s: INVALID AI RANGE OPTION: %d\n", DEV_NAME, (int) range);
			break;

		case 0:	dev->cache.ai_range	= DSI6LN4AO_AI_RANGE_10;	break;
		case 1:	dev->cache.ai_range	= DSI6LN4AO_AI_RANGE_5;		break;
		case 2:	dev->cache.ai_range	= DSI6LN4AO_AI_RANGE_2_5;	break;
	}
}



//*****************************************************************************
static void _temperature_compute(dev_data_t* dev)
{
	if (dev->cache.gsc_bcfgr_32 & D21)
		dev->cache.temperature	= DSI6LN4AO_TEMPERATURE_EXT;
	else
		dev->cache.temperature	= DSI6LN4AO_TEMPERATURE_COM;
}



/******************************************************************************
*
*	Function:	dev_device_create
*
*	Purpose:
*
*		Do everything needed to setup and use the given device.
*
*	Arguments:
*
*		dev		The structure to initialize.
*
*	Returned:
*
*		0		All is well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int dev_device_create(dev_data_t* dev)
{
	static const gsc_bar_maps_t	bar_map	=
	{
		{
			// mem	io	rw
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 0: PLX registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 1: PLX registers, I/O mapped
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RW },	// BAR 2: GSC registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 3: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 4: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO }	// BAR 5: unused
		}
	};

	u32	dma;
	int	ret;

	for (;;)	// A convenience loop.
	{
		// Verify some macro contents.
		ret	= gsc_macro_test_base_name(DSI6LN4AO_BASE_NAME);
		if (ret)	break;

		ret	= gsc_macro_test_model();
		if (ret)	break;

		// PCI setup.
		ret	= os_pci_dev_enable(&dev->pci);
		if (ret)	break;

		ret	= os_pci_master_set(&dev->pci);
		if (ret)	break;

		// Control ISR access to the device and data structure.
		ret	= os_spinlock_create(&dev->spinlock);
		if (ret)	break;

		// Control access to the device and data structure.
		ret	= os_sem_create(&dev->sem);
		if (ret)	break;

		// Access the BAR regions.
		ret	= gsc_bar_create(dev, &dev->bar, &bar_map);
		if (ret)	break;

		// Firmware access.
		dev->vaddr.gsc_bctlr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(DSI6LN4AO_GSC_BCTLR));
		dev->vaddr.gsc_diopr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(DSI6LN4AO_GSC_DIOPR));
		dev->vaddr.gsc_oc0r_32		= GSC_VADDR(dev, GSC_REG_OFFSET(DSI6LN4AO_GSC_OC0R));
		dev->vaddr.gsc_oc1r_32		= GSC_VADDR(dev, GSC_REG_OFFSET(DSI6LN4AO_GSC_OC1R));
		dev->vaddr.gsc_oc2r_32		= GSC_VADDR(dev, GSC_REG_OFFSET(DSI6LN4AO_GSC_OC2R));
		dev->vaddr.gsc_oc3r_32		= GSC_VADDR(dev, GSC_REG_OFFSET(DSI6LN4AO_GSC_OC3R));
		dev->vaddr.gsc_idbr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(DSI6LN4AO_GSC_IDBR));
		dev->vaddr.gsc_irar_32		= GSC_VADDR(dev, GSC_REG_OFFSET(DSI6LN4AO_GSC_IRAR));
		dev->vaddr.gsc_irdr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(DSI6LN4AO_GSC_IRDR));
		dev->vaddr.gsc_ibcr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(DSI6LN4AO_GSC_IBCR));
		dev->vaddr.gsc_ibsr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(DSI6LN4AO_GSC_IBSR));
		dev->vaddr.gsc_bcfgr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(DSI6LN4AO_GSC_BCFGR));
		dev->vaddr.gsc_ordr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(DSI6LN4AO_GSC_ORDR));

		// Data cache initialization.
		dev->cache.gsc_bcfgr_32		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bcfgr_32);

		dev->cache.ai_fifo_size		= _256K;	// size in 32-bit values
		dev->cache.ai_fref			= 32768000;	// AI Fref frequency
		dev->cache.ai_fsamp_max		= 200000;	// maximum AI Fsamp
		dev->cache.ai_fsamp_min		= 2000;		// minimum AI Fsamp
		dev->cache.ai_ndiv_max		= 25;		// maximum AI Ndiv
		dev->cache.ai_ndiv_min		= 0;		// minimum AI Ndiv
		dev->cache.ai_nref_max		= 1000;		// maximum AI Nref
		dev->cache.ai_nref_max_opt	= 99;		// maximum optimal AI Nref
		dev->cache.ai_nref_min		= 30;		// minimum AI Nref
		dev->cache.ai_nvco_max		= 1000;		// maximum AI Nvco
		dev->cache.ai_nvco_max_opt	= 99;		// maximum optimal AI Nvco
		dev->cache.ai_nvco_min		= 30;		// minimum AI Nvco

		dev->cache.ao_fclk_max		= 250000;	// maximum possible AO channels
		dev->cache.ao_fclk_min		= 2;		// minimum AO clocking rate
		dev->cache.ao_fref			= 30000000;	// AO Fref frequency
		dev->cache.ao_nrate_max		= 0xFFFFFF;	// maximum AO Nrate
		dev->cache.ao_nrate_min		= dev->cache.ao_fref / dev->cache.ao_fclk_max;

		dev->cache.board_type		= GSC_DEV_TYPE_24DSI6LN4AO;
		dev->cache.initialize_ms	= 5000;		// initialization time in ms

		_channels_compute(dev);
		_filter_compute(dev);
		_range_compute(dev);
		_temperature_compute(dev);

		// Initialize additional resources.
		ret	= dev_irq_create(dev);
		if (ret)	break;

		ret	= dev_io_create(dev);
		if (ret)	break;

		dma	= GSC_DMA_SEL_STATIC
			| GSC_DMA_CAP_BMDMA_READ
			| GSC_DMA_CAP_DMDMA_READ;
		ret	= gsc_dma_create(dev, dma, dma);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	dev_device_destroy
*
*	Purpose:
*
*		Do everything needed to release the referenced device.
*
*	Arguments:
*
*		dev		The partial data for the device of interest.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void dev_device_destroy(dev_data_t* dev)
{
	if (dev)
	{
		gsc_dma_destroy(dev);
		dev_io_destroy(dev);
		dev_irq_destroy(dev);
		gsc_bar_destroy(&dev->bar);
		os_sem_destroy(&dev->sem);
		os_spinlock_destroy(&dev->spinlock);
		os_pci_master_clear(&dev->pci);
		os_pci_dev_disable(&dev->pci);
	}
}


