// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI20C500K/utils/util_query.c $
// $Rev: 53471 $
// $Date: 2023-07-26 15:49:04 -0500 (Wed, 26 Jul 2023) $

// 24DSI20C500K: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	dsi20c500k_query
*
*	Purpose:
*
*		Provide a visual wrapper for the DSI20C500K_IOCTL_QUERY service.
*
*	Arguments:
*
*		fd		Use this handle to access the device.
*
*		index	The index of the device to access. Ignore if < 0.
*				This is for display purposes only.
*
*		verbose	Work verbosely?
*
*		set		This is the query option to access.
*
*		get		The results are reported here. This may be NULL.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int dsi20c500k_query(int fd, int index, int verbose, s32 set, s32* get)
{
	char		buf[128];
	int			errs	= 0;
	const char*	ptr;
	s32			query	= set;
	int			ret;
	s32			tmp;

	switch (query)
	{
		default:

			errs++;
			ptr	= buf;
			sprintf(buf, "Query Error (#%ld)", (long) query);
			break;

		case DSI20C500K_QUERY_AUTOCAL_MS:	ptr	= "Autocal Duration";	break;
		case DSI20C500K_QUERY_CHANNEL_MAX:	ptr	= "Max Channels";		break;
		case DSI20C500K_QUERY_CHANNEL_QTY:	ptr	= "Installed Channels";	break;
		case DSI20C500K_QUERY_COUNT:		ptr	= "Query Options";		break;
		case DSI20C500K_QUERY_DEVICE_TYPE:	ptr	= "Device Type";		break;
		case DSI20C500K_QUERY_FIFO_SIZE:	ptr	= "FIFO Size";			break;
		case DSI20C500K_QUERY_FILTER_FREQ:	ptr	= "Filter Freq";		break;
		case DSI20C500K_QUERY_FSAMP_MAX:	ptr	= "Fsamp Max";			break;
		case DSI20C500K_QUERY_FSAMP_MIN:	ptr	= "Fsamp Min";			break;
		case DSI20C500K_QUERY_INIT_MS:		ptr	= "Init. Duration";		break;
		case DSI20C500K_QUERY_FREF_DEFAULT:	ptr	= "Fref Default";		break;
		case DSI20C500K_QUERY_FGEN_MAX:		ptr	= "Fgen Max";			break;
		case DSI20C500K_QUERY_FGEN_MIN:		ptr	= "Fgen Min";			break;
		case DSI20C500K_QUERY_NDIV_MAX:		ptr	= "Ndiv Max";			break;
		case DSI20C500K_QUERY_NDIV_MIN:		ptr	= "Ndiv Min";			break;
		case DSI20C500K_QUERY_NREF_MAX:		ptr	= "Nref Max";			break;
		case DSI20C500K_QUERY_NREF_MIN:		ptr	= "Nref Min";			break;
		case DSI20C500K_QUERY_NVCO_MAX:		ptr	= "Nvco Max";			break;
		case DSI20C500K_QUERY_NVCO_MIN:		ptr	= "Nvco Min";			break;
		case DSI20C500K_QUERY_V_RANGE:		ptr	= "Voltage Range";		break;
	}

	if (verbose)
		gsc_label_index(ptr, index);

	ret		= dsi20c500k_ioctl(fd, DSI20C500K_IOCTL_QUERY, &set);
	errs	+= ret ? 1 : 0;

	switch (query)
	{
		default:

			errs++;
			sprintf(buf, "%ld", (long) set);
			break;

		case DSI20C500K_QUERY_CHANNEL_MAX:
		case DSI20C500K_QUERY_CHANNEL_QTY:
		case DSI20C500K_QUERY_COUNT:
		case DSI20C500K_QUERY_NDIV_MAX:
		case DSI20C500K_QUERY_NDIV_MIN:
		case DSI20C500K_QUERY_NREF_MAX:
		case DSI20C500K_QUERY_NREF_MIN:
		case DSI20C500K_QUERY_NVCO_MAX:
		case DSI20C500K_QUERY_NVCO_MIN:

			sprintf(buf, "%ld", (long) set);
			break;

		case DSI20C500K_QUERY_FGEN_MAX:
		case DSI20C500K_QUERY_FGEN_MIN:
		case DSI20C500K_QUERY_FILTER_FREQ:
		case DSI20C500K_QUERY_FREF_DEFAULT:

			sprintf(buf, "%ld Hz", (long) set);
			break;

		case DSI20C500K_QUERY_AUTOCAL_MS:
		case DSI20C500K_QUERY_INIT_MS:

			sprintf(buf, "%ld ms", (long) set);
			break;

		case DSI20C500K_QUERY_FSAMP_MAX:
		case DSI20C500K_QUERY_FSAMP_MIN:

			sprintf(buf, "%ld S/S", (long) set);
			break;

		case DSI20C500K_QUERY_FIFO_SIZE:

			ptr	= "";
			tmp	= set;

			if ((tmp) && ((tmp % 1024) == 0))
			{
				ptr	= "K";
				tmp	/= 1024;
			}

			sprintf(buf, "%ld%s Words", (long) tmp, ptr);
			break;

		case DSI20C500K_QUERY_DEVICE_TYPE:

			if (set == GSC_DEV_TYPE_24DSI20C500K)
			{
				strcpy(buf, "24DSI20C500K");
			}
			else
			{
				errs++;
				sprintf(buf, "INVALID: %ld", (long) set);
			}

			break;

		case DSI20C500K_QUERY_V_RANGE:

			switch (set)
			{
				default:

					errs++;
					sprintf(buf, "Invalid option: %ld", (long) set);
					break;

				case DSI20C500K_QUERY_V_RANGE_10:	strcpy(buf, "+- 10.0 Volts");	break;
				case DSI20C500K_QUERY_V_RANGE_6:	strcpy(buf, "+- 6.0 Volts");	break;
				case DSI20C500K_QUERY_V_RANGE_5:	strcpy(buf, "+- 5.0 Volts");	break;
				case DSI20C500K_QUERY_V_RANGE_2_5:	strcpy(buf, "+- 2.5 Volts");	break;
			}

			break;
	}

	if (verbose)
	{
		if (errs)
			printf("FAIL <---  (%s)\n", buf);
		else
			printf("%s\n", buf);
	}

	if (get)
		get[0]	= set;

	return(errs);
}


