// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI20C500K/utils/util_fsamp_ai.c $
// $Rev: 53471 $
// $Date: 2023-07-26 15:49:04 -0500 (Wed, 26 Jul 2023) $

// 24DSI20C500K: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	ABS(v)					(((v) < 0) ? (-(v)) : (v))
#define	LIMIT_MAX(v,max)		(((v) > (max)) ? (max) : (v))
#define	LIMIT_MIN(v,min)		(((v) < (min)) ? (min) : (v))
#define	LIMIT_RANGE(v,min,max)	(LIMIT_MIN((LIMIT_MAX((v),(max))),(min)))



// data types *****************************************************************

typedef struct
{
	s32		fsamp_want;
	double	fsamp_got;

	double	delta;
	s32		fgen_max;
	s32		fgen_min;
	s32		fref;
	s32		fsamp_max;
	s32		fsamp_min;
	s32		ndiv;
	s32		ndiv_max;
	s32		ndiv_min;
	s32		nref;
	s32		nref_max;
	s32		nref_min;
	s32		nvco;
	s32		nvco_max;
	s32		nvco_min;
	s32		osr;
} fsamp_t;



// variables ******************************************************************

static const s32 _osr[4]	= { 256, 128, 64, 32 };



//*****************************************************************************
static void _dsi20c500k_fsamp_compute(fsamp_t* data)
{
	double	delta;
	s32		div;
	double	fgen;
	s32		fgen_max	= data->fgen_max;
	s32		fgen_min	= data->fgen_min;
	double	fref		= (double) data->fref;
	double	fsamp_got	= (double) data->fsamp_got;
	double	fsamp_want	= (double) data->fsamp_want;
	s32		ndiv_max	= data->ndiv_max;
	s32		ndiv_min	= data->ndiv_min;
	s32		nref_max	= data->nref_max;
	s32		nref_min	= data->nref_min;
	s32		nvco_max	= data->nvco_max;
	s32		nvco_min	= data->nvco_min;
	s32		ref;
	s32		vco;

	for (vco = nvco_min; vco <= nvco_max; vco++)
	{
		for (ref = nref_min; ref <= nref_max; ref++)
		{
			fgen	= (double) fref * vco / ref;

			if ((fgen < fgen_min) || (fgen > fgen_max))
				continue;

			for (div = ndiv_min; div <= ndiv_max; div++)
			{
				fsamp_got	= fgen / (2 * _osr[data->osr] * div);
				delta		= fsamp_got - fsamp_want;

				if (ABS(delta) < ABS(data->delta))
				{
					data->fsamp_got	= fsamp_got;
					data->delta		= delta;
					data->nvco		= vco;
					data->nref		= ref;
					data->ndiv		= div;
				}
			}
		}
	}
}



//*****************************************************************************
static void _dsi20c500k_fsamp_compute_pll(fsamp_t* data)
{
	if (data->fsamp_want < 78125L)
		data->osr		= DSI20C500K_OSR_256;
	else if (data->fsamp_want < 156250L)
		data->osr		= DSI20C500K_OSR_128;
	else if (data->fsamp_want < 312500L)
		data->osr		= DSI20C500K_OSR_64;
	else
		data->osr		= DSI20C500K_OSR_32;

	_dsi20c500k_fsamp_compute(data);
}



//*****************************************************************************
static int _fsamp_report_rate_gen(
	s32*	sps,
	s32		fref,
	s32		ndiv,
	s32		nref,
	s32		nvco,
	s32		osr)
{
	int		errs	= 0;
	double	fgen;
	s32		fsamp;
	double	rate;
	double	ref;

	ref		= fref;
	fgen	= ref * nvco / nref;
	rate	= fgen / (2 * ndiv * _osr[osr]);
	rate	+= (rate >= 0) ? +0.5 : -0.5;
	fsamp	= (s32) rate;

	gsc_label_long_comma((long) fsamp);
	printf(" S/S  (Fref ");
	gsc_label_long_comma((long) fref);
	printf(", Nvco %ld", (long) nvco);
	printf(", Nref %ld", (long) nref);
	printf(", Ndiv %ld", (long) ndiv);
	printf(", OSR %ld)\n", (long) _osr[osr]);

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
static int _fsamp_report_grp_src(
	int		verbose,
	s32*	sps,
	s32		g_src,
	s32		fref,
	s32		ndiv,
	s32		nref,
	s32		nvco,
	s32		osr)
{
	int	errs	= 0;
	s32	fsamp;

	switch (g_src)
	{
		default:

			fsamp	= 0;
			errs++;

			if (verbose)
				printf("FAIL <---  (Unrecodnized source option: %ld)\n", (long) g_src);

			break;

		case DSI20C500K_CH_GRP_SRC_RATE_GEN:

			errs	+= _fsamp_report_rate_gen(&fsamp, fref, ndiv, nref, nvco, osr);
			break;

		case DSI20C500K_CH_GRP_SRC_EXTERN:
		case DSI20C500K_CH_GRP_SRC_DIR_EXTERN:

			fsamp	= 0;

			if (verbose)
				printf("0 S/S  (External Source)\n");

			break;

		case DSI20C500K_CH_GRP_SRC_DISABLE:

			fsamp	= 0;

			if (verbose)
				printf("0 S/S  (Disabled)\n");

			break;
	}

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



/******************************************************************************
*
*	Function:	dsi20c500k_fsamp_ai_compute
*
*	Purpose:
*
*		Calculate the best values for the specified sample rate.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fref	The is the Fref value.
*
*		fsamp	The desired sample rate.
*
*		nvco	Put the resulting Nvco value here.
*
*		nref	Put the resulting Nref value here.
*
*		ndiv	Put the resulting Ndiv value here.
*
*		osr		Put the oversampling rate selection here.
*
*		rate	Report the sample rate achieved here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int dsi20c500k_fsamp_ai_compute(
	int		fd,
	int		index,
	int		verbose,
	s32		fref,
	s32		fsamp,
	s32*	nvco,
	s32*	nref,
	s32*	ndiv,
	s32*	osr,
	double*	rate)
{
	fsamp_t	data;
	int		errs	= 0;

	if (verbose)
	{
		gsc_label_index("AI Sample Rate", index);
		gsc_label_long_comma(fsamp);
		printf(" S/S");
	}

	memset(&data, 0, sizeof(data));
	errs	+= dsi20c500k_query(fd, -1, 0, DSI20C500K_QUERY_FGEN_MAX, &data.fgen_max);
	errs	+= dsi20c500k_query(fd, -1, 0, DSI20C500K_QUERY_FGEN_MIN, &data.fgen_min);
	errs	+= dsi20c500k_query(fd, -1, 0, DSI20C500K_QUERY_FSAMP_MAX, &data.fsamp_max);
	errs	+= dsi20c500k_query(fd, -1, 0, DSI20C500K_QUERY_FSAMP_MIN, &data.fsamp_min);
	errs	+= dsi20c500k_query(fd, -1, 0, DSI20C500K_QUERY_NDIV_MAX, &data.ndiv_max);
	errs	+= dsi20c500k_query(fd, -1, 0, DSI20C500K_QUERY_NDIV_MIN, &data.ndiv_min);
	errs	+= dsi20c500k_query(fd, -1, 0, DSI20C500K_QUERY_NREF_MAX, &data.nref_max);
	errs	+= dsi20c500k_query(fd, -1, 0, DSI20C500K_QUERY_NREF_MIN, &data.nref_min);
	errs	+= dsi20c500k_query(fd, -1, 0, DSI20C500K_QUERY_NVCO_MAX, &data.nvco_max);
	errs	+= dsi20c500k_query(fd, -1, 0, DSI20C500K_QUERY_NVCO_MIN, &data.nvco_min);

	data.delta		= FLT_MAX;
	data.fref		= fref;
	data.fsamp_want	= LIMIT_RANGE(fsamp, data.fsamp_min, data.fsamp_max);
	data.fsamp_got	= LONG_MAX;
	data.delta		= LONG_MAX;
	data.nvco		= 100;
	data.nref		= 100;
	data.ndiv		= 100;
	data.osr		= DSI20C500K_OSR_256;

	_dsi20c500k_fsamp_compute_pll(&data);

	if (nvco)
		nvco[0]	= data.nvco;

	if (nref)
		nref[0]	= data.nref;

	if (ndiv)
		ndiv[0]	= data.ndiv;

	if (osr)
		osr[0]	= data.osr;

	if (rate)
		rate[0]	= data.fsamp_got;

	if (verbose)
	{
		printf("%s  (", errs ? " FAIL <---" : "");
		printf("Fref ");
		gsc_label_long_comma(data.fref);
		printf(", Nvco %ld", (long) data.nvco);
		printf(", Nref %ld", (long) data.nref);
		printf(", Ndiv %ld", (long) data.ndiv);
		printf(", OSR %ld", (long) _osr[data.osr]);
		printf(", Fsamp %.3f", (double) data.fsamp_got);
		printf(")\n");
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	dsi20c500k_fsamp_ai_report
*
*	Purpose:
*
*		Determine and report the sample rate for the specified channel.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely.
*
*		chan	The index of the channel of interest.
*
*		sps		Store the sample rate here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int dsi20c500k_fsamp_ai_report(int fd, int index, int verbose, int chan, s32* sps)
{
	char	buf[64];
	int		errs	= 0;
	s32		fref;
	s32		fsamp	= 0;
	s32		g_src;
	s32		g0_src;
	s32		g1_src;
	s32		group;
	s32		ndiv;
	s32		nref;
	s32		nvco;
	s32		osr;

	if (verbose)
	{
		sprintf(buf, "Channel %d", chan);
		gsc_label_index(buf, index);
	}

	errs	+= dsi20c500k_channel_group	(fd, -1, 0, chan, &group);;
	errs	+= dsi20c500k_ch_grp_0_src	(fd, -1, 0, -1, &g0_src);
	errs	+= dsi20c500k_ch_grp_1_src	(fd, -1, 0, -1, &g1_src);
	errs	+= dsi20c500k_query			(fd, -1, 0, DSI20C500K_QUERY_FREF_DEFAULT, &fref);
	errs	+= dsi20c500k_ndiv			(fd, -1, 0, -1, &ndiv);
	errs	+= dsi20c500k_nref			(fd, -1, 0, -1, &nref);
	errs	+= dsi20c500k_nvco			(fd, -1, 0, -1, &nvco);
	errs	+= dsi20c500k_osr			(fd, -1, 0, -1, &osr);

	g_src	= group ? g1_src : g0_src;
	errs	+= _fsamp_report_grp_src(verbose, &fsamp, g_src, fref, ndiv, nref, nvco, osr);

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



/******************************************************************************
*
*	Function:	dsi20c500k_fsamp_ai_report_all
*
*	Purpose:
*
*		Determine and report the sample rate for all channels.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely.
*
*		fsamp	The overall sample rate is reported here.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int dsi20c500k_fsamp_ai_report_all(int fd, int index, int verbose, s32* fsamp)
{
	s32	chans;
	int	chan;
	int	errs;
	s32	sps;
	s32	total	= 0;

	if (verbose)
		gsc_label_index("Sample Rates", index);

	errs	= dsi20c500k_query(fd, -1, 0, DSI20C500K_QUERY_CHANNEL_QTY, &chans);

	if (errs)
	{
		if (verbose)
			printf("FAIL <---\n");
	}
	else
	{
		if (verbose)
		{
			printf("\n");
			gsc_label_level_inc();
		}

		for (chan = 0; chan < chans; chan++)
		{
			errs	+= dsi20c500k_fsamp_ai_report(fd, -1, verbose, chan, &sps);
			total	+= sps;
		}

		if (verbose)
		{
			gsc_label("Overall Rate");
			gsc_label_long_comma(total);
			printf(" S/S\n");
			gsc_label_level_dec();
		}
	}

	if (fsamp)
		fsamp[0]	= total;

	return(errs);
}



/******************************************************************************
*
*	Function:	dsi20c500k_fsamp_ai_validate
*
*	Purpose:
*
*		Modify the target sample rate, if needed and report the results.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely.
*
*		fsamp	The desired sample rate is given here. We modifiy this if
*				needed.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int dsi20c500k_fsamp_ai_validate(int fd, int index, int verbose, s32* fsamp)
{
	int	errs	= 0;
	s32	max;
	s32	min;

	errs	+= dsi20c500k_query(fd, -1, 0, DSI20C500K_QUERY_FSAMP_MIN, &min);
	errs	+= dsi20c500k_query(fd, -1, 0, DSI20C500K_QUERY_FSAMP_MAX, &max);

	if (fsamp[0] < min)
	{
		errs++;

		if (verbose)
		{
			printf("FAIL <---  (The minimum is ");
			gsc_label_long_comma((long) min);
			printf(" S/S.)\n");
		}
	}
	else if (fsamp[0] > max)
	{
		errs++;

		if (verbose)
		{
			printf("FAIL <---  (The maximum is ");
			gsc_label_long_comma((long) max);
			printf(" S/S.)\n");
		}
	}

	return(errs);
}


