// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI20C500K/driver/read.c $
// $Rev: 53477 $
// $Date: 2023-07-26 16:02:44 -0500 (Wed, 26 Jul 2023) $

// 24DSI20C500K: Device Driver: source file

#include "main.h"



//*****************************************************************************
static void _dev_io_sw_init(dev_data_t* dev, dev_io_t* io)
{
	io->io_mode			= DSI20C500K_IO_MODE_DEFAULT;
	io->overflow_check	= DSI20C500K_IO_OVERFLOW_DEFAULT;
	io->pio_threshold	= 32;
	io->timeout_s		= DSI20C500K_IO_TIMEOUT_DEFAULT;
	io->underflow_check	= DSI20C500K_IO_UNDERFLOW_DEFAULT;
}



//*****************************************************************************
static void _dev_io_close(dev_data_t* dev, dev_io_t* io)
{
	io->dma_channel	= NULL;
}



//*****************************************************************************
static void _dev_io_open(dev_data_t* dev, dev_io_t* io)
{
	_dev_io_sw_init(dev, io);
	os_mem_dma_open(&io->mem);
}



//*****************************************************************************
static int _dev_io_startup(dev_data_t* dev, dev_io_t* io)
{
	u32		bufcr;
	long	ret		= 0;

	if ((io->overflow_check) || (io->underflow_check))
	{
		bufcr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bufcr_32);

		if ((io->overflow_check) && (bufcr & D25))
			ret	= -EIO;

		if ((io->underflow_check) && (bufcr & D26))
			ret	= -EIO;
	}

	return(ret);
}



//*****************************************************************************
static long _dev_pio_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	u32	bufsr;
	u32	samples;

	bufsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bufsr_32);
	samples	= bufsr;
	bytes	= samples * 4;
	return((long) bytes);
}



//*****************************************************************************
static long _dev_pio_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	long	qty;

	qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	return(qty);
}



//*****************************************************************************
static long _dev_bmdma_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	u32	available;
	u32	bufcr;
	u32	bufsr;
	u32	samples;
	u32	threshold;

	bufsr		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bufsr_32);
	samples		= bufsr;
	available	= samples * 4;

	if (available >= bytes)
	{
		// The desired amount is available.
	}
	else
	{
		bufcr		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bufcr_32);
		threshold	= bufcr & 0xFFFFF;

		if (samples >= threshold)
			bytes	= available;
		else
			bytes	= 0;
	}

	return((long) bytes);
}



//*****************************************************************************
static long _dev_bmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	long			qty;
	long			samples	= bytes / 4;
	gsc_dma_setup_t	setup;

	if (samples <= io->pio_threshold)
	{
		qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	}
	else
	{
		memset(&setup, 0, sizeof(gsc_dma_setup_t));
		setup.alt		= dev;
		setup.dev		= dev;
		setup.io		= io;
		setup.mem		= mem;
		setup.st_end	= st_end;
		setup.bytes		= bytes;
		setup.ability	= GSC_DMA_CAP_BMDMA_READ;

		setup.mode		= GSC_DMA_MODE_BLOCK_DMA
						| GSC_DMA_MODE_SIZE_32_BITS
						| GSC_DMA_MODE_INPUT_ENABLE
						| GSC_DMA_MODE_BURSTING_LOCAL
						| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
						| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
						| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		setup.dpr		= GSC_DMA_DPR_BOARD_TO_HOST
						| GSC_DMA_DPR_END_OF_CHAIN
						| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty	= gsc_dma_perform(&setup);
	}

	return(qty);
}



//*****************************************************************************
static long _dev_dmdma_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	u32	bufcr;
	u32	threshold;

	bufcr		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bufcr_32);
	threshold	= bufcr & 0xFFFFF;
	bytes		= (threshold + 1) * 4;
	return((long) bytes);
}



//*****************************************************************************
static long _dev_dmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	long			qty;
	long			samples	= bytes / 4;
	gsc_dma_setup_t	setup;

	if (samples < io->pio_threshold)
	{
		qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	}
	else
	{
		memset(&setup, 0, sizeof(gsc_dma_setup_t));
		setup.alt		= dev;
		setup.dev		= dev;
		setup.io		= io;
		setup.mem		= mem;
		setup.st_end	= st_end;
		setup.bytes		= bytes;
		setup.ability	= GSC_DMA_CAP_DMDMA_READ;

		setup.mode		= GSC_DMA_MODE_DM_DMA
						| GSC_DMA_MODE_SIZE_32_BITS
						| GSC_DMA_MODE_INPUT_ENABLE
						| GSC_DMA_MODE_BURSTING_LOCAL
						| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
						| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
						| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		setup.dpr		= GSC_DMA_DPR_BOARD_TO_HOST
						| GSC_DMA_DPR_END_OF_CHAIN
						| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty	= gsc_dma_perform(&setup);
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	dev_read_create
*
*	Purpose:
*
*		Perform a one-tine setup of the Read Input streaming I/O structure.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*		io		The I/O structure for this stream.
*
*	Returned:
*
*		None.
*
******************************************************************************/

int dev_read_create(dev_data_t* dev, dev_io_t* io)
{
	int	ret;

	io->bytes_per_sample	= 4;
	io->io_reg_offset		= GSC_REG_OFFSET(DSI20C500K_GSC_IDBR);
	io->io_reg_vaddr		= dev->vaddr.gsc_idbr_32;

	io->dev_io_sw_init		= _dev_io_sw_init;
	io->dev_io_close		= _dev_io_close;
	io->dev_io_open			= _dev_io_open;
	io->dev_io_startup		= _dev_io_startup;
	io->dev_pio_available	= _dev_pio_available;
	io->dev_bmdma_available	= _dev_bmdma_available;
	io->dev_dmdma_available	= _dev_dmdma_available;
	io->dev_pio_xfer		= _dev_pio_xfer;
	io->dev_bmdma_xfer		= _dev_bmdma_xfer;
	io->dev_dmdma_xfer		= _dev_dmdma_xfer;

	io->wait.abort			= DSI20C500K_WAIT_IO_RX_ABORT;
	io->wait.done			= DSI20C500K_WAIT_IO_RX_DONE;
	io->wait.error			= DSI20C500K_WAIT_IO_RX_ERROR;
	io->wait.timeout		= DSI20C500K_WAIT_IO_RX_TIMEOUT;

	ret	= gsc_io_create(dev, io, dev->cache.fifo_size * 4);
	return(ret);
}


