// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI20C500K/driver/main.h $
// $Rev: 53478 $
// $Date: 2023-07-26 16:04:02 -0500 (Wed, 26 Jul 2023) $

// 24DSI20C500K: Device Driver: header file

#ifndef	__MAIN_H__
#define	__MAIN_H__

#include "24dsi20c500k.h"

#define	DEV_BAR_SHOW						0
#define	DEV_METRICS_SHOW					0
#define	DEV_PCI_ID_SHOW						0
#define	DEV_SUPPORTS_IRQ
#define	DEV_SUPPORTS_READ
#define	DEV_SUPPORTS_WAIT
#define	GSC_ALT_DEV_GET(a)					(a)
#define	GSC_ALT_STRUCT_GET(i,d)				(d)
#define	GSC_ALT_STRUCT_T					dev_data_t
#define	GSC_DEVS_PER_BOARD					1

typedef struct _dev_data_t					dev_data_t;

#include "gsc_main.h"



// macros *********************************************************************

#define	DEV_MODEL							"24DSI20C500K"	// Upper case form of the below.
#define	DEV_NAME							"24dsi20c500k"	// MUST AGREE WITH DSI20C500K_BASE_NAME
#define	DEV_NAME_LC							DEV_NAME
#define	DEV_NAME_UC							DEV_MODEL

#define	DEV_VERSION_STR						GSC_STR(DEV_VERSION_MAJ_NUM) "." GSC_STR(DEV_VERSION_MIN_NUM)
#define	DEV_VERSION							DEV_VERSION_STR

#define	DEV_VERSION_MAJ_NUM					2	// FOR DEVICE SPECIFIC CODE ONLY!
#define	DEV_VERSION_MIN_NUM					2	// FOR DEVICE SPECIFIC CODE ONLY!
// 2.2	Updated to support the 5.x series kernel.
//		Updated to support the 6.x series kernel.
//		Standardized various IOCTL system log messages.
//		Standardized processing of the initialization IOCTL service.
//		Standardized processing of the autocalibration IOCTL service.
//		Renamed all Auto_Cal content to Autocal.
//		Renamed all Auto_Cal_Sts content to Autocal_Status.
//		White space cleanup for consistency.
//		Updated per segregation of the PLX specific support.
//		Removed compiler warnings for Fedora 31.
//		Updated initial file content for consistency.
//		Updated for porting to Windows.
//		White space cleanup.
//		Updated per changes to common I/O code.
//		Renamed some macros for consistency and clarity.
//		Modified so open succeeds even if initialization fails.
//		Updated for the 5.x kernel.
//		Exclude os_mem.c's DMA code when reading and writing are unsupported.
//		Bug fix: was ignoring BAR setup return value.
//		Updated per changes to the OS specific PCI services.
//		Made updates for DEV_SUPPORTS_XXX macro changes.
//		Modified code so open succeeds even if initialization fails.
// 2.1	The BAR code has been updated to include the data under a single structure.
//		The register definitions have been updated.
//		I/O services now pass around an os_mem_t structure instead of a buffer pointer.
//		White space cleanup.
//		Changed the arbitrary wait event callback argument to a void* type.
//		Modified to allow multiple apps to simultaneously access a single device.
//		Correct access to BAR0 and BAR1 (is RO, was RW).
//		Added an infinite I/O timeout option.
//		Corrected the IOCTL code for the register write and mod services.
// 2.0	Updated to use the newer common driver sources.
//		Removed GNU notice from non-Linux specific files.
//		Removed Linux specific content from non-Linux specific source files.
//		White space cleanup.
//		Now using a spinlock rather than enabling and disabling interrupts.
//		Updated gsc_irq_open() and gsc_irq_close().
//		Updated gsc_dma_open() and gsc_dma_close().
// 1.3	Reduced #include list in driver interface header.
// 1.2	Removed debug messages while driver loads.
// 1.1	Updated for the 3.x kernel.
//		Moved the macros for D0-D31 to common code.
//		Eliminated the global dev_check_id() routine.
//		Removed debug printk() statements from ioctl.c.
//		BAR0 and BAR2 are now the only BAR regions used.
//		Include all common sources, though not all are used.
//		Corrected system timer count roll over bug in ioctl.c.
// 1.0	Initial release.

// I/O services
#define	DEV_IO_STREAM_QTY					(DEV_IO_RX_STREAM_QTY + DEV_IO_TX_STREAM_QTY)
#define	DEV_IO_RX_STREAM_QTY				1
#define	DEV_IO_TX_STREAM_QTY				0

#define	GSC_READ_PIO_WORK_32_BIT

// WAIT services
#define	DEV_WAIT_GSC_ALL					DSI20C500K_WAIT_GSC_ALL
#define	DEV_WAIT_ALT_ALL					DSI20C500K_WAIT_ALT_ALL
#define	DEV_WAIT_IO_ALL						DSI20C500K_WAIT_IO_ALL



// data types *****************************************************************

struct _dev_io_t
{
	// Initialized by open and IOCTL services (initialize and/or service specific).

	s32					io_mode;			// PIO, DMA, DMDMA
	s32					overflow_check;		// Check overflow when reading?
	s32					pio_threshold;		// Use PIO if samples <= this.
	s32					timeout_s;			// I/O timeout in seconds.
	s32					underflow_check;	// Check underflow when reading?

	// Initialized by I/O service at time of use.

	int					abort;				// Also set via IOCTL service.
	gsc_dma_ch_t*		dma_channel;		// Use this channel for DMA.
	int					non_blocking;		// Is this non-blocking I/O?

	// Initialized by device specific I/O create code.

	int					bytes_per_sample;	// Sample size in bytes.
	u32					io_reg_offset;		// Offset of device's I/O FIFO.
	VADDR_T				io_reg_vaddr;		// Address of device's I/O FIFO.

	void				(*dev_io_sw_init)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	void				(*dev_io_close)			(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	void				(*dev_io_open)			(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	int					(*dev_io_startup)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	long				(*dev_pio_available)	(GSC_ALT_STRUCT_T* alt, dev_io_t* io, size_t count);
	long				(*dev_bmdma_available)	(GSC_ALT_STRUCT_T* alt, dev_io_t* io, size_t count);
	long				(*dev_dmdma_available)	(GSC_ALT_STRUCT_T* alt, dev_io_t* io, size_t count);
	long				(*dev_pio_xfer)			(GSC_ALT_STRUCT_T* alt, dev_io_t* io, const os_mem_t* mem, size_t count, os_time_tick_t st_end);
	long				(*dev_bmdma_xfer)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io, const os_mem_t* mem, size_t count, os_time_tick_t st_end);
	long				(*dev_dmdma_xfer)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io, const os_mem_t* mem, size_t count, os_time_tick_t st_end);

	struct
	{
		u32				abort;		// ...WAIT_IO_XXX_ABORT
		u32				done;		// ...WAIT_IO_XXX_DONE
		u32				error;		// ...WAIT_IO_XXX_ERROR
		u32				timeout;	// ...WAIT_IO_XXX_TIMEOUT
	} wait;

	// Initialize by GSC commone create code.

	os_mem_t			mem;				// I/O buffer.
	os_sem_t			sem;				// Only one Tx or Rx at a time.
};

struct _dev_data_t
{
	// Put an integral data type at beginning of structure.
	int					board_index;	// Index of the device being accessed.
	gsc_dev_type_t		board_type;		// Corresponds to basic device type.
	const char*			model;			// Base model number in upper case.
	int					share;			// Were we opened in shared mode?
	int					users;			// Number of currently active open() requests.

	os_pci_t			pci;			// The kernel PCI device descriptor.
	os_data_t			os;				// OS specific data.
	os_spinlock_t		spinlock;		// Control ISR access.
	os_sem_t			sem;			// Control thread access.

	gsc_bar_t			bar;			// device register mappings
	gsc_dma_t			dma;			// For DMA based I/O.
	gsc_irq_t			irq;			// For interrut support.
	gsc_wait_node_t*	wait_list;

	struct
	{					// This is for streaming I/O
		dev_io_t		rx;				// For read operations.
		dev_io_t*		io_streams[DEV_IO_STREAM_QTY];
	} io;

	struct
	{
		VADDR_T			plx_intcsr_32;	// Interrupt Control/Status Register
		VADDR_T			plx_dmaarb_32;	// DMA Arbitration Register
		VADDR_T			plx_dmathr_32;	// DMA Threshold Register

		VADDR_T			gsc_bctlr_32;	// 0x00 Board Control Register
		VADDR_T			gsc_rcr_32;		// 0x04 Rate Control Register
		VADDR_T			gsc_diopr_32;	// 0x08 Digital I/O Port Register
		VADDR_T			gsc_csar_32;	// 0x0C Clock Source Assignment Register
		VADDR_T			gsc_rdr_32;		// 0x10 Rate Divisors Register
										// 0x14 Reserved
										// 0x18 Reserved
		VADDR_T			gsc_bbsr_32;	// 0x1C Burst Block Size Register
		VADDR_T			gsc_bufcr_32;	// 0x20 Buffer Control Register
		VADDR_T			gsc_bcfgr_32;	// 0x24 Board Configuration Register
		VADDR_T			gsc_bufsr_32;	// 0x28 Buffer Size Register
		VADDR_T			gsc_avr_32;		// 0x2C Auto Cal Values Register
		VADDR_T			gsc_idbr_32;	// 0x30 Input Data Buffer Register
		VADDR_T			gsc_bttr_32;	// 0x3C Burst Trigger Timer Register

	} vaddr;

	struct
	{
		u32				gsc_bcfgr_32;	// Board Configuration Register

		s32				autocal_ms;		// Maximum ms for autocalibrate

		s32				channels_max;	// Maximum channels supported by model.
		s32				channel_qty;	// The number of channels on the device.

		s32				fifo_size;		// Size of FIFO - not the fill level.
		s32				filter_freq;	// 0, 150000, -1, ...
		s32				fref_default;	// 40,000,000
		s32				fsamp_max;		// The maximum Fsamp rate per channel.
		s32				fsamp_min;		// The minimum Fsamp rate per channel.

		s32				initialize_ms;	// Maximum ms for initialize

		s32				ndiv_max;		// Ndiv maximum
		s32				ndiv_min;		// Ndiv minimum
		s32				nref_max;		// Nref maximum
		s32				nref_min;		// Nref minimum
		s32				nvco_max;		// Nvco maximum
		s32				nvco_min;		// Nvco minimum

		s32				fgen_max;		// Rate Generator maximum output rate.
		s32				fgen_min;		// Rate Generator minimum output rate.

		s32				voltage_range;	// 10/6/5/2.5

	} cache;
};



// prototypes *****************************************************************

void		dev_io_close(dev_data_t* dev);
int			dev_io_create(dev_data_t* dev);
void		dev_io_destroy(dev_data_t* dev);
int			dev_io_open(dev_data_t* dev);
dev_io_t*	dev_io_read_select(dev_data_t* dev, size_t bytes);
int			dev_irq_create(dev_data_t* dev);
void		dev_irq_destroy(dev_data_t* dev);

int			dev_read_create(dev_data_t* dev, dev_io_t* io);

int			initialize_ioctl(dev_data_t* dev, void* arg);



#endif
