// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI20C500K/driver/ioctl.c $
// $Rev: 53477 $
// $Date: 2023-07-26 16:02:44 -0500 (Wed, 26 Jul 2023) $

// 24DSI20C500K: Device Driver: source file

#include "main.h"



// macros *********************************************************************

#define	AUTOCAL_START			D7		// BCTLR
#define	AUTOCAL_PASS			D12		// BCTLR
#define	INIT_START				D15		// BCTLR

#define	IRQ_INIT_DONE			0x0000	// BCTLR
#define	IRQ_AUTOCAL_DONE		0x0100	// BCTLR
#define	IRQ_MASK				0x0700	// BCTLR
#define	IRQ_ACTIVE				0x0800	// BCTLR



// prototypes *****************************************************************

static	int		_wait_for_ready(dev_data_t* dev, long ms_limit);



//*****************************************************************************
static void _bctlr_mod(dev_data_t* dev, u32 value, u32 mask)
{
	if (mask != IRQ_ACTIVE)
	{
		value	|= IRQ_ACTIVE;
		mask	|= IRQ_ACTIVE;
	}

	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, value, mask);
}



//*****************************************************************************
static int _query(dev_data_t* dev, void* arg)
{
	int		ret	= 0;
	s32*	ptr	= (s32*) arg;

	switch (ptr[0])
	{
		default:	ptr[0]	= DSI20C500K_IOCTL_QUERY_ERROR;
					ret		= -EINVAL;
					break;

		case DSI20C500K_QUERY_AUTOCAL_MS:	ptr[0]	= dev->cache.autocal_ms;	break;
		case DSI20C500K_QUERY_CHANNEL_MAX:	ptr[0]	= dev->cache.channels_max;	break;
		case DSI20C500K_QUERY_CHANNEL_QTY:	ptr[0]	= dev->cache.channel_qty;	break;
		case DSI20C500K_QUERY_COUNT:		ptr[0]	= DSI20C500K_QUERY_LAST;	break;
		case DSI20C500K_QUERY_DEVICE_TYPE:	ptr[0]	= dev->board_type;			break;
		case DSI20C500K_QUERY_FGEN_MAX:		ptr[0]	= dev->cache.fgen_max;		break;
		case DSI20C500K_QUERY_FGEN_MIN:		ptr[0]	= dev->cache.fgen_min;		break;
		case DSI20C500K_QUERY_FIFO_SIZE:	ptr[0]	= dev->cache.fifo_size;		break;
		case DSI20C500K_QUERY_FILTER_FREQ:	ptr[0]	= dev->cache.filter_freq;	break;
		case DSI20C500K_QUERY_FREF_DEFAULT:	ptr[0]	= dev->cache.fref_default;	break;
		case DSI20C500K_QUERY_FSAMP_MAX:	ptr[0]	= dev->cache.fsamp_max;		break;
		case DSI20C500K_QUERY_FSAMP_MIN:	ptr[0]	= dev->cache.fsamp_min;		break;
		case DSI20C500K_QUERY_INIT_MS:		ptr[0]	= dev->cache.initialize_ms;	break;
		case DSI20C500K_QUERY_NDIV_MAX:		ptr[0]	= dev->cache.ndiv_max;		break;
		case DSI20C500K_QUERY_NDIV_MIN:		ptr[0]	= dev->cache.ndiv_min;		break;
		case DSI20C500K_QUERY_NREF_MAX:		ptr[0]	= dev->cache.nref_max;		break;
		case DSI20C500K_QUERY_NREF_MIN:		ptr[0]	= dev->cache.nref_min;		break;
		case DSI20C500K_QUERY_NVCO_MAX:		ptr[0]	= dev->cache.nvco_max;		break;
		case DSI20C500K_QUERY_NVCO_MIN:		ptr[0]	= dev->cache.nvco_min;		break;
		case DSI20C500K_QUERY_V_RANGE:		ptr[0]	= dev->cache.voltage_range;	break;
	}

	return(ret);
}



//*****************************************************************************
static int _init_start(dev_data_t* dev, void* arg)
{
	// Initiate initialization.
	os_reg_mem_tx_u32(NULL, dev->vaddr.gsc_bctlr_32, INIT_START);
	return(0);
}



//*****************************************************************************
int initialize_ioctl(dev_data_t* dev, void* arg)
{
	int			i;
	int			mask;
	long		ms			= dev->cache.initialize_ms + 5000;
	long		ms_total	= ms;
	u32			reg;
	int			ret			= 0;
	os_sem_t	sem;
	int			tmp;
	VADDR_T		va			= dev->vaddr.gsc_bctlr_32;
	int			value;
	gsc_wait_t	wt;

	if ((dev->irq.opened) && (gsc_global.driver_unloading == 0))
	{
		ms_total	*= 2;

		// Safely select the Initialize Done interrupt.
		mask	= IRQ_MASK | IRQ_ACTIVE;
		value	= IRQ_INIT_DONE;
		os_reg_mem_mx_u32(dev, va, value, mask);

		// Wait for the local interrupt.
		os_sem_create(&sem);	// dummy, required for wait operations.
		memset(&wt, 0, sizeof(wt));
		wt.flags		= GSC_WAIT_FLAG_INTERNAL;
		wt.gsc			= DSI20C500K_WAIT_GSC_INIT_DONE;
		wt.timeout_ms	= ms;
		ret				= gsc_wait_event(dev, &wt, _init_start, NULL, &sem);
		os_sem_destroy(&sem);

		if (wt.flags & GSC_WAIT_FLAG_TIMEOUT)
		{
			ret	= ret ? ret : -ETIMEDOUT;
			printf(	"%s: INITILIZE DONE IRQ TIMED OUT AFTER %ld ms.\n",
					dev->model,
					(long) ms);
		}
	}
	else
	{
		_init_start(dev, NULL);
	}

	// Manually wait for completion as the IRQ wait may have ended early.
	tmp	= gsc_poll_u32(dev, ms, va, INIT_START, 0);

	if (tmp)
	{
		ret	= ret ? ret : tmp;
		printf(	"%s: INITIALIZATION DID NOT COMPLETE WITHIN %ld ms.\n",
				dev->model,
				ms_total);
	}

	// Manually check for completion as the IRQ wait may have ended early.
	reg	= os_reg_mem_rx_u32(dev, va);

	if (reg & INIT_START)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: INITILIZE STILL ACTIVE AFTER %ld ms.\n",
				dev->model,
				ms_total);
	}

	// Wait for settling.
	_wait_for_ready(dev, 20);

	// Initialize the software settings.

	for (i = 0; i < DEV_IO_STREAM_QTY; i++)
	{
		if (dev->io.io_streams[i])
		{
			if (dev->io.io_streams[i]->dev_io_sw_init)
			{
				(dev->io.io_streams[i]->dev_io_sw_init)(dev, dev->io.io_streams[i]);
			}
		}
	}

	return(ret);
}



//*****************************************************************************
static int _autocal_start(dev_data_t* dev, void* arg)
{
	// Initiate autocalibration.
	_bctlr_mod(dev, AUTOCAL_START, AUTOCAL_START);
	return(0);
}



//*****************************************************************************
static int _autocal(dev_data_t* dev, void* arg)
{
	u32			mask;
	long		ms		= dev->cache.autocal_ms + 5000;
	u32			reg;
	int			ret;
	os_sem_t	sem;
	int			tmp;
	VADDR_T		va		= dev->vaddr.gsc_bctlr_32;
	u32			value;
	gsc_wait_t	wt;

	// Safely select the Autocal Done interrupt.
	mask	= IRQ_MASK | IRQ_ACTIVE;
	value	= IRQ_AUTOCAL_DONE;
	_bctlr_mod(dev, value, mask);

	// Wait for the local interrupt.
	os_sem_create(&sem);	// dummy, required for wait operations.
	memset(&wt, 0, sizeof(wt));
	wt.flags		= GSC_WAIT_FLAG_INTERNAL;
	wt.gsc			= DSI20C500K_WAIT_GSC_AUTOCAL_DONE;
	wt.timeout_ms	= ms;
	ret				= gsc_wait_event(dev, &wt, _autocal_start, NULL, &sem);
	os_sem_destroy(&sem);

	if (wt.flags & GSC_WAIT_FLAG_TIMEOUT)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATE IRQ TIMED OUT AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually wait for completion in case something terminates our wait early.
	tmp	= gsc_poll_u32(dev, ms, va, AUTOCAL_START, 0);

	if (tmp)
	{
		ms	*= 2;
		ret	= ret ? ret : tmp;
		printf(	"%s: AUTOCALIBRATION DID NOT COMPLETE WITHIN %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually check for completion as the IRQ wait may have ended early.
	reg	= os_reg_mem_rx_u32(dev, va);

	if (reg & AUTOCAL_START)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATION STILL ACTIVE AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Final results.
	reg	= os_reg_mem_rx_u32(dev, va);

	if ((reg & AUTOCAL_PASS) == 0)
	{
		ret	= ret ? ret : -EIO;
		printf(	"%s: AUTOCALIBRATION FAILED (%ld ms).\n",
				dev->model,
				ms);
	}

	if (ret == 0)
	{
		// Wait for settling.
		os_time_sleep_ms(100);
	}

	return(ret);
}



//*****************************************************************************
static int _autocal_status(dev_data_t* dev, s32* arg)
{
	u32	reg;

	reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);

	if (reg & AUTOCAL_START)
		arg[0]	= DSI20C500K_AUTOCAL_STATUS_ACTIVE;
	else if (reg & AUTOCAL_PASS)
		arg[0]	= DSI20C500K_AUTOCAL_STATUS_PASS;
	else
		arg[0]	= DSI20C500K_AUTOCAL_STATUS_FAIL;

	return(0);
}



//*****************************************************************************
static int _ai_buf_clear(dev_data_t* dev, void* arg)
{
	#define	CLEAR	D21
	#define	OVER	D25
	#define	UNDER	D26

	int		i;
	u32		reg;
	int		ret	= 0;
	VADDR_T	va	= dev->vaddr.gsc_bufcr_32;

	// Clear the buffer.
	os_reg_mem_mx_u32(dev, va, CLEAR, CLEAR);

	// Wait for the bit to clear.

	for (i = 0;; i++)
	{
		reg	= os_reg_mem_rx_u32(dev, va);

		if ((reg & CLEAR) == 0)
			break;

		if (i >= 250)
		{
			ret	= -EINVAL;
			printf(	"%s: The analog input buffer took too long to clear.\n",
					DEV_NAME);
			break;
		}
	}

	// Clear the Overflow status bit.
	os_reg_mem_mx_u32(dev, va, 0, OVER);

	// Clear the Underflow status bit.
	os_reg_mem_mx_u32(dev, va, 0, UNDER);

	return(ret);
}



//*****************************************************************************
static int _ai_buf_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_AI_BUF_ENABLE_NO,
		DSI20C500K_AI_BUF_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bufcr_32, 20, 20);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_fill_lvl(dev_data_t* dev, s32* arg)
{
	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_range_reg(dev, arg, 0, 0x80000, dev->vaddr.gsc_bufsr_32, 19, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_overflow(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_AI_BUF_OVERFLOW_NO,
		DSI20C500K_AI_BUF_OVERFLOW_YES,
		DSI20C500K_AI_BUF_OVERFLOW_CLEAR,
		DSI20C500K_AI_BUF_OVERFLOW_TEST,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bufcr_32, 25, 25);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_thresh(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0x3FFFF, dev->vaddr.gsc_bufcr_32, 19, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_thr_sts(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_AI_BUF_THR_STS_IDLE,
		DSI20C500K_AI_BUF_THR_STS_ACTIVE,
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 14, 14);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_underflow(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_AI_BUF_UNDERFLOW_NO,
		DSI20C500K_AI_BUF_UNDERFLOW_YES,
		DSI20C500K_AI_BUF_UNDERFLOW_CLEAR,
		DSI20C500K_AI_BUF_UNDERFLOW_TEST,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bufcr_32, 26, 26);
	return(ret);
}



//*****************************************************************************
static int _ai_channel_tag(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_AI_CHANNEL_TAG_ENABLE,
		DSI20C500K_AI_CHANNEL_TAG_DISABLE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bufcr_32, 24, 24);
	return(ret);
}



//*****************************************************************************
static int _ai_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_AI_MODE_DIFF,
		DSI20C500K_AI_MODE_ZERO,
		DSI20C500K_AI_MODE_VREF,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 1, 0);
	return(ret);
}



//*****************************************************************************
static int _burst(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_BURST_DISABLE,
		DSI20C500K_BURST_ENABLE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 21, 21);
	return(ret);
}



//*****************************************************************************
static int _burst_rate_div(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFFFF, dev->vaddr.gsc_bttr_32, 23, 0);
	return(ret);
}



//*****************************************************************************
static int _burst_size(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFFFF, dev->vaddr.gsc_bbsr_32, 23, 0);
	return(ret);
}



//*****************************************************************************
static int _burst_timer(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_BURST_TIMER_DISABLE,
		DSI20C500K_BURST_TIMER_ENABLE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 16, 16);
	return(ret);
}



//*****************************************************************************
static int _burst_trigger(dev_data_t* dev, void* arg)
{
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, D22, D22);

	// Wait for settling.
	_wait_for_ready(dev, 20);
	return(0);
}



//*****************************************************************************
static int _ch_grp_0_src(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_CH_GRP_SRC_RATE_GEN,
		DSI20C500K_CH_GRP_SRC_EXTERN,
		DSI20C500K_CH_GRP_SRC_DIR_EXTERN,
		DSI20C500K_CH_GRP_SRC_DISABLE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_csar_32, 3, 0);

	// Wait for settling.
	_wait_for_ready(dev, 20);
	return(ret);
}



//*****************************************************************************
static int _ch_grp_1_src(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_CH_GRP_SRC_RATE_GEN,
		DSI20C500K_CH_GRP_SRC_EXTERN,
		DSI20C500K_CH_GRP_SRC_DIR_EXTERN,
		DSI20C500K_CH_GRP_SRC_DISABLE,
		-1	// terminate list
	};

	s32	g0	= -1;
	s32	g1	= -1;
	int	ret;
	int	ret0;
	int	ret1;

	if (arg[0] == -1)
	{
		ret0	= gsc_s32_list_reg(dev, &g0, options, dev->vaddr.gsc_csar_32, 3, 0);
		ret1	= gsc_s32_list_reg(dev, &g1, options, dev->vaddr.gsc_csar_32, 7, 4);

		arg[0]	= (g1 == DSI20C500K_CH_GRP_SRC_DISABLE) ? g1 : g0;
		ret		= ret0 ? ret0 : ret1;
	}
	else
	{
		ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_csar_32, 7, 4);
	}

	// Wait for settling.
	_wait_for_ready(dev, 20);
	return(ret);
}



//*****************************************************************************
static int _channels_ready(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_CHANNELS_READY_NO,
		DSI20C500K_CHANNELS_READY_YES,
		-1	// terminate list
	};

	int	ret	= 0;

	if (arg[0] == DSI20C500K_CHANNELS_READY_WAIT)
	{
		gsc_poll_u32(dev, 1000, dev->vaddr.gsc_bctlr_32, D13, D13);
		arg[0]	= DSI20C500K_CHANNELS_READY_TEST;
	}

	if (arg[0] == DSI20C500K_CHANNELS_READY_TEST)
		ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 13, 13);
	else
		ret	= -EINVAL;

	return(ret);
}



//*****************************************************************************
static int _control_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_CONTROL_MODE_TARGET,
		DSI20C500K_CONTROL_MODE_INITIATOR,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 5, 5);
	return(ret);
}



//*****************************************************************************
static int _data_format(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_DATA_FORMAT_2S_COMP,
		DSI20C500K_DATA_FORMAT_OFF_BIN,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 4, 4);
	return(ret);
}



//*****************************************************************************
static int _data_width(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_DATA_WIDTH_16,
		DSI20C500K_DATA_WIDTH_18,
		DSI20C500K_DATA_WIDTH_20,
		DSI20C500K_DATA_WIDTH_24,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bufcr_32, 23, 22);
	return(ret);
}



//*****************************************************************************
static int _dio_dir_out(dev_data_t* dev, s32* arg)
{
	u32	mask	= D15 | D7;
	u32	reg;
	int	ret;
	u32	v;

	if (arg[0] == -1)
	{
		ret	= 0;
		reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_diopr_32);

		switch (reg & (D15 | D7))
		{
			default:
			case 0:			arg[0]	= DSI20C500K_DIO_DIR_OUT_NONE;	break;
			case D7:		arg[0]	= DSI20C500K_DIO_DIR_OUT_0_3;	break;
			case D15:		arg[0]	= DSI20C500K_DIO_DIR_OUT_4_7;	break;
			case D15 | D7:	arg[0]	= DSI20C500K_DIO_DIR_OUT_0_7;	break;
		}
	}
	else
	{
		switch (arg[0])
		{
			default:							v	= 999;		break;
			case DSI20C500K_DIO_DIR_OUT_NONE:	v	= 0;		break;
			case DSI20C500K_DIO_DIR_OUT_0_3:	v	= D7;		break;
			case DSI20C500K_DIO_DIR_OUT_4_7:	v	= D15;		break;
			case DSI20C500K_DIO_DIR_OUT_0_7:	v	= D15 | D7;	break;
		}

		if (v == 999)
		{
			ret		= -EINVAL;
		}
		else
		{
			ret		= 0;
			os_reg_mem_mx_u32(dev, dev->vaddr.gsc_diopr_32, v, mask);
		}
	}

	return(ret);
}



//*****************************************************************************
static int _dio_read(dev_data_t* dev, s32* arg)
{
	u32	reg;

	reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_diopr_32);
	reg		= ((reg >> 4) & 0xF0) | (reg & 0xF);
	arg[0]	= reg;
	return(0);
}



//*****************************************************************************
static int _dio_write(dev_data_t* dev, s32* arg)
{
	int	ret;
	u32	v;

	if (arg[0] == -1)
	{
		ret	= _dio_read(dev, arg);
	}
	else if ((arg[0] > 0xFF) || (arg[0] < 0))
	{
		ret	= -EINVAL;
	}
	else
	{
		ret	= 0;
		v	= ((arg[0] & 0xF0) << 4) | (arg[0] & 0xF);
		os_reg_mem_mx_u32(dev, dev->vaddr.gsc_diopr_32, v, 0xF0F);
	}

	return(ret);
}



//*****************************************************************************
static int _ext_clk_src(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_EXT_CLK_SRC_GRP_0,
		DSI20C500K_EXT_CLK_SRC_GEN,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 18, 18);
	return(ret);
}



//*****************************************************************************
static int _irq_sel(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_IRQ_INIT_DONE,
		DSI20C500K_IRQ_AUTOCAL_DONE,
		DSI20C500K_IRQ_CHAN_READY,
		DSI20C500K_IRQ_AI_BUF_THR_L2H,
		DSI20C500K_IRQ_AI_BUF_THR_H2L,
		DSI20C500K_IRQ_AI_BURST_DONE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 10, 8);
	return(ret);
}



//*****************************************************************************
static int _led(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_LED_OFF,
		DSI20C500K_LED_ON,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 19, 19);
	return(ret);
}



//*****************************************************************************
static int _ndiv(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 1, 300, dev->vaddr.gsc_rdr_32, 8, 0);

	// Wait for settling.
	_wait_for_ready(dev, 20);
	return(ret);
}



//*****************************************************************************
static int _nref(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 25, 300, dev->vaddr.gsc_rcr_32, 23, 12);

	// Wait for settling.
	_wait_for_ready(dev, 20);
	return(ret);
}



//*****************************************************************************
static int _nvco(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 25, 300, dev->vaddr.gsc_rcr_32, 11, 0);

	// Wait for settling.
	_wait_for_ready(dev, 20);
	return(ret);
}



//*****************************************************************************
static int _osr(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_OSR_32,
		DSI20C500K_OSR_64,
		DSI20C500K_OSR_128,
		DSI20C500K_OSR_256,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 3, 2);
	return(ret);
}



//*****************************************************************************
static int _rx_io_abort(dev_data_t* dev, s32* arg)
{
	arg[0]	= gsc_read_abort_active_xfer(dev, &dev->io.rx);
	return(0);
}



//*****************************************************************************
static int _rx_io_mode(dev_data_t* dev, s32* arg)
{
	static const s32	list[]	=
	{
		GSC_IO_MODE_PIO,
		GSC_IO_MODE_BMDMA,
		GSC_IO_MODE_DMDMA,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.rx.io_mode);
	return(ret);
}



//*****************************************************************************
static int _rx_io_overflow(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_IO_OVERFLOW_IGNORE,
		DSI20C500K_IO_OVERFLOW_CHECK,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, options, &dev->io.rx.overflow_check);
	return(ret);
}



//*****************************************************************************
static int _rx_io_timeout(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_var(
			arg,
			DSI20C500K_IO_TIMEOUT_MIN,
			DSI20C500K_IO_TIMEOUT_INFINITE,
			&dev->io.rx.timeout_s);
	return(ret);
}



//*****************************************************************************
static int _rx_io_underflow(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_IO_UNDERFLOW_IGNORE,
		DSI20C500K_IO_UNDERFLOW_CHECK,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, options, &dev->io.rx.underflow_check);
	return(ret);
}



//*****************************************************************************
static int _sw_sync(dev_data_t* dev, void* arg)
{
	_bctlr_mod(dev, D6, D6);
	return(0);
}



//*****************************************************************************
static int _sw_sync_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_SW_SYNC_MODE_SYNC,
		DSI20C500K_SW_SYNC_MODE_CLR_BUF,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 17, 17);
	return(ret);
}



//*****************************************************************************
static int _xcvr_type(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI20C500K_XCVR_TYPE_LVDS,
		DSI20C500K_XCVR_TYPE_TTL,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 20, 20);
	return(ret);
}



// variables ******************************************************************

const gsc_ioctl_t	dev_ioctl_list[]	=
{
	{ DSI20C500K_IOCTL_REG_READ,			(void*) gsc_reg_read_ioctl		},
	{ DSI20C500K_IOCTL_REG_WRITE,			(void*) gsc_reg_write_ioctl		},
	{ DSI20C500K_IOCTL_REG_MOD,				(void*) gsc_reg_mod_ioctl		},
	{ DSI20C500K_IOCTL_QUERY,				(void*) _query					},
	{ DSI20C500K_IOCTL_INITIALIZE,			(void*) initialize_ioctl		},
	{ DSI20C500K_IOCTL_AUTOCAL,				(void*) _autocal				},
	{ DSI20C500K_IOCTL_AUTOCAL_STATUS,		(void*) _autocal_status			},
	{ DSI20C500K_IOCTL_AI_BUF_CLEAR,		(void*) _ai_buf_clear			},
	{ DSI20C500K_IOCTL_AI_BUF_ENABLE,		(void*) _ai_buf_enable			},
	{ DSI20C500K_IOCTL_AI_BUF_FILL_LVL,		(void*) _ai_buf_fill_lvl		},
	{ DSI20C500K_IOCTL_AI_BUF_OVERFLOW,		(void*) _ai_buf_overflow		},
	{ DSI20C500K_IOCTL_AI_BUF_THR_STS,		(void*) _ai_buf_thr_sts			},
	{ DSI20C500K_IOCTL_AI_BUF_THRESH,		(void*) _ai_buf_thresh			},
	{ DSI20C500K_IOCTL_AI_BUF_UNDERFLOW,	(void*) _ai_buf_underflow		},
	{ DSI20C500K_IOCTL_AI_CHANNEL_TAG,		(void*) _ai_channel_tag			},
	{ DSI20C500K_IOCTL_AI_MODE,				(void*) _ai_mode				},
	{ DSI20C500K_IOCTL_BURST,				(void*) _burst					},
	{ DSI20C500K_IOCTL_BURST_RATE_DIV,		(void*) _burst_rate_div			},
	{ DSI20C500K_IOCTL_BURST_SIZE,			(void*) _burst_size				},
	{ DSI20C500K_IOCTL_BURST_TIMER,			(void*) _burst_timer			},
	{ DSI20C500K_IOCTL_BURST_TRIGGER,		(void*) _burst_trigger			},
	{ DSI20C500K_IOCTL_CH_GRP_0_SRC,		(void*) _ch_grp_0_src			},
	{ DSI20C500K_IOCTL_CH_GRP_1_SRC,		(void*) _ch_grp_1_src			},
	{ DSI20C500K_IOCTL_CHANNELS_READY,		(void*) _channels_ready			},
	{ DSI20C500K_IOCTL_CONTROL_MODE,		(void*) _control_mode			},
	{ DSI20C500K_IOCTL_DATA_FORMAT,			(void*) _data_format			},
	{ DSI20C500K_IOCTL_DATA_WIDTH,			(void*) _data_width				},
	{ DSI20C500K_IOCTL_DIO_DIR_OUT,			(void*) _dio_dir_out			},
	{ DSI20C500K_IOCTL_DIO_READ,			(void*) _dio_read				},
	{ DSI20C500K_IOCTL_DIO_WRITE,			(void*) _dio_write				},
	{ DSI20C500K_IOCTL_EXT_CLK_SRC,			(void*) _ext_clk_src			},
	{ DSI20C500K_IOCTL_IRQ_SEL,				(void*) _irq_sel				},
	{ DSI20C500K_IOCTL_LED,					(void*) _led					},
	{ DSI20C500K_IOCTL_NDIV,				(void*) _ndiv					},
	{ DSI20C500K_IOCTL_NREF,				(void*) _nref					},
	{ DSI20C500K_IOCTL_NVCO,				(void*) _nvco					},
	{ DSI20C500K_IOCTL_OSR,					(void*) _osr					},
	{ DSI20C500K_IOCTL_RX_IO_ABORT,			(void*) _rx_io_abort			},
	{ DSI20C500K_IOCTL_RX_IO_MODE,			(void*) _rx_io_mode				},
	{ DSI20C500K_IOCTL_RX_IO_OVERFLOW,		(void*) _rx_io_overflow			},
	{ DSI20C500K_IOCTL_RX_IO_TIMEOUT,		(void*) _rx_io_timeout			},
	{ DSI20C500K_IOCTL_RX_IO_UNDERFLOW,		(void*) _rx_io_underflow		},
	{ DSI20C500K_IOCTL_SW_SYNC,				(void*) _sw_sync				},
	{ DSI20C500K_IOCTL_SW_SYNC_MODE,		(void*) _sw_sync_mode			},
	{ DSI20C500K_IOCTL_WAIT_EVENT,			(void*) gsc_wait_event_ioctl	},
	{ DSI20C500K_IOCTL_WAIT_CANCEL,			(void*) gsc_wait_cancel_ioctl	},
	{ DSI20C500K_IOCTL_WAIT_STATUS,			(void*) gsc_wait_status_ioctl	},
	{ DSI20C500K_IOCTL_XCVR_TYPE,			(void*) _xcvr_type				},
	{ -1, NULL }
};



//*****************************************************************************
static int _wait_for_ready(dev_data_t* dev, long ms_limit)
{
	int	ret;

	ret	= gsc_poll_u32(dev, ms_limit, dev->vaddr.gsc_bctlr_32, D13, D13);
	return(ret);
}


