// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI12WRCIEPE/24DSI12WRCIEPE_linux_1.x.x.x_gsc_dn/utils/util_reg.c $
// $Rev: 52535 $
// $Date: 2023-03-02 14:55:56 -0600 (Thu, 02 Mar 2023) $

// 24DSI12WRCIEPE: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	_GSC_REG(a)			"GSC " #a, DSI12WI_GSC_##a
#define	NAME_WIDTH			24
#define	SIZEOF_ARRAY(a)		(sizeof((a)) / sizeof((a)[0]))

#define	FIELD_SHOW(beg,end,eol,lst,dsc)		\
			gsc_reg_field_show(width+10,NAME_WIDTH,value,(beg),(end),(eol),(lst),(dsc))



//*****************************************************************************
static int _decode_bctlr(int fd, int supported, u32 value, int width)
{
	static const char*	_aim[]	=
	{
		"Differential",
		"Reserved",
		"Zero Test",
		"Vref Test"
	};

	static const char*	_range[]	=
	{
		"+-0.01 Volts",
		"+-0.1 Volts",
		"+-1 Volts",
		"+-10 Volts"
	};

	static const char*	_irq[]	=
	{
		"Initialization Complete",
		"Auto-Calibration Complete",
		"Channels Ready",
		"Threshold Flag Low-to-High",
		"Threshold Flag High-to-Low",
		"Burst Complete",
		"Reserved",
		"Reserved"
	};

	static const char*	_adc[]		= { "High Resolution",	"High Speed"		};
	static const char*	_control[]	= { "Target",			"Initiator"			};
	static const char*	_disable[]	= { "Disabled",			"Enabled"			};
	static const char*	_extern[]	= { "Group 0 Source",	"Rate generator"	};
	static const char*	_fail[]		= { "Fail",				"Pass"				};
	static const char*	_format[]	= { "Twos Compliment",	"Offset Binary"		};
	static const char*	_idle[]		= { "Idle",				"Active"			};
	static const char*	_not_rdy[]	= { "Not Ready",		"Ready"				};
	static const char*	_sync[]		= { "Sync",				"Clear Buffer"		};

	FIELD_SHOW( 1,  0, 1, _aim,		"Analog Input Mode"		);
	FIELD_SHOW( 3,  2, 1, _range,	"Analog Input Range"	);
	FIELD_SHOW( 4,  4, 1, _format,	"Data Format"			);
	FIELD_SHOW( 5,  5, 1, _control,	"Clock Control Mode"	);
	FIELD_SHOW( 6,  6, 1, _idle,	"SW Sync"				);
	FIELD_SHOW( 7,  7, 1, _idle,	"Auto-Calibration"		);
	FIELD_SHOW(10,  8, 1, _irq,		"IRQ Select"			);
	FIELD_SHOW(11, 11, 1, _idle,	"IRQ Request"			);
	FIELD_SHOW(12, 12, 1, _fail,	"Auto-Cal Status"		);
	FIELD_SHOW(13, 13, 1, _not_rdy,	"Channels Ready"		);
	FIELD_SHOW(14, 14, 1, _idle,	"Buffer Threshold Flag"	);
	FIELD_SHOW(15, 15, 1, _idle,	"Initialize"			);
	FIELD_SHOW(16, 16, 1, _disable,	"Burst Trigger Timer"	);
	FIELD_SHOW(17, 17, 1, _sync,	"Sync Mode"				);
	FIELD_SHOW(18, 18, 1, _extern,	"External Clock Source"	);
	FIELD_SHOW(19, 19, 1, _adc,		"ADC Mode"				);
	FIELD_SHOW(20, 20, 1, _control,	"SYNC Control Mode"		);
	FIELD_SHOW(21, 21, 1, _disable,	"Bursting"				);
	FIELD_SHOW(22, 22, 1, _idle,	"Burst Trigger"			);
	FIELD_SHOW(23, 23, 1, _disable,	"Ext SYNC Input"		);
	FIELD_SHOW(31, 24, 1, NULL,		"Reserved"				);
	return(0);
}



//*****************************************************************************
static int _decode_rcr(int fd, int supported, u32 value, int width)
{
	FIELD_SHOW(11,  0, 1, NULL,	"Nvco"		);
	FIELD_SHOW(23, 12, 1, NULL,	"Nref"		);
	FIELD_SHOW(31, 24, 1, NULL,	"Reserved"	);
	return(0);
}



//*****************************************************************************
static int _decode_ipcr(int fd, int supported, u32 value, int width)
{
	static const char*	_ac[]		= { "No, AC Coupling",	"Yes"		};
	static const char*	_disable[]	= { "Disabled",			"Enabled"	};

	FIELD_SHOW( 0, 0, 1, _ac,		"DC Coupling"			);
	FIELD_SHOW( 1, 1, 1, _disable,	"Excitation Current"	);
	FIELD_SHOW( 2, 2, 1, _disable,	"Input Compensation"	);
	FIELD_SHOW( 3, 3, 1, _disable,	"Aux Current"			);
	FIELD_SHOW(31, 4, 1, NULL,		"Reserved"				);
	return(0);
}



//*****************************************************************************
static int _decode_csar(int fd, int supported, u32 value, int width)
{
	static const char*	_src_0[]	=
	{
		"Rate Generator",
		"Reserved",
		"Reserved",
		"Reserved",
		"External Sample Clock",
		"Direct External Sample Clock",
		"INVALID",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved"
	};

	static const char*	_src_1[]	=
	{
		"Rate Generator",
		"Reserved",
		"Reserved",
		"Reserved",
		"External Sample Clock",
		"Direct External Sample Clock",
		"Disable",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved"
	};

	FIELD_SHOW( 3,  0, 1, _src_0,	"Group 0 Clock Source"	);
	FIELD_SHOW( 7,  4, 1, _src_1,	"Group 1 Clock Source"	);
	FIELD_SHOW(31,  8, 1, NULL,		"Reserved"				);
	return(0);
}



//*****************************************************************************
static int _decode_rdr(int fd, int supported, u32 value, int width)
{
	FIELD_SHOW( 8,  0, 1, NULL,	"Ndiv"		);
	FIELD_SHOW(31,  9, 1, NULL,	"Reserved"	);
	return(0);
}



//*****************************************************************************
static int _decode_bbsr(int fd, int supported, u32 value, int width)
{
	FIELD_SHOW(23,  0, 1, NULL,	"Size"		);
	FIELD_SHOW(31, 24, 1, NULL,	"Reserved"	);
	return(0);
}



//*****************************************************************************
static int _decode_bufcr(int fd, int supported, u32 value, int width)
{
	static const char*	_width[]	=
	{
		"16-bits",
		"18-bits",
		"20-bits",
		"24-bits"
	};

	static const char*	_disable[]	= { "Disabled",	"Enabled"	};
	static const char*	_enable[]	= { "Enabled",	"Disabled"	};
	static const char*	_idle[]		= { "Idle",		"Active"	};
	static const char*	_no[]		= { "No",		"Yes"		};

	FIELD_SHOW(18,  0, 1, NULL,		"Threshold"			);
	FIELD_SHOW(19, 19, 1, _disable,	"Buffer Enable"		);
	FIELD_SHOW(20, 20, 1, _idle,	"Clear Buffer"		);
	FIELD_SHOW(22, 21, 1, _width,	"Data Width"		);
	FIELD_SHOW(23, 23, 1, _enable,	"Channel Tag"		);
	FIELD_SHOW(24, 24, 1, _no,		"Buffer Overflow"	);
	FIELD_SHOW(25, 25, 1, _no,		"Buffer Underflow"	);
	FIELD_SHOW(31, 26, 1, NULL,		"Reserved"			);
	return(0);
}



//*****************************************************************************
static int _decode_bcgfr(int fd, int supported, u32 value, int width)
{
	static const char*	_cannels[]	=
	{
		"12 Channels",
		"8 Channels",
		"4 Channels",
		"Reserved"
	};

	static const char*	_filter[]	=
	{
		"150KHz",
		"No Filter",
		"Reserved",
		"Reserved"
	};

	static const char*	_cutoff[]	=
	{
		"0.5 Hz",
		"2 Hz",
		"Reserved",
		"Reserved"
	};

	static const char*	_excite[]	=
	{
		"4ma",
		"Reserved",
		"Reserved",
		"Reserved"
	};

	u32	field;

	FIELD_SHOW(11,  0, 0, NULL,		"Firmware Revision"	);
	field	= GSC_FIELD_DECODE(value, 11, 0);
	printf("%03lX\n", (long) field);

	FIELD_SHOW(15, 12, 1, NULL,		"Reserved"				);
	FIELD_SHOW(17, 16, 1, _cannels,	"Channels"				);
	FIELD_SHOW(19, 18, 1, _filter,	"Filter Frequency"		);
	FIELD_SHOW(21, 20, 1, _cutoff,	"AC Cutoff Freq."		);
	FIELD_SHOW(23, 22, 1, _excite,	"Excitation Current"	);
	FIELD_SHOW(31, 24, 1, NULL,		"Reserved"				);
	return(0);
}



//*****************************************************************************
static int _decode_bufsr(int fd, int supported, u32 value, int width)
{
	u32	field;

	FIELD_SHOW(18,  0, 0, NULL,	"Buffer Fill Level"	);
	field	= GSC_FIELD_DECODE(value, 18, 0);
	printf("%ld sample%s\n", (long) field, (field == 1) ? "" : "s");

	FIELD_SHOW(31, 19, 1, NULL,	"Reserved"			);
	return(0);
}



//*****************************************************************************
static int _decode_asiocr(int fd, int supported, u32 value, int width)
{
	static const char*	_mode[]	=
	{
		"Inactive",
		"Active Input",
		"Active Output",
		"Active Output"
	};

	FIELD_SHOW( 1,  0, 1, _mode,	"Aux Clock Control Mode"	);
	FIELD_SHOW( 3,  2, 1, _mode,	"Aux SYNC Control Mode"		);
	FIELD_SHOW(31,  4, 1, NULL,		"Reserved"					);
	return(0);
}



//*****************************************************************************
static int _decode_bttr(int fd, int supported, u32 value, int width)
{
	FIELD_SHOW(23,  0, 1, NULL,	"Divisor"	);
	FIELD_SHOW(31, 24, 1, NULL,	"Reserved"	);
	return(0);
}



// variables ******************************************************************

static gsc_reg_def_t	_gsc[]	=
{
	// name	reg			err	value	ask_support	decode			desc
	{ _GSC_REG(BCTLR),	0,	0,		0,			_decode_bctlr,	"Board Control Register"			},
	{ _GSC_REG(RCR),	0,	0,		0,			_decode_rcr,	"Rate Control Register"				},
	{ _GSC_REG(IPCR),	0,	0,		0,			_decode_ipcr,	"Input Port Config Register"		},
	{ _GSC_REG(CSAR),	0,	0,		0,			_decode_csar,	"Clock Source Assignment Register"	},
	{ _GSC_REG(RDR),	0,	0,		0,			_decode_rdr,	"Rate Divisors Register"			},
	{ _GSC_REG(BBSR),	0,	0,		0,			_decode_bbsr,	"Burst Block Size Register"			},
	{ _GSC_REG(BUFCR),	0,	0,		0,			_decode_bufcr,	"Buffer Control Register"			},
	{ _GSC_REG(BCFGR),	0,	0,		0,			_decode_bcgfr,	"Board Configuration Register"		},
	{ _GSC_REG(BUFSR),	0,	0,		0,			_decode_bufsr,	"Buffer Size Register"				},
	{ _GSC_REG(AVR),	0,	0,		0,			NULL,			"Auto-Cal Values Register"			},
	{ _GSC_REG(IDBR),	0,	0,		0,			NULL,			"Input Data Buffer Register"		},
	{ _GSC_REG(ASIOCR),	0,	0,		0,			_decode_asiocr,	"Aux Sync I/O Control Register"		},
	{ _GSC_REG(BTTR),	0,	0,		0,			_decode_bttr,	"Burst Trigger Timer Register"		},
	{ NULL, 0,			0,	0,		0,			NULL,			NULL								}
};



//*****************************************************************************
static const gsc_reg_def_t* _find_reg(u32 reg, const gsc_reg_def_t* list)
{
	const gsc_reg_def_t*	def	= NULL;
	int						i;

	for (i = 0; list[i].name; i++)
	{
		if (reg == list[i].reg)
		{
			def	= &list[i];
			break;
		}
	}

	return(def);
}



/******************************************************************************
*
*	Function:	dsi12wi_reg_get_def_id
*
*	Purpose:
*
*		Retrieve the register definition structure given the register id.
*
*	Arguments:
*
*		reg		The id of the register to access.
*
*	Returned:
*
*		NULL	The register id wasn't found.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* dsi12wi_reg_get_def_id(u32 reg)
{
	const gsc_reg_def_t*	def;

	def	= _find_reg(reg, _gsc);
	return(def);
}



/******************************************************************************
*
*	Function:	dsi12wi_reg_get_def_index
*
*	Purpose:
*
*		Retrieve the register definition structure based on an index.
*
*	Arguments:
*
*		index	The index of the register to access.
*
*	Returned:
*
*		NULL	The index doesn't correspond to a known register.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* dsi12wi_reg_get_def_index(int index)
{
	const gsc_reg_def_t*	def;

	if (index < 0)
		def	= NULL;
	else if (index >= (SIZEOF_ARRAY(_gsc) - 1))
		def	= NULL;
	else
		def	= &_gsc[index];

	return(def);
}



/******************************************************************************
*
*	Function:	dsi12wi_reg_get_desc
*
*	Purpose:
*
*		Retrieve the description of the specified register.
*
*	Arguments:
*
*		reg		The register whose description is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* dsi12wi_reg_get_desc(u32 reg)
{
	const gsc_reg_def_t*	def;
	const char*				desc;

	def	= _find_reg(reg, _gsc);

	if (def)
		desc	= def->desc;
	else
		desc	= "UNKNOWN";

	return(desc);
}



/******************************************************************************
*
*	Function:	dsi12wi_reg_get_name
*
*	Purpose:
*
*		Retrieve the name of the specified register.
*
*	Arguments:
*
*		reg		The register whose name is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* dsi12wi_reg_get_name(u32 reg)
{
	const gsc_reg_def_t*	def;
	const char*				name;

	def	= _find_reg(reg, _gsc);

	if (def)
		name	= def->name;
	else
		name	= "UNKNOWN";

	return(name);
}



/******************************************************************************
*
*	Function:	dsi12wi_reg_list
*
*	Purpose:
*
*		List the GSC registers and their values.
*
*	Arguments:
*
*		fd		The handle to access the device.
*
*		detail	List the register details?
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int dsi12wi_reg_list(int fd, int detail)
{
	int	errs;

	errs	= gsc_reg_list(fd, _gsc, detail, dsi12wi_reg_read);
	return(errs);
}


