// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI12WRCIEPE/24DSI12WRCIEPE_linux_1.x.x.x_gsc_dn/utils/util_query.c $
// $Rev: 52535 $
// $Date: 2023-03-02 14:55:56 -0600 (Thu, 02 Mar 2023) $

// 24DSI12WRCIEPE: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	dsi12wi_query
*
*	Purpose:
*
*		Provide a visual wrapper for the DSI12WI_IOCTL_QUERY service.
*
*	Arguments:
*
*		fd		Use this handle to access the device.
*
*		index	The index of the device to access. Ignore if < 0.
*				This is for display purposes only.
*
*		verbose	Work verbosely?
*
*		set		This is the query option to access.
*
*		get		The results are reported here. This may be NULL.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int dsi12wi_query(int fd, int index, int verbose, s32 set, s32* get)
{
	char		buf[128];
	int			errs	= 0;
	const char*	ptr;
	s32			query	= set;
	int			ret;

	switch (query)
	{
		default:

			errs++;
			ptr	= buf;
			sprintf(buf, "Query Error (#%ld)", (long) query);
			break;

		case DSI12WI_QUERY_AUTO_CAL_MS:		ptr	= "Auto-Cal Duration";	break;
		case DSI12WI_QUERY_CHANNEL_MAX:		ptr	= "Max Channels";		break;
		case DSI12WI_QUERY_CHANNEL_QTY:		ptr	= "Installed Channels";	break;
		case DSI12WI_QUERY_COUNT:			ptr	= "Query Options";		break;
		case DSI12WI_QUERY_CUTOFF_FREQ:		ptr	= "Cutoff Freq";		break;
		case DSI12WI_QUERY_DEVICE_TYPE:		ptr	= "Device Type";		break;
		case DSI12WI_QUERY_EXCITATION_MA:	ptr	= "Excitation Current";	break;
		case DSI12WI_QUERY_FIFO_SIZE:		ptr	= "FIFO Size";			break;
		case DSI12WI_QUERY_FILTER_FREQ:		ptr	= "Filter Freq";		break;
		case DSI12WI_QUERY_FGEN_MAX:		ptr	= "Fgen Max";			break;
		case DSI12WI_QUERY_FGEN_MIN:		ptr	= "Fgen Min";			break;
		case DSI12WI_QUERY_FREF_DEFAULT:	ptr	= "Fref Default";		break;
		case DSI12WI_QUERY_FSAMP_MAX:		ptr	= "Fsamp Max";			break;
		case DSI12WI_QUERY_FSAMP_MIN:		ptr	= "Fsamp Min";			break;
		case DSI12WI_QUERY_INIT_MS:			ptr	= "Init. Duration";		break;
		case DSI12WI_QUERY_NDIV_MAX:		ptr	= "Ndiv Max";			break;
		case DSI12WI_QUERY_NDIV_MIN:		ptr	= "Ndiv Min";			break;
		case DSI12WI_QUERY_NREF_MAX:		ptr	= "Nref Max";			break;
		case DSI12WI_QUERY_NREF_MIN:		ptr	= "Nref Min";			break;
		case DSI12WI_QUERY_NVCO_MAX:		ptr	= "Nvco Max";			break;
		case DSI12WI_QUERY_NVCO_MIN:		ptr	= "Nvco Min";			break;
	}

	if (verbose)
		gsc_label_index(ptr, index);

	ret		= dsi12wi_ioctl(fd, DSI12WI_IOCTL_QUERY, &set);
	errs	+= ret ? 1 : 0;

	switch (query)
	{
		default:

			errs++;
			sprintf(buf, "%ld", (long) set);
			break;

		case DSI12WI_QUERY_COUNT:
		case DSI12WI_QUERY_NDIV_MAX:
		case DSI12WI_QUERY_NDIV_MIN:
		case DSI12WI_QUERY_NREF_MAX:
		case DSI12WI_QUERY_NREF_MIN:
		case DSI12WI_QUERY_NVCO_MAX:
		case DSI12WI_QUERY_NVCO_MIN:
		case DSI12WI_QUERY_CHANNEL_MAX:
		case DSI12WI_QUERY_CHANNEL_QTY:

			gsc_label_long_comma_buf(set, buf);
			break;

		case DSI12WI_QUERY_FGEN_MAX:
		case DSI12WI_QUERY_FGEN_MIN:
		case DSI12WI_QUERY_FILTER_FREQ:
		case DSI12WI_QUERY_FREF_DEFAULT:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " Hz");
			break;

		case DSI12WI_QUERY_EXCITATION_MA:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " ma");
			break;

		case DSI12WI_QUERY_AUTO_CAL_MS:
		case DSI12WI_QUERY_INIT_MS:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " ms");
			break;

		case DSI12WI_QUERY_FSAMP_MAX:
		case DSI12WI_QUERY_FSAMP_MIN:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " S/S");
			break;

		case DSI12WI_QUERY_FIFO_SIZE:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " Words");

			if ((set) && ((set % 1024) == 0))
			{
				strcat(buf, " (");
				gsc_label_long_comma_buf(set / 1024, buf + strlen(buf));
				strcat(buf, "K)");
			}

			break;

		case DSI12WI_QUERY_DEVICE_TYPE:

			if (set == GSC_DEV_TYPE_24DSI12WRCIEPE)
			{
				strcpy(buf, "24DSI12WRCIEPE");
			}
			else
			{
				errs++;
				sprintf(buf, "INVALID: %ld", (long) set);
			}

			break;

		case DSI12WI_QUERY_CUTOFF_FREQ:

			switch (set)
			{
				default:

					errs++;
					sprintf(buf, "Invalid option: %ld", (long) set);
					break;

				case DSI12WI_QUERY_CUTOFF_FREQ_05_HZ:	strcpy(buf, "0.5 Hz");	break;
				case DSI12WI_QUERY_CUTOFF_FREQ_2_HZ:	strcpy(buf, "2 Hz");	break;
				case DSI12WI_QUERY_CUTOFF_FREQ_ERROR:	strcpy(buf, "ERROR");	break;
			}

			break;
	}

	if (verbose)
	{
		if (errs)
			printf("FAIL <---  (%s)\n", buf);
		else
			printf("%s\n", buf);
	}

	if (get)
		get[0]	= set;

	return(errs);
}


