// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI12WRCIEPE/24DSI12WRCIEPE_linux_1.x.x.x_gsc_dn/utils/util_fsamp_ai.c $
// $Rev: 52535 $
// $Date: 2023-03-02 14:55:56 -0600 (Thu, 02 Mar 2023) $

// 24DSI12WRCIEPE: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	ABS(v)					(((v) < 0) ? (-(v)) : (v))
#define	LIMIT_MAX(v,max)		(((v) > (max)) ? (max) : (v))
#define	LIMIT_MIN(v,min)		(((v) < (min)) ? (min) : (v))
#define	LIMIT_RANGE(v,min,max)	(LIMIT_MIN((LIMIT_MAX((v),(max))),(min)))



// data types *****************************************************************

typedef struct
{
	s32		fsamp_want;	// provided as input
	double	fsamp_got;	// computed

	double	delta;		// computed
	long	divisor;	// per mode
	s32		fgen_max;	// fixed
	s32		fgen_min;	// fixed
	s32		fref;		// fixed
	s32		fsamp_max;	// fixed, but changed per mode
	s32		fsamp_min;	// fixed, but changed per mode
	s32		mode;		// per mode
	s32		ndiv;		// computed
	s32		ndiv_max;	// fixed
	s32		ndiv_min;	// fixed
	s32		nref;		// computed
	s32		nref_max;	// fixed
	s32		nref_min;	// fixed
	s32		nvco;		// computed
	s32		nvco_max;	// fixed
	s32		nvco_min;	// fixed
} fsamp_t;



//*****************************************************************************
static void _fsamp_compute_mode(fsamp_t* data)
{
	double	delta;
	double	fgen;
	double	fsamp;
	s32		ndiv;
	s32		ndiv_start;
	s32		ndiv_stop;
	s32		nref;
	s32		nvco;

	for (nvco = data->nvco_min; nvco <= data->nvco_max; nvco++)
	{
		for (nref = data->nref_min; nref <= data->nref_max; nref++)
		{
			fgen	= (double) data->fref * nvco / nref;

			if ((fgen < data->fgen_min) || (fgen > data->fgen_max))
				continue;

			ndiv		= (s32) (fgen / data->divisor / data->fsamp_want);
			ndiv_start	= ndiv - 1;
			ndiv_stop	= ndiv + 1;

			for (ndiv = ndiv_start; ndiv <= ndiv_stop; ndiv++)
			{
				if ((ndiv < data->ndiv_min) || (ndiv > data->ndiv_max))
					continue;

				fsamp	= fgen / ndiv / data->divisor;

				if ((fsamp < data->fsamp_min) || (fsamp > data->fsamp_max))
					continue;

				delta	= fsamp - data->fsamp_want;

				if (ABS(delta) < ABS(data->delta))
				{
					data->fsamp_got	= fsamp;
					data->delta		= delta;
					data->nvco		= nvco;
					data->nref		= nref;
					data->ndiv		= ndiv;
				}
			}
		}
	}
}



//*****************************************************************************
static void _fsamp_compute_hi_res(fsamp_t* data)
{
	data->mode		= DSI12WI_ADC_MODE_HI_RES;
	data->divisor	= 1024;
	// fsamp_min: unchanged
	data->fsamp_max	/= 2;
	_fsamp_compute_mode(data);
}



//*****************************************************************************
static void _fsamp_compute_hi_speed(fsamp_t* data)
{
	data->mode		= DSI12WI_ADC_MODE_HI_SPEED;
	data->divisor	= 512;
	data->fsamp_min	*= 2;
	// fsamp_max: unchanged
	_fsamp_compute_mode(data);
}



//*****************************************************************************
static void _fsamp_compute(fsamp_t* data)
{
	fsamp_t	hi_res		= data[0];
	fsamp_t	hi_speed	= data[0];

	_fsamp_compute_hi_res(&hi_res);
	_fsamp_compute_hi_speed(&hi_speed);

	if (data->fsamp_want >= hi_speed.fsamp_min)
		data[0]	= hi_speed;
	else
		data[0]	= hi_res;
}



/******************************************************************************
*
*	Function:	dsi12wi_fsamp_ai_compute
*
*	Purpose:
*
*		Calculate the best values for the specified sample rate.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fsamp	The desired sample rate.
*
*		mode	Put the computed mode here. (Hi Speed vd Hi Res)
*
*		nvco	Put the resulting Nvco value here.
*
*		nref	Put the resulting Nref value here.
*
*		ndiv	Put the resulting Ndiv value here.
*
*		rate	Report the sample rate achieved here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int dsi12wi_fsamp_ai_compute(
	int		fd,
	int		index,
	int		verbose,
	s32		fsamp,
	s32*	mode,
	s32*	nvco,
	s32*	nref,
	s32*	ndiv,
	double*	rate)
{
	fsamp_t	data;
	int		errs	= 0;

	if (verbose)
	{
		gsc_label_index("AI Sample Rate", index);
		gsc_label_long_comma(fsamp);
		printf(" S/S");
	}

	memset(&data, 0, sizeof(data));
	errs	+= dsi12wi_query(fd, -1, 0, DSI12WI_QUERY_FGEN_MAX, &data.fgen_max);
	errs	+= dsi12wi_query(fd, -1, 0, DSI12WI_QUERY_FGEN_MIN, &data.fgen_min);
	errs	+= dsi12wi_query(fd, -1, 0, DSI12WI_QUERY_FREF_DEFAULT, &data.fref);
	errs	+= dsi12wi_query(fd, -1, 0, DSI12WI_QUERY_FSAMP_MAX, &data.fsamp_max);
	errs	+= dsi12wi_query(fd, -1, 0, DSI12WI_QUERY_FSAMP_MIN, &data.fsamp_min);
	errs	+= dsi12wi_query(fd, -1, 0, DSI12WI_QUERY_NDIV_MAX, &data.ndiv_max);
	errs	+= dsi12wi_query(fd, -1, 0, DSI12WI_QUERY_NDIV_MIN, &data.ndiv_min);
	errs	+= dsi12wi_query(fd, -1, 0, DSI12WI_QUERY_NREF_MAX, &data.nref_max);
	errs	+= dsi12wi_query(fd, -1, 0, DSI12WI_QUERY_NREF_MIN, &data.nref_min);
	errs	+= dsi12wi_query(fd, -1, 0, DSI12WI_QUERY_NVCO_MAX, &data.nvco_max);
	errs	+= dsi12wi_query(fd, -1, 0, DSI12WI_QUERY_NVCO_MIN, &data.nvco_min);

	data.delta		= FLT_MAX;
	data.fsamp_want	= LIMIT_RANGE(fsamp, data.fsamp_min, data.fsamp_max);

	_fsamp_compute(&data);

	mode[0]		= data.mode;
	nvco[0]		= data.nvco;
	nref[0]		= data.nref;
	ndiv[0]		= data.ndiv;

	if (rate)
		rate[0]	= data.fsamp_got;

	if (verbose)
	{
		printf("%s  (", errs ? " FAIL <---" : "");
		printf((data.mode == DSI12WI_ADC_MODE_HI_RES) ? "Hi Res" : "Hi Speed");
		printf(", Fref ");
		gsc_label_long_comma(data.fref);
		printf(", Nvco %ld", (long) data.nvco);
		printf(", Nref %ld", (long) data.nref);
		printf(", Ndiv %ld", (long) data.ndiv);
		printf(", Fsamp %.3f", (double) data.fsamp_got);
		printf(")\n");
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	dsi12wi_fsamp_ai_report
*
*	Purpose:
*
*		Determine and report the sample rate for the specified channel.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		chan	The index of the channel of interest.
*
*		sps		Store the sample rate here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int dsi12wi_fsamp_ai_report(int fd, int index, int verbose, int chan, s32* sps)
{
	char	buf[64];
	s32		buffer		= 0;
	s32		burst		= 0;
	s32		clock		= 0;
	int		div;
	int		errs		= 0;
	s32		fref		= 0;
	s32		fsamp		= 0;
	s32		g0_src		= 0;
	s32		g1_src		= 0;
	int		group;
	s32		mode		= 0;
	s32		ndiv		= 1;
	s32		nref		= 1;
	s32		nvco		= 0;
	s32		qty			= 12;
	s32		ready		= 0;

	if (verbose)
	{
		sprintf(buf, "Channel %d", chan);
		gsc_label_index(buf, index);
	}

	errs	+= dsi12wi_ai_buf_enable		(fd, -1, 0, -1, &buffer);
	errs	+= dsi12wi_burst_enable			(fd, -1, 0, -1, &burst);
	errs	+= dsi12wi_clock_control_mode	(fd, -1, 0, -1, &clock);
	errs	+= dsi12wi_channels_ready		(fd, -1, 0, -1, &ready);
	errs	+= dsi12wi_ch_grp_0_src			(fd, -1, 0, -1, &g0_src);
	errs	+= dsi12wi_ch_grp_1_src			(fd, -1, 0, -1, &g1_src);
	errs	+= dsi12wi_adc_mode				(fd, -1, 0, -1, &mode);
	errs	+= dsi12wi_ndiv					(fd, -1, 0, -1, &ndiv);
	errs	+= dsi12wi_nref					(fd, -1, 0, -1, &nref);
	errs	+= dsi12wi_nvco					(fd, -1, 0, -1, &nvco);
	errs	+= dsi12wi_query				(fd, -1, 0, DSI12WI_QUERY_FREF_DEFAULT, &fref);
	errs	+= dsi12wi_query				(fd, -1, 0, DSI12WI_QUERY_CHANNEL_QTY, &qty);

	group	= (chan >= (qty / 2)) ? 1 : 0;

	for (;;)	// A convenience loop.
	{
		if (errs)
		{
			if (verbose)
				printf("0 S/S  FAIL <---\n");

			break;
		}

		if (buffer == DSI12WI_AI_BUF_ENABLE_NO)
		{
			if (verbose)
				printf("0 S/S  (Input buffer is disabled.)\n");

			break;
		}

		if ((group == 1) && (g1_src == DSI12WI_CH_GRP_SRC_DISABLE))
		{
			if (verbose)
				printf("0 S/S  (Channel group is disabled.)\n");

			break;
		}

		if (g0_src != DSI12WI_CH_GRP_SRC_RATE_GEN)
		{
			if (verbose)
				printf("0 S/S  (Using an external clock.)\n");

			break;
		}

		if (burst == DSI12WI_BURST_ENABLE_YES)
		{
			if (verbose)
				printf("0 S/S  (Bursting is enabled.)\n");

			break;
		}

		if (clock == DSI12WI_CONTROL_MODE_TARGET)
		{
			if (verbose)
				printf("0 S/S  (Using target mode clock.)\n");

			break;
		}

		if (ready == DSI12WI_CHANNELS_READY_NO)
		{
			if (verbose)
				printf("0 S/S  (Device is not ready.)\n");

			break;
		}

		div		= (mode == DSI12WI_ADC_MODE_HI_RES) ? 1024 : 512;
		fsamp	= (s32) ((double) fref * nvco / ndiv / div / nref + 0.5);

		if (verbose)
		{
			gsc_label_long_comma((long) fsamp);
			printf(" S/S  (Ref ");
			gsc_label_long_comma(fref);
			printf(", ");
			printf((mode == DSI12WI_ADC_MODE_HI_RES) ? "Hi Res" : "Hi Speed");
			printf(", Nvco %ld", (long) nvco);
			printf(", Nref %ld", (long) nref);
			printf(", Ndiv %ld)\n", (long) ndiv);
		}

		break;
	}

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



/******************************************************************************
*
*	Function:	dsi12wi_fsamp_ai_report_all
*
*	Purpose:
*
*		Determine and report the sample rate for all channels.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fsamp	Report the overall sample rate here.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int dsi12wi_fsamp_ai_report_all(int fd, int index, int verbose, s32* fsamp)
{
	s32	chans;
	int	chan;
	int	errs;
	s32	sps;
	s32	total	= 0;

	if (verbose)
		gsc_label_index("Sample Rates", index);

	errs	= dsi12wi_query(fd, -1, 0, DSI12WI_QUERY_CHANNEL_QTY, &chans);

	if (errs)
	{
		if (verbose)
			printf("FAIL <---\n");
	}
	else
	{
		if (verbose)
		{
			printf("\n");
			gsc_label_level_inc();
		}

		for (chan = 0; chan < chans; chan++)
		{
			errs	+= dsi12wi_fsamp_ai_report(fd, -1, verbose, chan, &sps);
			total	+= sps;
		}

		if (verbose)
		{
			gsc_label("Overall Rate");
			gsc_label_long_comma(total);
			printf(" S/S\n");
			gsc_label_level_dec();
		}
	}

	if (fsamp)
		fsamp[0]	= total;

	return(errs);
}


