// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI12WRCIEPE/24DSI12WRCIEPE_linux_1.x.x.x_gsc_dn/samples/savedata/main.c $
// $Rev: 52540 $
// $Date: 2023-03-02 14:58:02 -0600 (Thu, 02 Mar 2023) $

// 24DSI12WRCIEPE: Sample Application: source file

#include "main.h"



//*****************************************************************************
static int _parse_args(int argc, char** argv, args_t* args)
{
	const gsc_arg_item_t	list[]	=
	{
		//	type				var						arg			values								desc
		{	GSC_ARG_S32_FLAG,	&args->width,			"-16",		{ DSI12WI_DATA_WIDTH_16 },			"Set the data width to 16-bits."						},
		{	GSC_ARG_S32_FLAG,	&args->width,			"-18",		{ DSI12WI_DATA_WIDTH_18 },			"Set the data width to 18-bits."						},
		{	GSC_ARG_S32_FLAG,	&args->width,			"-20",		{ DSI12WI_DATA_WIDTH_20 },			"Set the data width to 20-bits."						},
		{	GSC_ARG_S32_FLAG,	&args->width,			"-24",		{ DSI12WI_DATA_WIDTH_24 },			"Set the data width to 24-bits."						},
		{	GSC_ARG_S32_FLAG,	&args->io_mode,			"-bmdma",	{ GSC_IO_MODE_BMDMA },				"Transfer data using Block Mode DMA."					},
		{	GSC_ARG_S32_FLAG,	&args->continuous,		"-c",		{ -1 },								"Continue testing until an error occurs."				},
		{	GSC_ARG_S32_FLAG,	&args->continuous,		"-C",		{ +1 },								"Continue testing even if errors occur."				},
		{	GSC_ARG_S32_FLAG,	&args->decimal,			"-dec",		{ 1 },								"Save data in decimal format."							},
		{	GSC_ARG_S32_FLAG,	&args->io_mode,			"-dmdma",	{ GSC_IO_MODE_DMDMA },				"Transfer data using Demand Mode DMA."					},
		{	GSC_ARG_S32_FLAG,	&args->force_save,		"-f",		{ 1 },								"Force data to be saved even if errors occur."			},
		{	GSC_ARG_S32_FLAG,	&args->decimal,			"-hex",		{ 0 },								"Save data in hexidecimal format (default)."			},
		{	GSC_ARG_S32_MIN,	&args->minute_limit,	"-m",		{ 1 },								"Continue testing for up to # minutes."					},
		{	GSC_ARG_S32_MIN,	&args->test_limit,		"-n",		{ 1 },								"Continue testing for up to # iterations."				},
		{	GSC_ARG_S32_FLAG,	&args->data_format,		"-off",		{ DSI12WI_DATA_FORMAT_OFF_BIN },	"Format the data as Offset Binary."						},
		{	GSC_ARG_S32_FLAG,	&args->io_mode,			"-pio",		{ GSC_IO_MODE_PIO },				"Transfer data using PIO (the default)."				},
		{	GSC_ARG_S32_RANGE,	&args->repeat,			"-r",		{ 1, 10000 },						"Repeat data retreaval # times before saving the data."	},
		{	GSC_ARG_S32_RANGE,	&args->fsamp,			"-s",		{ 200, 105000 },					"Sample data at this Fsamp rate."						},
		{	GSC_ARG_S32_FLAG,	&args->tag_strip,		"-t",		{ 1 },								"Strip the channel tag from the saved data."			},
		{	GSC_ARG_S32_FLAG,	&args->data_format,		"-two",		{ DSI12WI_DATA_FORMAT_2S_COMP },	"Format the data as Twos Compliment."					},
		{	GSC_ARG_S32_FLAG,	&args->v_range,			"-v10m",	{ DSI12WI_RANGE_10MV },				"Select the +- 10 m volt range."						},
		{	GSC_ARG_S32_FLAG,	&args->v_range,			"-v100m",	{ DSI12WI_RANGE_100MV },			"Select the +- 100 m volt range."						},
		{	GSC_ARG_S32_FLAG,	&args->v_range,			"-v1",		{ DSI12WI_RANGE_1V },				"Select the +- 1 volt range."							},
		{	GSC_ARG_S32_FLAG,	&args->v_range,			"-v10",		{ DSI12WI_RANGE_10V },				"Select the +- 10 volt range."							},
		{	GSC_ARG_DEV_INDEX,	&args->index,			"index",	{ 1 },								"The zero based index of the device to access."			}
	};

	const gsc_arg_set_t	set	=
	{
		/* name	*/	"savedata",
		/* desc	*/	"Capture and save data to disk.",
		/* qty	*/	SIZEOF_ARRAY(list),
		/* list	*/	list
	};

	int	errs;

	memset(args, 0, sizeof(args_t));
	args->data_format	= -1;
	args->fsamp			= 10000;
	args->io_mode		= -1;
	args->repeat		= 1;
	args->v_range		= -1;
	args->width			= -1;

	errs	= gsc_args_parse(argc, argv, &set);

	return(errs);
}



//*****************************************************************************
static void _show_access_index(const args_t* args)
{
	gsc_label("Accessing Device");
	printf("%d\n", args->index);
}



//*****************************************************************************
static void _show_time_stamp(void)
{
	const char*	psz;
	struct tm*	stm;
	time_t		tt;

	time(&tt);
	stm	= localtime(&tt);
	psz	= (char*) asctime(stm);
	gsc_label("Time Stamp");
	printf("%s", psz);
}



/******************************************************************************
*
*	Function:	main
*
*	Purpose:
*
*		Control the overall flow of the application.
*
*	Arguments:
*
*		argc			The number of command line arguments.
*
*		argv			The list of command line arguments.
*
*	Returned:
*
*		EXIT_SUCCESS	No errors were encounterred.
*		EXIT_FAILURE	One or more problems were encounterred.
*
******************************************************************************/

int main(int argc, char** argv)
{
	args_t	args;
	int		errs;
	time_t	exec		= time(NULL);
	long	failures	= 0;
	long	hours;
	long	mins;
	time_t	now;
	long	passes		= 0;
	int		qty			= 0;
	int		ret;
	long	secs;
	time_t	t_limit;
	time_t	test;
	long	tests		= 0;

	for (;;)
	{
		gsc_label_init(27);
		test	= time(NULL);
		errs	= _parse_args(argc, argv, &args);

		// Introduction

		t_limit	= exec + (args.minute_limit * 60);
		os_id_host();
		_show_time_stamp();
		_show_access_index(&args);
		errs	+= dsi12wi_init_util(1);
		errs	+= os_id_driver(dsi12wi_open, dsi12wi_read, dsi12wi_close);
		errs	+= dsi12wi_count_boards(1, &qty);

		if (qty <= 0)
			errs++;

		if (errs)
			args.continuous	= 0;

		// Open access to device

		if (errs == 0)
			errs	= dsi12wi_open_util(args.index, 0, -1, 1, &args.fd);

		if (errs == 0)
		{
			// Device identification

			errs	= dsi12wi_id_device(args.fd, -1, 1);

			// Perform testing

			errs	+= perform_tests(&args);

			// Close device access

			errs	+= dsi12wi_close_util(args.fd, -1, 1);
		}

		// End of processing.

		now	= time(NULL);
		tests++;

		if (errs)
		{
			failures++;
			printf(	"\nRESULTS: FAIL <---  (%d error%s)",
					errs,
					(errs == 1) ? "" : "s");
		}
		else
		{
			passes++;
			printf("\nRESULTS: PASS");
		}

		secs	= (long) (now - test);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(" (duration %ld:%ld:%02ld)\n", hours, mins, secs);

		secs	= (long) (now - exec);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(	"SUMMARY: tests %ld, pass %ld, fail %ld"
				" (duration %ld:%ld:%02ld)\n\n",
				tests,
				passes,
				failures,
				hours,
				mins,
				secs);

		if (args.continuous == 0)
			break;

		if ((args.continuous < 0) && (errs))
			break;

		if ((args.minute_limit > 0) && (now >= t_limit))
			break;

		if ((args.test_limit > 0) && (tests >= args.test_limit))
			break;
	}

	ret	= ((qty <= 0) || (errs) || (failures)) ? EXIT_FAILURE : EXIT_SUCCESS;
	return(ret);
}



