// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/24DSI12WRCIEPE/24DSI12WRCIEPE_linux_1.x.x.x_gsc_dn/driver/ioctl.c $
// $Rev: 52546 $
// $Date: 2023-03-02 15:06:54 -0600 (Thu, 02 Mar 2023) $

// 24DSI12WRCIEPE: Device Driver: source file

#include "main.h"



// macros *********************************************************************

#define	AUTO_CAL_START			D7		// BCTLR
#define	AUTO_CAL_PASS			D12		// BCTLR
#define	INIT_START				D15		// BCTLR

#define	IRQ_INIT_DONE			0x0000	// BCTLR
#define	IRQ_AUTO_CAL_DONE		0x0100	// BCTLR
#define	IRQ_MASK				0x0700	// BCTLR
#define	IRQ_ACTIVE				0x0800	// BCTLR



// prototypes *****************************************************************

static	int		_wait_for_ready(dev_data_t* dev, long ms_limit);



//*****************************************************************************
static void _bctlr_mod(dev_data_t* dev, u32 value, u32 mask)
{
	if (mask != IRQ_ACTIVE)
	{
		value	|= IRQ_ACTIVE;
		mask	|= IRQ_ACTIVE;
	}

	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, value, mask);
}



//*****************************************************************************
static int _query(dev_data_t* dev, void* arg)
{
	int		ret	= 0;
	s32*	ptr	= (s32*) arg;

	switch (ptr[0])
	{
		default:	ptr[0]	= DSI12WI_IOCTL_QUERY_ERROR;
					ret		= -EINVAL;
					break;

		case DSI12WI_QUERY_AUTO_CAL_MS:		ptr[0]	= dev->cache.auto_cal_ms;	break;
		case DSI12WI_QUERY_CHANNEL_MAX:		ptr[0]	= dev->cache.channels_max;	break;
		case DSI12WI_QUERY_CHANNEL_QTY:		ptr[0]	= dev->cache.channel_qty;	break;
		case DSI12WI_QUERY_COUNT:			ptr[0]	= DSI12WI_QUERY_LAST;		break;
		case DSI12WI_QUERY_CUTOFF_FREQ:		ptr[0]	= dev->cache.cutoff_freq;	break;
		case DSI12WI_QUERY_DEVICE_TYPE:		ptr[0]	= dev->board_type;			break;
		case DSI12WI_QUERY_EXCITATION_MA:	ptr[0]	= dev->cache.excitation_ma;	break;
		case DSI12WI_QUERY_FGEN_MAX:		ptr[0]	= dev->cache.fgen_max;		break;
		case DSI12WI_QUERY_FGEN_MIN:		ptr[0]	= dev->cache.fgen_min;		break;
		case DSI12WI_QUERY_FIFO_SIZE:		ptr[0]	= dev->cache.fifo_size;		break;
		case DSI12WI_QUERY_FILTER_FREQ:		ptr[0]	= dev->cache.filter_freq;	break;
		case DSI12WI_QUERY_FREF_DEFAULT:	ptr[0]	= dev->cache.fref_default;	break;
		case DSI12WI_QUERY_FSAMP_MAX:		ptr[0]	= dev->cache.fsamp_max;		break;
		case DSI12WI_QUERY_FSAMP_MIN:		ptr[0]	= dev->cache.fsamp_min;		break;
		case DSI12WI_QUERY_INIT_MS:			ptr[0]	= dev->cache.initialize_ms;	break;
		case DSI12WI_QUERY_NDIV_MAX:		ptr[0]	= dev->cache.ndiv_max;		break;
		case DSI12WI_QUERY_NDIV_MIN:		ptr[0]	= dev->cache.ndiv_min;		break;
		case DSI12WI_QUERY_NREF_MAX:		ptr[0]	= dev->cache.nref_max;		break;
		case DSI12WI_QUERY_NREF_MIN:		ptr[0]	= dev->cache.nref_min;		break;
		case DSI12WI_QUERY_NVCO_MAX:		ptr[0]	= dev->cache.nvco_max;		break;
		case DSI12WI_QUERY_NVCO_MIN:		ptr[0]	= dev->cache.nvco_min;		break;
	}

	return(ret);
}



//*****************************************************************************
static int _init_start(dev_data_t* dev, void* arg)
{
	// Initiate initialization.
	os_reg_mem_tx_u32(NULL, dev->vaddr.gsc_bctlr_32, INIT_START);
	return(0);
}



//*****************************************************************************
int initialize_ioctl(dev_data_t* dev, void* arg)
{
	u32			bctlr;
	int			i;
	int			mask;
	long		ms			= dev->cache.initialize_ms + 5000;
	long		ms_total	= ms;
	os_sem_t	sem;
	int			ret			= 0;
	int			tmp;
	int			value;
	gsc_wait_t	wt;

	if (dev->irq.opened)
	{
		ms_total	*= 2;

		// Safely select the Initialize Done interrupt.
		mask	= IRQ_MASK | IRQ_ACTIVE;
		value	= IRQ_INIT_DONE;
		os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, value, mask);

		// Wait for the local interrupt.
		os_sem_create(&sem);	// dummy, required for wait operations.
		memset(&wt, 0, sizeof(wt));
		wt.flags		= GSC_WAIT_FLAG_INTERNAL;
		wt.gsc			= DSI12WI_WAIT_GSC_INIT_DONE;
		wt.timeout_ms	= ms;
		ret				= gsc_wait_event(dev, &wt, _init_start, NULL, &sem);
		os_sem_destroy(&sem);

		if (wt.flags & GSC_WAIT_FLAG_TIMEOUT)
		{
			ret	= ret ? ret : -ETIMEDOUT;
			printf(	"%s: INITILIZE DONE IRQ TIMED OUT AFTER %ld ms.\n",
					dev->model,
					(long) ms);
		}
	}

	// Manually wait for completion as the IRQ wait may have ended early.
	tmp	= gsc_poll_u32(dev, ms, dev->vaddr.gsc_bctlr_32, INIT_START, 0);

	if (tmp)
	{
		ret	= ret ? ret : tmp;
		printf(	"%s: INITIALIZATION DID NOT COMPLETE WITHIN %ld ms.\n",
				dev->model,
				ms_total);
	}

	// Manually check for completion as the IRQ wait may have ended early.
	bctlr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);

	if (bctlr & INIT_START)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: INITILIZE STILL ACTIVE AFTER %ld ms.\n",
				dev->model,
				ms_total);
	}

	// Initialize the software settings.

	for (i = 0; i < DEV_IO_STREAM_QTY; i++)
	{
		if (dev->io.io_streams[i])
		{
			if (dev->io.io_streams[i]->dev_io_sw_init)
			{
				(dev->io.io_streams[i]->dev_io_sw_init)(dev, dev->io.io_streams[i]);
			}
		}
	}

	// Wait for settling.
	_wait_for_ready(dev, 20);

	return(ret);
}



//*****************************************************************************
static int _auto_cal_start(dev_data_t* dev, void* arg)
{
	// Initiate auto-calibration.
	_bctlr_mod(dev, AUTO_CAL_START, AUTO_CAL_START);
	return(0);
}



//*****************************************************************************
static int _auto_cal(dev_data_t* dev, void* arg)
{
	u32			bctlr;
	u32			mask;
	long		ms		= dev->cache.auto_cal_ms + 5000;
	u32			reg;
	int			ret;
	os_sem_t	sem;
	int			tmp;
	u32			value;
	gsc_wait_t	wt;

	// Safely select the Auto-Cal Done interrupt.
	mask	= IRQ_MASK | IRQ_ACTIVE;
	value	= IRQ_AUTO_CAL_DONE;
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, value, mask);

	// Wait for the local interrupt.
	os_sem_create(&sem);	// dummy, required for wait operations.
	memset(&wt, 0, sizeof(wt));
	wt.flags		= GSC_WAIT_FLAG_INTERNAL;
	wt.gsc			= DSI12WI_WAIT_GSC_AUTO_CAL_DONE;
	wt.timeout_ms	= ms;
	ret				= gsc_wait_event(dev, &wt, _auto_cal_start, NULL, &sem);
	os_sem_destroy(&sem);

	if (wt.flags & GSC_WAIT_FLAG_TIMEOUT)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATE IRQ TIMED OUT AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually wait for completion in case something terminates our wait early.
	tmp	= gsc_poll_u32(dev, ms, dev->vaddr.gsc_bctlr_32, AUTO_CAL_START, 0);

	if (tmp)
	{
		ms	*= 2;
		ret	= ret ? ret : tmp;
		printf(	"%s: AUTOCALIBRATION DID NOT COMPLETE WITHIN %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually check for completion as the IRQ wait may have ended early.
	bctlr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);

	if (bctlr & AUTO_CAL_START)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATION STILL ACTIVE AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Final results.
	reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);

	if ((reg & AUTO_CAL_PASS) == 0)
	{
		ret	= ret ? ret : -EIO;
		printf(	"%s: AUTOCALIBRATION FAILED (%ld ms).\n",
				dev->model,
				ms);
	}

	if (ret == 0)
	{
		// Wait for settling.
		os_time_sleep_ms(100);
	}

	return(ret);
}



//*****************************************************************************
static int _auto_cal_sts(dev_data_t* dev, s32* arg)
{
	u32	reg;

	reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);

	if (reg & AUTO_CAL_START)
		arg[0]	= DSI12WI_AUTO_CAL_STS_ACTIVE;
	else if (reg & AUTO_CAL_PASS)
		arg[0]	= DSI12WI_AUTO_CAL_STS_PASS;
	else
		arg[0]	= DSI12WI_AUTO_CAL_STS_FAIL;

	return(0);
}



//*****************************************************************************
static int _adc_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_ADC_MODE_HI_RES,
		DSI12WI_ADC_MODE_HI_SPEED,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 19, 19);

	// Wait for settling.
	_wait_for_ready(dev, 700);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_clear(dev_data_t* dev, void* arg)
{
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bufcr_32, D20, D20 | D24 | D25);

	// Wait for settling.
	_wait_for_ready(dev, 20);
	return(0);
}



//*****************************************************************************
static int _ai_buf_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_AI_BUF_ENABLE_NO,
		DSI12WI_AI_BUF_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bufcr_32, 19, 19);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_fill_lvl(dev_data_t* dev, s32* arg)
{
	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_range_reg(dev, arg, 0, 0x40000, dev->vaddr.gsc_bufsr_32, 18, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_overflow(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_AI_BUF_UNDERFLOW_NO,
		DSI12WI_AI_BUF_UNDERFLOW_YES,
		-1	// terminate list
	};

	int	ret;

	if (arg[0] == 1)
		ret	= -EINVAL;
	else
		ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bufcr_32, 24, 24);

	return(ret);
}



//*****************************************************************************
static int _ai_buf_thr_sts(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_AI_BUF_THR_STS_IDLE,
		DSI12WI_AI_BUF_THR_STS_ACTIVE,
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 14, 14);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_thresh(dev_data_t* dev, void* arg)
{
	int	ret;

	ret		= gsc_s32_range_reg(dev, arg, 0, 0x40000, dev->vaddr.gsc_bufcr_32, 18, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_underflow(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_AI_BUF_OVERFLOW_NO,
		DSI12WI_AI_BUF_OVERFLOW_YES,
		-1	// terminate list
	};

	int	ret;

	if (arg[0] == 1)
		ret	= -EINVAL;
	else
		ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bufcr_32, 25, 25);

	return(ret);
}



//*****************************************************************************
static int _ai_channel_tag(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_AI_CHANNEL_TAG_ENABLE,
		DSI12WI_AI_CHANNEL_TAG_DISABLE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bufcr_32, 23, 23);
	return(ret);
}



//*****************************************************************************
static int _ai_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_AI_MODE_DIFF,
		DSI12WI_AI_MODE_ZERO,
		DSI12WI_AI_MODE_VREF,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 1, 0);
	return(ret);
}



//*****************************************************************************
static int _aux_clk_ctl_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_AUX_CLK_CTL_MODE_INACTIVE,
		DSI12WI_AUX_CLK_CTL_MODE_INPUT,
		DSI12WI_AUX_CLK_CTL_MODE_OUTPUT,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_asiocr_32, 1, 0);
	return(ret);
}



//*****************************************************************************
static int _aux_sync_ctl_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_AUX_SYNC_CTL_MODE_INACTIVE,
		DSI12WI_AUX_SYNC_CTL_MODE_INPUT,
		DSI12WI_AUX_SYNC_CTL_MODE_OUTPUT,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_asiocr_32, 3, 2);
	return(ret);
}



//*****************************************************************************
static int _burst_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_BURST_ENABLE_NO,
		DSI12WI_BURST_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 21, 21);
	return(ret);
}



//*****************************************************************************
static int _burst_rate_div(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFFFF, dev->vaddr.gsc_bttr_32, 23, 0);
	return(ret);
}



//*****************************************************************************
static int _burst_size(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFFFF, dev->vaddr.gsc_bbsr_32, 23, 0);
	return(ret);
}



//*****************************************************************************
static int _burst_timer(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_BURST_TIMER_DISABLE,
		DSI12WI_BURST_TIMER_ENABLE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 16, 16);
	return(ret);
}



//*****************************************************************************
static int _burst_trigger(dev_data_t* dev, void* arg)
{
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, D22, D22);

	// Wait for settling.
	_wait_for_ready(dev, 20);
	return(0);
}



//*****************************************************************************
static int _ch_grp_0_src(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_CH_GRP_SRC_RATE_GEN,
		DSI12WI_CH_GRP_SRC_EXTERN,
		DSI12WI_CH_GRP_SRC_DIR_EXTERN,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_csar_32, 3, 0);

	// Wait for settling.
	_wait_for_ready(dev, 20);
	return(ret);
}



//*****************************************************************************
static int _ch_grp_1_src(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_CH_GRP_SRC_RATE_GEN,
		DSI12WI_CH_GRP_SRC_EXTERN,
		DSI12WI_CH_GRP_SRC_DIR_EXTERN,
		DSI12WI_CH_GRP_SRC_DISABLE,
		-1	// terminate list
	};

	s32	g0	= -1;
	s32	g1	= -1;
	int	ret;
	int	ret0;
	int	ret1;

	if (arg[0] == -1)
	{
		ret0	= gsc_s32_list_reg(dev, &g0, options, dev->vaddr.gsc_csar_32, 3, 0);
		ret1	= gsc_s32_list_reg(dev, &g1, options, dev->vaddr.gsc_csar_32, 7, 4);

		arg[0]	= (g1 == DSI12WI_CH_GRP_SRC_DISABLE) ? g1 : g0;
		ret		= ret0 ? ret0 : ret1;
	}
	else
	{
		ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_csar_32, 7, 4);
	}

	// Wait for settling.
	_wait_for_ready(dev, 20);
	return(ret);
}



//*****************************************************************************
static int _channels_ready(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_CHANNELS_READY_NO,
		DSI12WI_CHANNELS_READY_YES,
		-1	// terminate list
	};

	int	ret	= 0;

	if (arg[0] == DSI12WI_CHANNELS_READY_WAIT)
	{
		gsc_poll_u32(dev, 1000, dev->vaddr.gsc_bctlr_32, D13, D13);
		arg[0]	= DSI12WI_CHANNELS_READY_TEST;
	}

	if (arg[0] == DSI12WI_CHANNELS_READY_TEST)
		ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 13, 13);
	else
		ret	= -EINVAL;

	return(ret);
}



//*****************************************************************************
static int _clock_control_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_CONTROL_MODE_TARGET,
		DSI12WI_CONTROL_MODE_INITIATOR,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 5, 5);
	return(ret);
}



//*****************************************************************************
static int _coupling_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_COUPLING_MODE_AC,
		DSI12WI_COUPLING_MODE_DC,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_ipcr_32, 0, 0);
	return(ret);
}



//*****************************************************************************
static int _data_format(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_DATA_FORMAT_2S_COMP,
		DSI12WI_DATA_FORMAT_OFF_BIN,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 4, 4);
	return(ret);
}



//*****************************************************************************
static int _data_width(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_DATA_WIDTH_16,
		DSI12WI_DATA_WIDTH_18,
		DSI12WI_DATA_WIDTH_20,
		DSI12WI_DATA_WIDTH_24,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bufcr_32, 22, 21);
	return(ret);
}



//*****************************************************************************
static int _ext_clk_src(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_EXT_CLK_SRC_GRP_0,
		DSI12WI_EXT_CLK_SRC_GEN,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 18, 18);
	return(ret);
}



//*****************************************************************************
static int _ext_sync_dir(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_EXT_SYNC_DIR_OUT,
		DSI12WI_EXT_SYNC_DIR_IN,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 23, 23);
	return(ret);
}



//*****************************************************************************
static int _excitation_current(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_EXCITATION_CURRENT_OFF,
		DSI12WI_EXCITATION_CURRENT_ON,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_ipcr_32, 1, 1);
	return(ret);
}



//*****************************************************************************
static int _input_comp(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_INPUT_COMP_OFF,
		DSI12WI_INPUT_COMP_ON,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_ipcr_32, 2, 2);
	return(ret);
}



//*****************************************************************************
static int _irq_sel(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_IRQ_INIT_DONE,
		DSI12WI_IRQ_AUTO_CAL_DONE,
		DSI12WI_IRQ_CHAN_READY,
		DSI12WI_IRQ_AI_BUF_THRESH_L2H,
		DSI12WI_IRQ_AI_BUF_THRESH_H2L,
		DSI12WI_IRQ_AI_BURST_DONE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 10, 8);
	return(ret);
}



//*****************************************************************************
static int _ndiv(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 1, 300, dev->vaddr.gsc_rdr_32, 8, 0);

	// Wait for settling.
	_wait_for_ready(dev, 20);
	return(ret);
}



//*****************************************************************************
static int _nref(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 25, 300, dev->vaddr.gsc_rcr_32, 23, 12);

	// Wait for settling.
	_wait_for_ready(dev, 20);
	return(ret);
}



//*****************************************************************************
static int _nvco(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 25, 300, dev->vaddr.gsc_rcr_32, 11, 0);

	// Wait for settling.
	_wait_for_ready(dev, 20);
	return(ret);
}



//*****************************************************************************
static int _range(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_RANGE_10MV,
		DSI12WI_RANGE_100MV,
		DSI12WI_RANGE_1V,
		DSI12WI_RANGE_10V,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 3, 2);
	return(ret);
}



//*****************************************************************************
static int _sync_control_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_CONTROL_MODE_TARGET,
		DSI12WI_CONTROL_MODE_INITIATOR,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 20, 20);
	return(ret);
}



//*****************************************************************************
static int _sw_sync(dev_data_t* dev, void* arg)
{
	_bctlr_mod(dev, D6, D6);
	return(0);
}



//*****************************************************************************
static int _sw_sync_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_SW_SYNC_MODE_SYNC,
		DSI12WI_SW_SYNC_MODE_CLR_BUF,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 17, 17);
	return(ret);
}



//*****************************************************************************
static int _rx_io_abort(dev_data_t* dev, s32* arg)
{
	arg[0]	= gsc_read_abort_active_xfer(dev, &dev->io.rx);
	return(0);
}



//*****************************************************************************
static int _rx_io_mode(dev_data_t* dev, s32* arg)
{
	static const s32	list[]	=
	{
		GSC_IO_MODE_PIO,
		GSC_IO_MODE_BMDMA,
		GSC_IO_MODE_DMDMA,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.rx.io_mode);
	return(ret);
}



//*****************************************************************************
static int _rx_io_overflow(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_IO_OVERFLOW_IGNORE,
		DSI12WI_IO_OVERFLOW_CHECK,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, options, &dev->io.rx.overflow_check);
	return(ret);
}



//*****************************************************************************
static int _rx_io_timeout(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_var(
			arg,
			DSI12WI_IO_TIMEOUT_MIN,
			DSI12WI_IO_TIMEOUT_INFINITE,
			&dev->io.rx.timeout_s);
	return(ret);
}



//*****************************************************************************
static int _rx_io_underflow(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		DSI12WI_IO_UNDERFLOW_IGNORE,
		DSI12WI_IO_UNDERFLOW_CHECK,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, options, &dev->io.rx.underflow_check);
	return(ret);
}



// variables ******************************************************************

const gsc_ioctl_t	dev_ioctl_list[]	=
{
	{ DSI12WI_IOCTL_REG_READ,			(void*) gsc_reg_read_ioctl		},
	{ DSI12WI_IOCTL_REG_WRITE,			(void*) gsc_reg_write_ioctl		},
	{ DSI12WI_IOCTL_REG_MOD,			(void*) gsc_reg_mod_ioctl		},
	{ DSI12WI_IOCTL_QUERY,				(void*) _query					},
	{ DSI12WI_IOCTL_INITIALIZE,			(void*) initialize_ioctl		},
	{ DSI12WI_IOCTL_AUTO_CAL,			(void*) _auto_cal				},
	{ DSI12WI_IOCTL_AUTO_CAL_STS,		(void*) _auto_cal_sts			},
	{ DSI12WI_IOCTL_ADC_MODE,			(void*) _adc_mode				},
	{ DSI12WI_IOCTL_AI_BUF_CLEAR,		(void*) _ai_buf_clear			},
	{ DSI12WI_IOCTL_AI_BUF_ENABLE,		(void*) _ai_buf_enable			},
	{ DSI12WI_IOCTL_AI_BUF_FILL_LVL,	(void*) _ai_buf_fill_lvl		},
	{ DSI12WI_IOCTL_AI_BUF_OVERFLOW,	(void*) _ai_buf_overflow		},
	{ DSI12WI_IOCTL_AI_BUF_THR_STS,		(void*) _ai_buf_thr_sts			},
	{ DSI12WI_IOCTL_AI_BUF_THRESH,		(void*) _ai_buf_thresh			},
	{ DSI12WI_IOCTL_AI_BUF_UNDERFLOW,	(void*) _ai_buf_underflow		},
	{ DSI12WI_IOCTL_AI_CHANNEL_TAG,		(void*) _ai_channel_tag			},
	{ DSI12WI_IOCTL_AI_MODE,			(void*) _ai_mode				},
	{ DSI12WI_IOCTL_AUX_CLK_CTL_MODE,	(void*) _aux_clk_ctl_mode		},
	{ DSI12WI_IOCTL_AUX_SYNC_CTL_MODE,	(void*) _aux_sync_ctl_mode		},
	{ DSI12WI_IOCTL_BURST_ENABLE,		(void*) _burst_enable			},
	{ DSI12WI_IOCTL_BURST_RATE_DIV,		(void*) _burst_rate_div			},
	{ DSI12WI_IOCTL_BURST_SIZE,			(void*) _burst_size				},
	{ DSI12WI_IOCTL_BURST_TIMER,		(void*) _burst_timer			},
	{ DSI12WI_IOCTL_BURST_TRIGGER,		(void*) _burst_trigger			},
	{ DSI12WI_IOCTL_CH_GRP_0_SRC,		(void*) _ch_grp_0_src			},
	{ DSI12WI_IOCTL_CH_GRP_1_SRC,		(void*) _ch_grp_1_src			},
	{ DSI12WI_IOCTL_CHANNELS_READY,		(void*) _channels_ready			},
	{ DSI12WI_IOCTL_CLOCK_CONTROL_MODE,	(void*) _clock_control_mode		},
	{ DSI12WI_IOCTL_COUPLING_MODE,		(void*) _coupling_mode			},
	{ DSI12WI_IOCTL_DATA_FORMAT,		(void*) _data_format			},
	{ DSI12WI_IOCTL_DATA_WIDTH,			(void*) _data_width				},
	{ DSI12WI_IOCTL_EXT_CLK_SRC,		(void*) _ext_clk_src			},
	{ DSI12WI_IOCTL_EXT_SYNC_DIR,		(void*) _ext_sync_dir			},
	{ DSI12WI_IOCTL_EXCITATION_CURRENT,	(void*) _excitation_current		},
	{ DSI12WI_IOCTL_INPUT_COMP,			(void*) _input_comp				},
	{ DSI12WI_IOCTL_IRQ_SEL,			(void*) _irq_sel				},
	{ DSI12WI_IOCTL_NDIV,				(void*) _ndiv					},
	{ DSI12WI_IOCTL_NREF,				(void*) _nref					},
	{ DSI12WI_IOCTL_NVCO,				(void*) _nvco					},
	{ DSI12WI_IOCTL_RANGE,				(void*) _range					},
	{ DSI12WI_IOCTL_SYNC_CONTROL_MODE,	(void*) _sync_control_mode		},
	{ DSI12WI_IOCTL_SW_SYNC,			(void*) _sw_sync				},
	{ DSI12WI_IOCTL_SW_SYNC_MODE,		(void*) _sw_sync_mode			},
	{ DSI12WI_IOCTL_RX_IO_ABORT,		(void*) _rx_io_abort			},
	{ DSI12WI_IOCTL_RX_IO_MODE,			(void*) _rx_io_mode				},
	{ DSI12WI_IOCTL_RX_IO_OVERFLOW,		(void*) _rx_io_overflow			},
	{ DSI12WI_IOCTL_RX_IO_TIMEOUT,		(void*) _rx_io_timeout			},
	{ DSI12WI_IOCTL_RX_IO_UNDERFLOW,	(void*) _rx_io_underflow		},
	{ DSI12WI_IOCTL_WAIT_EVENT,			(void*) gsc_wait_event_ioctl	},
	{ DSI12WI_IOCTL_WAIT_CANCEL,		(void*) gsc_wait_cancel_ioctl	},
	{ DSI12WI_IOCTL_WAIT_STATUS,		(void*) gsc_wait_status_ioctl	},
	{ -1, NULL }
};



//*****************************************************************************
static int _wait_for_ready(dev_data_t* dev, long ms_limit)
{
	int	ret;

	ret	= gsc_poll_u32(dev, ms_limit, dev->vaddr.gsc_bctlr_32, D13, D13);
	return(ret);
}



