// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/18AISS6C/utils/util_reg.c $
// $Rev: 53510 $
// $Date: 2023-08-03 10:05:51 -0500 (Thu, 03 Aug 2023) $

// 18AISS6C: Utilities: source file

#include "main.h"




// macros *********************************************************************

#define	_GSC_REG(a)				"GSC " #a, AISS6C_GSC_##a, 0, 0
#define	SHOW_FIELD(b,e,eol,l,d)	gsc_reg_field_show(width+10,23, value,(b),(e),(eol),(l),(d))
#define	SIZEOF_ARRAY(a)			((sizeof((a)) / sizeof((a)[0]))



//*****************************************************************************
static int _bcr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	clear[]		= { "Clear",			"Set"			};
	static const char*	disable[]	= { "Disable",			"Enable"		};
	static const char*	fail[]		= { "Fail",				"Pass"			};
	static const char*	idle[]		= { "Idle",				"Active"		};
	static const char*	idle_2[]	= { "Idle",				"Busy"			};
	static const char*	no[]		= { "No",				"Yes"			};
	static const char*	twos[]		= { "Twos Compliment",	"Offset Binary"	};

	SHOW_FIELD( 7,  0, 1, NULL,		"Reserved"				);
	SHOW_FIELD( 8,  8, 1, idle,		"Input SW Clock"		);
	SHOW_FIELD( 9,  9, 1, disable,	"Input Burst"			);
	SHOW_FIELD(10, 10, 1, idle_2,	"Input Burst Status"	);
	SHOW_FIELD(11, 11, 1, idle,		"Input SW Trigger"		);
	SHOW_FIELD(12, 12, 1, disable,	"Input Buffer"			);
	SHOW_FIELD(13, 13, 1, idle,		"Clear Input Buffer"	);
	SHOW_FIELD(14, 14, 1, clear,	"Input Buf Thresh Sts"	);
	SHOW_FIELD(15, 15, 1, no,		"Buffer Overflow"		);
	SHOW_FIELD(16, 16, 1, no,		"Buffer Underflow"		);
	SHOW_FIELD(17, 17, 1, twos,		"Data Format"			);
	SHOW_FIELD(18, 18, 1, disable,	"Rate-A Generator"		);
	SHOW_FIELD(19, 19, 1, disable,	"Rate-B Generator"		);
	SHOW_FIELD(20, 20, 1, idle,		"Autocalibration"		);
	SHOW_FIELD(21, 21, 1, fail,		"Autocal Status"		);
	SHOW_FIELD(22, 22, 1, disable,	"Warp Mode"				);
	SHOW_FIELD(30, 23, 1, NULL,		"Reserved"				);
	SHOW_FIELD(31, 31, 1, idle,		"Initialize"			);
	return(0);
}



//*****************************************************************************
static int _diopr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	input[]	= { "Input",	"Output"	};
	static const char*	low[]	= { "0 (low)",	"1 (high)"	};

	SHOW_FIELD( 0,  0, 1, low,		"DIO 0 Data"		);
	SHOW_FIELD( 1,  1, 1, low,		"DIO 1 Data"		);
	SHOW_FIELD( 2,  2, 1, low,		"DIO 2 Data"		);
	SHOW_FIELD( 3,  3, 1, low,		"DIO 3 Data"		);
	SHOW_FIELD( 6,  4, 1, NULL,		"Reserved"			);
	SHOW_FIELD( 7,  7, 1, input,	"DIO 0-3 Direction"	);

	SHOW_FIELD( 8,  8, 1, low,		"DIO 4 Data"		);
	SHOW_FIELD( 9,  9, 1, low,		"DIO 5 Data"		);
	SHOW_FIELD(10, 10, 1, low,		"DIO 6 Data"		);
	SHOW_FIELD(11, 11, 1, low,		"DIO 7 Data"		);
	SHOW_FIELD(14, 12, 1, NULL,		"Reserved"			);
	SHOW_FIELD(15, 15, 1, input,	"DIO 4-7 Direction"	);
	SHOW_FIELD(31, 16, 1, NULL,		"Reserved"			);
	return(0);
}



//*****************************************************************************
static int _asior_decode(int fd, int supported, u32 value, int width)
{
	static const char*	mode[]	=
	{
		"Inactive",
		"Active Input (Low-to-High Edge)",
		"Active Output (Positive Pulse)",
		"RESERVED"
	};

	static const char*	no[]	= { "No",	"Yes"	};
	static const char*	noise[]	= { "Low",	"High"	};

	SHOW_FIELD( 1,  0, 1, mode,		"Aux Clk Control Mode"	);
	SHOW_FIELD( 3,  2, 1, mode,		"Aux Sync Control Mode"	);
	SHOW_FIELD( 7,  4, 1, NULL,		"Reserved"				);
	SHOW_FIELD( 8,  8, 1, no,		"Invert Inputs"			);
	SHOW_FIELD( 9,  9, 1, no,		"Invert Outputs"		);
	SHOW_FIELD(10, 10, 1, noise,	"Noise Suppression"		);
	SHOW_FIELD(31, 11, 1, NULL,		"Reserved"				);
	return(0);
}



//*****************************************************************************
static int _icr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_ai_cfg[]	=
	{
		"External Analog Inputs",
		"RESERVED",
		"Zero Selftest",
		"+Vref Selftest"
	};

	static const char*	_clk_src[]	=
	{
		"Rate-A Generator",
		"Rate-B Generator",
		"External Clock Input",
		"RESERVED"
	};

	static const char*	_trg_src[]	=
	{
		"Rate-B Generator",
		"Rate-A Generator",
		"External Clock Input",
		"RESERVED"
	};

	static const char*	disable[]	= { "Disable",	"Enable"	};
	static const char*	low[]		= { "Low",		"High"		};

	SHOW_FIELD( 0,  0, 1, disable,	"Input 0"				);
	SHOW_FIELD( 1,  1, 1, disable,	"Input 1"				);
	SHOW_FIELD( 2,  2, 1, disable,	"Input 2"				);
	SHOW_FIELD( 3,  3, 1, disable,	"Input 3"				);
	SHOW_FIELD( 4,  4, 1, disable,	"Input 4"				);
	SHOW_FIELD( 5,  5, 1, disable,	"Input 5"				);
	SHOW_FIELD( 7,  6, 1, NULL,		"Reserved"				);
	SHOW_FIELD( 9,  8, 1, _clk_src,	"Input Clock Source"	);
	SHOW_FIELD(11, 10, 1, NULL,		"Reserved"				);
	SHOW_FIELD(13, 12, 1, _trg_src,	"Input Trigger Source"	);
	SHOW_FIELD(15, 14, 1, NULL,		"Reserved"				);
	SHOW_FIELD(17, 16, 1, _ai_cfg,	"Input Configuration"	);
	SHOW_FIELD(19, 18, 1, NULL,		"Reserved"				);
	SHOW_FIELD(20, 20, 1, low,		"Input Voltage Range"	);
	SHOW_FIELD(23, 21, 1, NULL,		"Reserved"				);
	SHOW_FIELD(24, 24, 1, disable,	"Input Filter"			);
	SHOW_FIELD(31, 25, 1, NULL,		"Reserved"				);
	return(0);
}



//*****************************************************************************
static int _rgxr_decode(int fd, int supported, u32 value, int width)
{
	SHOW_FIELD(23,  0, 1, NULL,	"Ndiv"		);
	SHOW_FIELD(31, 24, 1, NULL,	"Reserved"	);
	return(0);
}



//*****************************************************************************
static int _ibbsr_decode(int fd, int supported, u32 value, int width)
{
	SHOW_FIELD(23,  0, 1, NULL,	"Size"		);
	SHOW_FIELD(31, 24, 1, NULL,	"Reserved"	);
	return(0);
}



//*****************************************************************************
static int _ibsr_decode(int fd, int supported, u32 value, int width)
{
	SHOW_FIELD(18,  0, 1, NULL,	"Size"		);
	SHOW_FIELD(31, 19, 1, NULL,	"Reserved"	);
	return(0);
}



//*****************************************************************************
static int _ibtr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	clear[]	= { "Clear",	"Set"	};

	SHOW_FIELD(18,  0, 1, NULL,		"Threshold"			);
	SHOW_FIELD(19, 19, 1, NULL,		"Reserved"			);
	SHOW_FIELD(20, 20, 1, clear,	"Threshold Status"	);
	SHOW_FIELD(31, 21, 1, NULL,		"Reserved"			);
	return(0);
}



//*****************************************************************************
static int _psr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	idle[]		= { "Idle",		"Active"	};
	static const char*	ignore[]	= { "Ignore",	"Selected"	};

	SHOW_FIELD( 0,  0, 1, ignore,	"Autocal Complete"		);
	SHOW_FIELD( 1,  1, 1, ignore,	"AI Thresh High -> Low"	);
	SHOW_FIELD( 2,  2, 1, ignore,	"AI Thresh Low -> High"	);
	SHOW_FIELD( 3,  3, 1, ignore,	"Input Buffer Fault"	);
	SHOW_FIELD( 4,  4, 1, ignore,	"AI Burst Initiated"	);
	SHOW_FIELD( 5,  5, 1, ignore,	"AI Burst Complete"		);
	SHOW_FIELD( 6,  6, 1, ignore,	"AI Clock"				);
	SHOW_FIELD( 7,  7, 1, ignore,	"DIO 0 Low->High"		);
	SHOW_FIELD(15,  8, 1, NULL,		"Reserved"				);
	SHOW_FIELD(16, 16, 1, idle,		"Autocal Complete"		);
	SHOW_FIELD(17, 17, 1, idle,		"AI Thresh High -> Low"	);
	SHOW_FIELD(18, 18, 1, idle,		"AI Thresh Low -> High"	);
	SHOW_FIELD(19, 19, 1, idle,		"Input Buffer Fault"	);
	SHOW_FIELD(20, 20, 1, idle,		"AI Burst Initiated"	);
	SHOW_FIELD(21, 21, 1, idle,		"AI Burst Complete"		);
	SHOW_FIELD(22, 22, 1, idle,		"AI Clock"				);
	SHOW_FIELD(23, 23, 1, idle,		"DIO 0 Low->High"		);
	SHOW_FIELD(31, 24, 1, NULL,		"Reserved"				);
	return(0);
}



//*****************************************************************************
static int _acr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	_clock[]	=
	{
		"45.000 MHz",
		"RESERVED",
		"RESERVED",
		"RESERVED"
	};

	static const char*	_in_chan[]	=
	{
		"2 Input Channels",
		"4 Input Channels",
		"6 Input Channels",
		"RESERVED"
	};

	static const char*	_in_filt[]	=
	{
		"Type-F1: 200KHz Chebyshev, 5th Order, 0.5db",
		"RESERVED",
		"RESERVED",
		"RESERVED"
	};

	static const char*	_range[]	=
	{
		"High Range: +-20V, +-10V",
		"Low Range: +-10V, +-5V",
		"RESERVED",
		"RESERVED"
	};

	u32	val;

	SHOW_FIELD(11,  0, 0, NULL,		"Firmware Revision"	);
	val	= GSC_FIELD_DECODE(value, 11, 0);
	printf("%03lX\n", (long) val);

	SHOW_FIELD(15, 12, 1, NULL,		"Reserved"			);
	SHOW_FIELD(17, 16, 1, _in_chan,	"Input Channels"	);
	SHOW_FIELD(19, 18, 1, _in_filt,	"Input Filter"		);
	SHOW_FIELD(21, 20, 1, _clock,	"Master Clock Freq"	);
	SHOW_FIELD(23, 22, 1, _range,	"Input Range"		);
	SHOW_FIELD(31, 24, 1, NULL,		"Reserved"			);
	return(0);
}



// variables ******************************************************************

static gsc_reg_def_t	_gsc[]	=
{
	{ _GSC_REG(BCR),	0,	_bcr_decode,	"Board Control Register"			},
	{ _GSC_REG(DIOPR),	0,	_diopr_decode,	"Digital I/O Port Register"			},
	{ _GSC_REG(ASIOR),	0,	_asior_decode,	"Auxiliary Sync I/O Register"		},
	{ _GSC_REG(ICR),	0,	_icr_decode,	"Input Configuration Register"		},
	{ _GSC_REG(IDBR),	0,	NULL,			"Input Data Buffer Register"		},
	{ _GSC_REG(RGAR),	0,	_rgxr_decode,	"Rate Generator A Register"			},
	{ _GSC_REG(RGBR),	0,	_rgxr_decode,	"Rate Generator B Register"			},
	{ _GSC_REG(IBBSR),	0,	_ibbsr_decode,	"Input Burst Block Size Register"	},
	{ _GSC_REG(IBSR),	0,	_ibsr_decode,	"Input Buffer Size Register"		},
	{ _GSC_REG(IBTR),	0,	_ibtr_decode,	"Input Buffer Threshold Register"	},
	{ _GSC_REG(PSR),	0,	_psr_decode,	"Primary Status Register"			},
	{ _GSC_REG(ACR),	0,	_acr_decode,	"Assymbly Configuration Register"	},
	{ _GSC_REG(AVR),	0,	NULL,			"Autocal Values Register"			},

	{ NULL, 0, 0, 0, 0,		NULL	}
};



//*****************************************************************************
static const gsc_reg_def_t* _find_reg(u32 reg, const gsc_reg_def_t* list)
{
	const gsc_reg_def_t*	def	= NULL;
	int						i;

	for (i = 0; list[i].name; i++)
	{
		if (reg == list[i].reg)
		{
			def	= &list[i];
			break;
		}
	}

	return(def);
}



/******************************************************************************
*
*	Function:	aiss6c_reg_get_def_id
*
*	Purpose:
*
*		Retrieve the register definition structure given the register id.
*
*	Arguments:
*
*		reg		The id of the register to access.
*
*	Returned:
*
*		NULL	The register id wasn't found.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* aiss6c_reg_get_def_id(u32 reg)
{
	const gsc_reg_def_t*	def;

	def	= _find_reg(reg, _gsc);
	return(def);
}



/******************************************************************************
*
*	Function:	aiss6c_reg_get_def_index
*
*	Purpose:
*
*		Retrieve the register definition structure based on an index.
*
*	Arguments:
*
*		index	The index of the register to access.
*
*	Returned:
*
*		NULL	The index doesn't correspond to a known register.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* aiss6c_reg_get_def_index(int index)
{
	const gsc_reg_def_t*	def;

	if (index < 0)
		def	= NULL;
	else if (index >= (SIZEOF_ARRAY(_gsc) - 1)))
		def	= NULL;
	else
		def	= &_gsc[index];

	return(def);
}



/******************************************************************************
*
*	Function:	aiss6c_reg_get_desc
*
*	Purpose:
*
*		Retrieve the description of the specified register.
*
*	Arguments:
*
*		reg		The register whose description is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* aiss6c_reg_get_desc(u32 reg)
{
	const gsc_reg_def_t*	def;
	const char*				desc;

	def	= _find_reg(reg, _gsc);

	if (def)
		desc	= def->desc;
	else
		desc	= "UNKNOWN";

	return(desc);
}



/******************************************************************************
*
*	Function:	aiss6c_reg_get_name
*
*	Purpose:
*
*		Retrieve the name of the specified register.
*
*	Arguments:
*
*		reg		The register whose name is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* aiss6c_reg_get_name(u32 reg)
{
	const gsc_reg_def_t*	def;
	const char*				name;

	def	= _find_reg(reg, _gsc);

	if (def)
		name	= def->name;
	else
		name	= "UNKNOWN";

	return(name);
}



/******************************************************************************
*
*	Function:	aiss6c_reg_list
*
*	Purpose:
*
*		List the GSC registers and their values.
*
*	Arguments:
*
*		fd		The handle to access the device.
*
*		detail	List the register details?
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int aiss6c_reg_list(int fd, int detail)
{
	int	errs;

	errs	= gsc_reg_list(fd, _gsc, detail, aiss6c_reg_read);
	return(errs);
}



