// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/18AISS6C/utils/util_query.c $
// $Rev: 53510 $
// $Date: 2023-08-03 10:05:51 -0500 (Thu, 03 Aug 2023) $

// 18AISS6C: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	aiss6c_query
*
*	Purpose:
*
*		Provide a visual wrapper for the AISS6C_IOCTL_QUERY service.
*
*	Arguments:
*
*		fd		Use this handle to access the device.
*
*		index	The index of the device to access. Ignore if < 0.
*				This is for display purposes only.
*
*		verbose	Work verbosely?
*
*		set		This is the query option to access.
*
*		get		The results are reported here. This may be NULL.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int	aiss6c_query(int fd, int index, int verbose, s32 set, s32* get)
{
	char		buf[128];
	int			errs	= 0;
	const char*	ptr;
	s32			query	= set;
	int			ret;
	s32			tmp;

	switch (query)
	{
		default:

			errs++;
			ptr	= buf;
			sprintf(buf, "Query Error (#%ld)", (long) query);
			break;

		case AISS6C_QUERY_AI_FILTER:	ptr	= "AI Filter";			break;
		case AISS6C_QUERY_AI_RANGE:		ptr	= "Voltage Range";		break;
		case AISS6C_QUERY_AUTOCAL_MS:	ptr	= "Autocal Duration";	break;
		case AISS6C_QUERY_CHANNEL_MAX:	ptr	= "Max Channels";		break;
		case AISS6C_QUERY_CHANNEL_QTY:	ptr	= "Installed Channels";	break;
		case AISS6C_QUERY_COUNT:		ptr	= "Query Options";		break;
		case AISS6C_QUERY_DEVICE_TYPE:	ptr	= "Device Type";		break;
		case AISS6C_QUERY_FGEN_MAX:		ptr	= "Fgen Max";			break;
		case AISS6C_QUERY_FGEN_MIN:		ptr	= "Fgen Min";			break;
		case AISS6C_QUERY_FIFO_SIZE:	ptr	= "FIFO Size";			break;
		case AISS6C_QUERY_FSAMP_MAX:	ptr	= "Fsamp Max";			break;
		case AISS6C_QUERY_FSAMP_MIN:	ptr	= "Fsamp Min";			break;
		case AISS6C_QUERY_INIT_MS:		ptr	= "Init. Duration";		break;
		case AISS6C_QUERY_MASTER_CLOCK:	ptr	= "Master Clock";		break;
		case AISS6C_QUERY_NDIV_MAX:		ptr	= "Ndiv Max";			break;
		case AISS6C_QUERY_NDIV_MIN:		ptr	= "Ndiv Min";			break;
		case AISS6C_QUERY_RATE_GEN_QTY:	ptr	= "Rate Gen Qty";		break;
	}

	if (verbose)
		gsc_label_index(ptr, index);

	ret		= aiss6c_ioctl(fd, AISS6C_IOCTL_QUERY, &set);
	errs	+= ret ? 1 : 0;

	switch (query)
	{
		default:

			errs++;
			sprintf(buf, "%ld", (long) set);
			break;

		case AISS6C_QUERY_CHANNEL_MAX:
		case AISS6C_QUERY_CHANNEL_QTY:
		case AISS6C_QUERY_COUNT:
		case AISS6C_QUERY_NDIV_MAX:
		case AISS6C_QUERY_NDIV_MIN:
		case AISS6C_QUERY_RATE_GEN_QTY:

			gsc_label_long_comma_buf(set, buf);
			break;

		case AISS6C_QUERY_FGEN_MAX:
		case AISS6C_QUERY_FGEN_MIN:
		case AISS6C_QUERY_MASTER_CLOCK:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " Hz");
			break;

		case AISS6C_QUERY_AUTOCAL_MS:
		case AISS6C_QUERY_INIT_MS:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " ms");
			break;

		case AISS6C_QUERY_FSAMP_MAX:
		case AISS6C_QUERY_FSAMP_MIN:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " S/S");
			break;

		case AISS6C_QUERY_FIFO_SIZE:

			ptr	= "";
			tmp	= set;

			if ((tmp) && ((tmp % 1024) == 0))
			{
				ptr	= "K";
				tmp	/= 1024;
			}

			gsc_label_long_comma_buf(tmp, buf);
			strcat(buf, ptr);
			strcat(buf, " Words");
			break;

		case AISS6C_QUERY_DEVICE_TYPE:

			if (set == GSC_DEV_TYPE_18AISS6C)
			{
				strcpy(buf, "18AISS6C");
			}
			else
			{
				errs++;
				sprintf(buf, "INVALID: %ld", (long) set);
			}

			break;

		case AISS6C_QUERY_AI_FILTER:

			switch (set)
			{
				default:

					errs++;
					sprintf(buf, "Invalid option: %ld", (long) set);
					break;

				case AISS6C_AI_FILTER_F1:	strcpy(buf, "200KHz Chebysheb, 5th order, 0.5db");	break;
			}

			break;

		case AISS6C_QUERY_AI_RANGE:

			switch (set)
			{
				default:

					errs++;
					sprintf(buf, "Invalid option: %ld", (long) set);
					break;

				case AISS6C_AI_RANGE_20_10:	strcpy(buf, "+-20 Volts, +-10 Volts");	break;
				case AISS6C_AI_RANGE_10_5:	strcpy(buf, "+-10 Volts, +-5 Volts");	break;
			}

			break;
	}

	if (verbose)
	{
		if (errs)
			printf("FAIL <---  (%s)\n", buf);
		else
			printf("%s\n", buf);
	}

	if (get)
		get[0]	= set;

	return(errs);
}


