// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/18AISS6C/utils/util_fsamp_ai.c $
// $Rev: 53510 $
// $Date: 2023-08-03 10:05:51 -0500 (Thu, 03 Aug 2023) $

// 18AISS6C: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	ABS(v)					(((v) < 0) ? (-(v)) : (v))
#define	LIMIT_MAX(v,max)		(((v) > (max)) ? (max) : (v))
#define	LIMIT_MIN(v,min)		(((v) < (min)) ? (min) : (v))
#define	LIMIT_RANGE(v,min,max)	(LIMIT_MIN((LIMIT_MAX((v),(max))),(min)))



/******************************************************************************
*
*	Function:	aiss6c_fsamp_ai_compute
*
*	Purpose:
*
*		Determine the best configuration for the specified input sample rate.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fsamp	The desired sample rate.
*
*		ndiv	The Ndiv value is put here. May be NULL.
*
*		warp	The warp mode value is put here. May be NULL.
*
*		rate	Report the sample rate achieved here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int	aiss6c_fsamp_ai_compute(int fd, int index, int verbose, s32 fsamp, s32* ndiv, s32* warp, double* rate)
{
	double	delta;
	double	delta_best	= FLT_MAX;
	int		errs		= 0;
	double	fsamp_best	= 0;
	double	fsamp_got;
	s32		fsamp_max;	// Queried value.
	s32		fsamp_min;	// Queried value.
	double	fsamp_want;
	s32		master;
	s32		nd;
	s32		nd_best		= 0;
	s32		nd_max;
	s32		nd_min;
	s32		nd_use;
	s32		ndiv_max;	// Queried value.
	s32		ndiv_min;	// Queried value.
	s32		wp_best;

	if (verbose)
	{
		gsc_label_index("AI Sample Rate", index);
		gsc_label_long_comma(fsamp);
		printf(" S/S");
	}

	errs	+= aiss6c_query(fd, -1, 0, AISS6C_QUERY_FSAMP_MAX, &fsamp_max);
	errs	+= aiss6c_query(fd, -1, 0, AISS6C_QUERY_FSAMP_MIN, &fsamp_min);
	errs	+= aiss6c_query(fd, -1, 0, AISS6C_QUERY_MASTER_CLOCK, &master);
	errs	+= aiss6c_query(fd, -1, 0, AISS6C_QUERY_NDIV_MAX, &ndiv_max);
	errs	+= aiss6c_query(fd, -1, 0, AISS6C_QUERY_NDIV_MIN, &ndiv_min);

	fsamp_want	= (double) LIMIT_RANGE(fsamp, fsamp_min, fsamp_max);
	nd			= master / fsamp_want;
	nd_min		= nd - 1;
	nd_max		= nd + 1;
	nd_min		= (nd_min < ndiv_min) ? ndiv_min : nd_min;
	nd_max		= (nd_max > ndiv_max) ? ndiv_max : nd_max;

	for (nd = nd_min; nd <= nd_max; nd++)
	{
		if (nd < nd_min)
			nd_use	= nd_min;
		else if (nd > nd_max)
			nd_use	= nd_max;
		else
			nd_use	= nd;

		fsamp_got	= ((double) master) / nd_use;
		delta		= fsamp_got - fsamp_want;

		if ((ABS(delta)) < (ABS(delta_best)))
		{
			delta_best	= delta;
			fsamp_best	= fsamp_got;
			nd_best		= nd_use;
		}
	}

	if (fsamp_best >= 500000L)
		wp_best	= AISS6C_AI_WARP_MODE_ENABLE;
	else
		wp_best	= AISS6C_AI_WARP_MODE_DISABLE;

	if (ndiv)
		ndiv[0]	= nd_best;

	if (warp)
		warp[0]	= wp_best;

	if (rate)
		rate[0]	= fsamp_best;

	if (verbose)
	{
		printf("%s  (", errs ? " FAIL <---" : "");
		printf("Master ");
		gsc_label_long_comma((long) master);
		printf(", Ndiv ");
		gsc_label_long_comma((long) nd_best);
		printf(", Warp Mode %s", wp_best ? "On" : "Off");
		printf(", Fsamp %.3f", fsamp_best);
		printf(")\n");
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	aiss6c_fsamp_ai_get
*
*	Purpose:
*
*		Determine the input sample rate.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		title	Show a title?
*
*		get		Store the sample rate here, if non-NULL.
*
*		warp	Store the warp mode setting here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int	aiss6c_fsamp_ai_get(int fd, int title, s32* get, s32* warp)
{
	s32		burst;
	s32		enable;
	s32		enable_a;
	s32		enable_b;
	s32		enable_gen;
	int		errs		= 0;
	char	gen;
	s32		master;
	s32		ndiv_a;
	s32		ndiv_b;
	s32		ndiv_gen;
	s32		set			= 0;
	s32		src;

	if (title)
		gsc_label("AI Sample Rate");

	errs	+= aiss6c_query(fd, -1, 0, AISS6C_QUERY_MASTER_CLOCK, &master);

	errs	+= aiss6c_ai_enable			(fd, -1, 0, -1, &enable);
	errs	+= aiss6c_ai_clk_src		(fd, -1, 0, -1, &src);
	errs	+= aiss6c_ai_burst_enable	(fd, -1, 0, -1, &burst);
	errs	+= aiss6c_gen_a_enable		(fd, -1, 0, -1, &enable_a);
	errs	+= aiss6c_gen_b_enable		(fd, -1, 0, -1, &enable_b);
	errs	+= aiss6c_gen_a_ndiv		(fd, -1, 0, -1, &ndiv_a);
	errs	+= aiss6c_gen_b_ndiv		(fd, -1, 0, -1, &ndiv_b);

	gen			= (src == AISS6C_AI_CLK_SRC_RAG) ? 'A' : 'B';
	enable_gen	= (src == AISS6C_AI_CLK_SRC_RAG) ? enable_a : enable_b;
	ndiv_gen	= (src == AISS6C_AI_CLK_SRC_RAG) ? ndiv_a : ndiv_b;

	if (errs)
	{
		set	= 0;
		printf("FAIL <---  (%d. service errors)\n", __LINE__);
	}
	else if (enable == AISS6C_AI_ENABLE_NO)
	{
		set	= 0;
		printf("0  (%d. Input is disabled.)\n", __LINE__);
	}
	else if (burst == AISS6C_BURST_ENABLE_YES)
	{
		set	= 0;
		printf("Unknown  (%d. Bursting is enabled.)\n", __LINE__);
	}
	else if (src == AISS6C_AI_CLK_SRC_EXT)
	{
		set	= 0;
		printf("Unknown  (%d. The clock source is external.)\n", __LINE__);
	}
	else if (enable_gen == AISS6C_GEN_ENABLE_NO)
	{
		set	= 0;
		printf("0  (%d. Rate Generator %c is disabled.)\n", __LINE__, gen);
	}
	else
	{
		set	= master / ndiv_gen;
		gsc_label_long_comma(set);
		printf(" S/S  (Source = Rate Gen %c, Ref = ", gen);
		gsc_label_long_comma(master);
		printf(", Ndiv = ");
		gsc_label_long_comma(ndiv_gen);
		printf(", Warp Mode %s)\n", warp[0] ? "On" : "Off");
	}

	if (get)
		get[0]	= set;

	return(errs);
}



/******************************************************************************
*
*	Function:	aiss6c_fsamp_ai_report
*
*	Purpose:
*
*		Determine and report the input sample rate for the specified channel.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		chan	The index of the channel of interest.
*
*		get		Store the sample rate here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int	aiss6c_fsamp_ai_report(int fd, int index, int verbose, int chan, s32* get)
{
	s32		bit			= 0x1L << chan;
	char	buf[64];
	int		errs;
	s32		select;
	s32		set			= 0;
	s32		warp		= 0;

	if (verbose)
	{
		sprintf(buf, "AI Channel %d", chan);
		gsc_label_index(buf, index);
	}

	errs	= aiss6c_ai_chan_sel(fd, -1, 0, -1, &select);

	if (select & bit)
	{
		// The channel is enabled.
		errs	+= aiss6c_fsamp_ai_get(fd, 0, &set, &warp);
	}
	else
	{
		// The channel is disabled.
		set	= 0;
	}

	if (get)
		get[0]	= set;

	return(errs);
}



/******************************************************************************
*
*	Function:	aiss6c_fsamp_ai_report_all
*
*	Purpose:
*
*		Determine and report the input sample rate for all channels.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fsamp	The overall sample rate is reported here.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int	aiss6c_fsamp_ai_report_all(int fd, int index, int verbose, s32* fsamp)
{
	s32	chans;
	int	chan;
	int	errs;
	s32	sps;
	s32	total	= 0;

	if (verbose)
		gsc_label_index("AI Sample Rates", index);

	errs	= aiss6c_query(fd, -1, 0, AISS6C_QUERY_CHANNEL_QTY, &chans);

	if (errs)
	{
		if (verbose)
			printf("FAIL <---\n");
	}
	else
	{
		if (verbose)
		{
			printf("\n");
			gsc_label_level_inc();
		}

		for (chan = 0; chan < chans; chan++)
		{
			sps		= 0;
			errs	+= aiss6c_fsamp_ai_report(fd, -1, verbose, chan, &sps);
			total	+= sps;
		}

		if (verbose)
		{
			gsc_label("Overall AI Rate");
			gsc_label_long_comma(total);
			printf(" S/S\n");
			gsc_label_level_dec();
		}
	}

	if (fsamp)
		fsamp[0]	= total;

	return(errs);
}



