// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/18AISS6C/utils/util_config_ai.c $
// $Rev: 53510 $
// $Date: 2023-08-03 10:05:51 -0500 (Thu, 03 Aug 2023) $

// 18AISS6C: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	aiss6c_config_ai
*
*	Purpose:
*
*		Configure the given device's analog input options.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fsamp	This is the desired Fsamp rate, of -1 to use the default.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int aiss6c_config_ai(int fd, int index, int verbose, s32 fsamp)
{
	s32	chans;
	int	errs	= 0;
	s32	ndiv;
	s32	warp;

	if (verbose)
	{
		gsc_label_index("Input Configuration", index);
		printf("\n");
		gsc_label_level_inc();
	}

	errs	+= aiss6c_initialize			(fd, index, verbose);

	// Settings must be applied after the initialization call!

	errs	+= aiss6c_rx_io_mode			(fd, index, verbose, GSC_IO_MODE_PIO,				NULL);
	errs	+= aiss6c_rx_io_overflow		(fd, index, verbose, AISS6C_IO_OVERFLOW_CHECK,		NULL);
	errs	+= aiss6c_rx_io_timeout			(fd, index, verbose, 30,							NULL);
	errs	+= aiss6c_rx_io_underflow		(fd, index, verbose, AISS6C_IO_UNDERFLOW_CHECK,		NULL);

	errs	+= aiss6c_query					(fd, index, verbose, AISS6C_QUERY_CHANNEL_QTY, &chans);
	chans	= (chans >= 6) ? 0x3F : (chans >= 4) ? 0xF : 0x3;
	errs	+= aiss6c_ai_chan_sel			(fd, index, verbose, chans,							NULL);

	errs	+= aiss6c_data_format			(fd, index, verbose, AISS6C_DATA_FORMAT_OFF_BIN,	NULL);
	errs	+= aiss6c_ai_enable				(fd, index, verbose, AISS6C_AI_ENABLE_YES,			NULL);
	errs	+= aiss6c_ai_filter				(fd, index, verbose, AISS6C_AI_FILTER_ENABLE,		NULL);
	errs	+= aiss6c_ai_config				(fd, index, verbose, AISS6C_AI_CONFIG_EXT,			NULL);
	errs	+= aiss6c_ai_clk_src			(fd, index, verbose, AISS6C_AI_CLK_SRC_RAG,			NULL);
	errs	+= aiss6c_ai_thresh_lvl			(fd, index, verbose, 200L * 1024,					NULL);
	errs	+= aiss6c_ai_burst_enable		(fd, index, verbose, AISS6C_BURST_ENABLE_NO,		NULL);
	errs	+= aiss6c_ai_chan_sel			(fd, index, verbose, chans,							NULL);
	errs	+= aiss6c_ai_range				(fd, index, verbose, AISS6C_RANGE_10V,				NULL);

	errs	+= aiss6c_gen_a_enable			(fd, index, verbose, AISS6C_GEN_ENABLE_YES,			NULL);
	errs	+= aiss6c_gen_b_enable			(fd, index, verbose, AISS6C_GEN_ENABLE_NO,			NULL);
	errs	+= aiss6c_fsamp_ai_compute		(fd, index, verbose, fsamp, &ndiv, &warp,			NULL);
	errs	+= aiss6c_gen_a_ndiv			(fd, index, verbose, ndiv,							NULL);
	errs	+= aiss6c_ai_warp_mode			(fd, index, verbose, warp,							NULL);

	errs	+= aiss6c_fsamp_ai_report_all	(fd, index, verbose,								NULL);

	// Settings affecting the sample rate or voltage range must generally
	// be applied before autocalibration.
	// Refer to the device user manual for clarification.
	errs	+= aiss6c_autocal				(fd, index, verbose);

	// Interrupt selections should be applied after autocalibration as the
	// driver overwrites the current selection to detect the Autocalibration
	// Done interrupt. If an application intends to detect the Autocalibration
	// Done interrupt, then the interrupt selection may be made beforehand.
	errs	+= aiss6c_irq_enable			(fd, index, verbose, 0,								NULL);

	// The final step is to clear the buffer. This service clears the FIFO and
	// resets the overflow and underflow status bits.
	errs	+= aiss6c_ai_buf_clear			(fd, index, verbose);

	if (verbose)
		gsc_label_level_dec();

	return(errs);
}


