// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/18AISS6C/driver/ioctl.c $
// $Rev: 53515 $
// $Date: 2023-08-03 10:20:05 -0500 (Thu, 03 Aug 2023) $

// 18AISS6C: Device Driver: source file

#include "main.h"



// macros *********************************************************************

#define	AUTOCAL_START			D20			// BCR
#define	AUTOCAL_PASS			D21			// BCR
#define	INIT_START				D31			// BCR

#define	IRQ_AUTOCAL_DONE		0x00000001L	// PSR



//*****************************************************************************
static void _psr_mod(dev_data_t* dev, u32 value, u32 mask)
{
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_psr_32, value & 0xFFFF, mask | 0xFFFF0000);
}



//*****************************************************************************
static int _query(dev_data_t* dev, s32* arg)
{
	switch (arg[0])
	{
		default:						arg[0]	= AISS6C_IOCTL_QUERY_ERROR;			break;
		case AISS6C_QUERY_AI_FILTER:	arg[0]	= dev->cache.ai_filter;				break;
		case AISS6C_QUERY_AI_RANGE:		arg[0]	= dev->cache.ai_range;				break;
		case AISS6C_QUERY_AUTOCAL_MS:	arg[0]	= dev->cache.autocal_ms;			break;
		case AISS6C_QUERY_CHANNEL_MAX:	arg[0]	= dev->cache.chans_max;				break;
		case AISS6C_QUERY_CHANNEL_QTY:	arg[0]	= dev->cache.chan_qty;				break;
		case AISS6C_QUERY_COUNT:		arg[0]	= AISS6C_IOCTL_QUERY_LAST + 1;		break;
		case AISS6C_QUERY_DEVICE_TYPE:	arg[0]	= dev->board_type;					break;
		case AISS6C_QUERY_FGEN_MAX:		arg[0]	= dev->cache.rate_gen_fgen_max;		break;
		case AISS6C_QUERY_FGEN_MIN:		arg[0]	= dev->cache.rate_gen_fgen_min;		break;
		case AISS6C_QUERY_FIFO_SIZE:	arg[0]	= dev->cache.fifo_size;				break;
		case AISS6C_QUERY_FSAMP_MAX:	arg[0]	= dev->cache.fsamp_max;				break;
		case AISS6C_QUERY_FSAMP_MIN:	arg[0]	= dev->cache.fsamp_min;				break;
		case AISS6C_QUERY_INIT_MS:		arg[0]	= dev->cache.initialize_ms;			break;
		case AISS6C_QUERY_MASTER_CLOCK:	arg[0]	= dev->cache.master_clock;			break;
		case AISS6C_QUERY_NDIV_MAX:		arg[0]	= dev->cache.rate_gen_ndiv_max;		break;
		case AISS6C_QUERY_NDIV_MIN:		arg[0]	= dev->cache.rate_gen_ndiv_min;		break;
		case AISS6C_QUERY_RATE_GEN_QTY:	arg[0]	= dev->cache.rate_gen_qty;			break;
	}

	return(0);
}



//*****************************************************************************
int initialize_ioctl(dev_data_t* dev, void* arg)
{
	int		i;
	long	ms	= dev->cache.initialize_ms + 5000;
	u32		reg;
	int		ret	= 0;
	int		tmp;
	VADDR_T	va	= dev->vaddr.gsc_bcr_32;

	// Initiate initialization.
	os_reg_mem_tx_u32(dev, va, INIT_START);

	// There is no Initialize Done interrupt so we wait manually.
	tmp	= gsc_poll_u32(dev, ms, va, INIT_START, 0);

	if (tmp)
	{
		ret	= ret ? ret : tmp;
		printf(	"%s: INITIALIZATION DID NOT COMPLETE WITHIN %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually check for completion.
	reg	= os_reg_mem_rx_u32(dev, va);

	if (reg & INIT_START)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: INITILIZE STILL ACTIVE AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Initialize the software settings.

	for (i = 0; i < DEV_IO_STREAM_QTY; i++)
	{
		if (dev->io.io_streams[i])
		{
			if (dev->io.io_streams[i]->dev_io_sw_init)
			{
				(dev->io.io_streams[i]->dev_io_sw_init)(dev, dev->io.io_streams[i]);
			}
		}
	}

	return(ret);
}



//*****************************************************************************
static int _autocal_start(dev_data_t* dev, void* arg)
{
	// Initiate autocalibration.
	os_reg_mem_mx_u32(NULL, dev->vaddr.gsc_bcr_32, AUTOCAL_START, AUTOCAL_START);
	return(0);
}



//*****************************************************************************
static int _autocal(dev_data_t* dev, void* arg)
{
	u32			mask;
	long		ms		= dev->cache.autocal_ms + 5000;
	u32			reg;
	int			ret;
	os_sem_t	sem;
	int			tmp;
	VADDR_T		va		= dev->vaddr.gsc_bcr_32;
	u32			value;
	gsc_wait_t	wt;

	// Safely select the Autocal Done interrupt.
	mask	= IRQ_AUTOCAL_DONE;
	value	= IRQ_AUTOCAL_DONE;
	_psr_mod(dev, value, mask);

	// Wait for the local interrupt.
	os_sem_create(&sem);	// dummy, required for wait operations.
	memset(&wt, 0, sizeof(wt));
	wt.flags		= GSC_WAIT_FLAG_INTERNAL;
	wt.gsc			= AISS6C_WAIT_GSC_AUTOCAL_DONE;
	wt.timeout_ms	= ms;
	ret				= gsc_wait_event(dev, &wt, _autocal_start, NULL, &sem);
	os_sem_destroy(&sem);

	if (wt.flags & GSC_WAIT_FLAG_TIMEOUT)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATE DONE IRQ TIMED OUT AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually wait for completion in case something terminates our wait early.
	tmp	= gsc_poll_u32(dev, ms, va, AUTOCAL_START, 0);

	if (tmp)
	{
		ms	*= 2;
		ret	= ret ? ret : tmp;
		printf(	"%s: AUTOCALIBRATION DID NOT COMPLETE WITHIN %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually check for completion as the IRQ wait may have ended early.
	reg	= os_reg_mem_rx_u32(dev, va);

	if (reg & AUTOCAL_START)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATION STILL ACTIVE AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Final results.
	reg	= os_reg_mem_rx_u32(dev, va);

	if ((reg & AUTOCAL_PASS) == 0)
	{
		ret	= ret ? ret : -EIO;
		printf(	"%s: AUTOCALIBRATION FAILED (%ld ms).\n",
				dev->model,
				ms);
	}

	if (ret == 0)
	{
		// Wait for settling.
		os_time_sleep_ms(100);
	}

	return(ret);
}



//*****************************************************************************
static int _autocal_status(dev_data_t* dev, s32* arg)
{
	u32	reg;

	reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bcr_32);

	if (reg & AUTOCAL_START)
		arg[0]	= AISS6C_AUTOCAL_STATUS_ACTIVE;
	else if (reg & AUTOCAL_PASS)
		arg[0]	= AISS6C_AUTOCAL_STATUS_PASS;
	else
		arg[0]	= AISS6C_AUTOCAL_STATUS_FAIL;

	return(0);
}



//*****************************************************************************
static int _ai_buf_clear(dev_data_t* dev, void* arg)
{
	#define	CLEAR	D13

	int		i;
	u32		reg;
	int		ret	= 0;
	VADDR_T	va	= dev->vaddr.gsc_bcr_32;

	// Clear the buffer.
	os_reg_mem_mx_u32(dev, va, CLEAR, CLEAR);

	// Wait for the bit to clear.

	for (i = 0;; i++)
	{
		reg	= os_reg_mem_rx_u32(dev, va);

		if ((reg & CLEAR) == 0)
			break;

		if (i >= 250)
		{
			ret	= -EINVAL;
			printf(	"%s: The analog input buffer took too long to clear.\n",
					DEV_NAME);
			break;
		}
	}

	// Clear the Overflow status bit.
	// This status bit is cleared by the Clear Buffer operation.

	// Clear the Underflow status bit.
	// This status bit is cleared by the Clear Buffer operation.

	return(ret);
}



//*****************************************************************************
static int _ai_buf_level(dev_data_t* dev, s32* arg)
{
	arg[0]	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ibsr_32);
	arg[0]	&= 0x7FFFF;
	return(0);
}



//*****************************************************************************
static int _ai_buf_overflow(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_BUF_OVERFLOW_NO,
		AISS6C_BUF_OVERFLOW_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 15, 15);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_underflow(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_BUF_UNDERFLOW_NO,
		AISS6C_BUF_UNDERFLOW_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 16, 16);
	return(ret);
}



//*****************************************************************************
static int _ai_burst_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_BURST_ENABLE_NO,
		AISS6C_BURST_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 9, 9);
	return(ret);
}



//*****************************************************************************
static int _ai_burst_size(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFFFF, dev->vaddr.gsc_ibbsr_32, 23, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_burst_status(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS6C_BURST_STATUS_READY,
		AISS6C_BURST_STATUS_BUSY,
		-1	// terminate list
	};

	int		ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 10, 10);
	return(ret);
}



//*****************************************************************************
static int _ai_chan_sel(dev_data_t* dev, void* arg)
{
	int	ret;

	switch (dev->cache.chan_qty)
	{
		default:

			ret	= -EINVAL;
			break;

		case 2:

			ret	= gsc_s32_range_reg(dev, arg, 0, 0x03, dev->vaddr.gsc_icr_32, 5, 0);
			break;

		case 4:

			ret	= gsc_s32_range_reg(dev, arg, 0, 0x0F, dev->vaddr.gsc_icr_32, 5, 0);
			break;

		case 6:

			ret	= gsc_s32_range_reg(dev, arg, 0, 0x3F, dev->vaddr.gsc_icr_32, 5, 0);
			break;
	}

	return(ret);
}



//*****************************************************************************
static int _ai_clk_src(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_AI_CLK_SRC_RAG,
		AISS6C_AI_CLK_SRC_RBG,
		AISS6C_AI_CLK_SRC_EXT,
		-1	// terminate list
	};

	int		ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_icr_32, 9, 8);
	return(ret);
}



//*****************************************************************************
static int _ai_config(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_AI_CONFIG_EXT,
		AISS6C_AI_CONFIG_ZERO,
		AISS6C_AI_CONFIG_VREF,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_icr_32, 17, 16);
	return(ret);
}



//*****************************************************************************
static int _ai_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_AI_ENABLE_NO,
		AISS6C_AI_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 12, 12);
	return(ret);
}



//*****************************************************************************
static int _ai_filter(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_AI_FILTER_DISABLE,
		AISS6C_AI_FILTER_ENABLE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_icr_32, 24, 24);
	return(ret);
}



//*****************************************************************************
static int _ai_range(dev_data_t* dev, s32* arg)
{
	u32	reg;
	int	ret	= 0;

	switch (dev->cache.ai_range)
	{
		default:

			ret	= -EINVAL;
			break;

		case AISS6C_AI_RANGE_20_10:

			switch (arg[0])
			{
				default:

					ret	= -EINVAL;
					break;

				case -1:

					reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_icr_32);
					arg[0]	= (reg & D20)
							? AISS6C_RANGE_20V
							: AISS6C_RANGE_10V;
					break;

				case AISS6C_RANGE_10V:

					os_reg_mem_mx_u32(dev, dev->vaddr.gsc_icr_32, 0, D20);
					break;

				case AISS6C_RANGE_20V:

					os_reg_mem_mx_u32(dev, dev->vaddr.gsc_icr_32, D20, D20);
					break;
			}

			break;

		case AISS6C_AI_RANGE_10_5:

			switch (arg[0])
			{
				default:

					ret	= -EINVAL;
					break;

				case -1:

					reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_icr_32);
					arg[0]	= (reg & D20)
							? AISS6C_RANGE_10V
							: AISS6C_RANGE_5V;
					break;

				case AISS6C_RANGE_5V:

					os_reg_mem_mx_u32(dev, dev->vaddr.gsc_icr_32, 0, D20);
					break;

				case AISS6C_RANGE_10V:

					os_reg_mem_mx_u32(dev, dev->vaddr.gsc_icr_32, D20, D20);
					break;
			}
	}

	return(ret);
}



//*****************************************************************************
static int _ai_sw_clock(dev_data_t* dev, void* arg)
{
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bcr_32, D8, D8);
	return(0);
}



//*****************************************************************************
static int _ai_sw_trigger(dev_data_t* dev, void* arg)
{
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bcr_32, D11, D11);
	return(0);
}



//*****************************************************************************
static int _ai_thresh_lvl(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0x7FFFF, dev->vaddr.gsc_ibtr_32, 18, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_thresh_sts(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS6C_THRESH_STS_CLEAR,
		AISS6C_THRESH_STS_SET,
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_ibtr_32, 20, 20);
	return(ret);
}



//*****************************************************************************
static int _ai_trig_src(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_AI_TRIG_SRC_RBG,
		AISS6C_AI_TRIG_SRC_RAG,
		AISS6C_AI_TRIG_SRC_EXT,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_icr_32, 13, 12);
	return(ret);
}



//*****************************************************************************
static int _ai_warp_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_AI_WARP_MODE_DISABLE,
		AISS6C_AI_WARP_MODE_ENABLE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 22, 22);
	return(ret);
}



//*****************************************************************************
static int _aux_clk_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_AUX_CLK_MODE_DISABLE,
		AISS6C_AUX_CLK_MODE_IN,
		AISS6C_AUX_CLK_MODE_OUT,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_asior_32, 1, 0);
	return(ret);
}



//*****************************************************************************
static int _aux_in_pol(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_AUX_IN_POL_L2H,
		AISS6C_AUX_IN_POL_H2L,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_asior_32, 8, 8);
	return(ret);
}



//*****************************************************************************
static int _aux_noise(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_AUX_NOISE_LOW,
		AISS6C_AUX_NOISE_HIGH,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_asior_32, 10, 10);
	return(ret);
}



//*****************************************************************************
static int _aux_out_pol(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_AUX_OUT_POL_L2H,
		AISS6C_AUX_OUT_POL_H2L,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_asior_32, 9, 9);
	return(ret);
}



//*****************************************************************************
static int _aux_sync_mode(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS6C_AUX_SYNC_MODE_DISABLE,
		AISS6C_AUX_SYNC_MODE_IN,
		AISS6C_AUX_SYNC_MODE_OUT
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_asior_32, 3, 2);
	return(ret);
}



//*****************************************************************************
static int _data_format(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_DATA_FORMAT_2S_COMP,
		AISS6C_DATA_FORMAT_OFF_BIN,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 17, 17);
	return(ret);
}



//*****************************************************************************
static int _dio_dir_out(dev_data_t* dev, s32* arg)
{
	int	ret	= 0;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case AISS6C_DIO_DIR_OUT_NONE:

			os_reg_mem_mx_u32(dev, dev->vaddr.gsc_diopr_32, 0, D15 | D7);
			break;

		case AISS6C_DIO_DIR_OUT_D0_D3:

			os_reg_mem_mx_u32(dev, dev->vaddr.gsc_diopr_32, D7, D15 | D7);
			break;

		case AISS6C_DIO_DIR_OUT_D4_D7:

			os_reg_mem_mx_u32(dev, dev->vaddr.gsc_diopr_32, D15, D15 | D7);
			break;

		case AISS6C_DIO_DIR_OUT_D0_D7:

			os_reg_mem_mx_u32(dev, dev->vaddr.gsc_diopr_32, D15 | D7, D15 | D7);
			break;
	}

	return(ret);
}



//*****************************************************************************
static int _dio_read(dev_data_t* dev, s32* arg)
{
	arg[0]	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_diopr_32);
	arg[0]	= (arg[0] & 0x0F) | ((arg[0] >> 4) & 0xF0);
	return(0);
}



//*****************************************************************************
static int _dio_write(dev_data_t* dev, s32* arg)
{
	int	ret	= 0;
	s32	v;

	if (arg[0] == -1)
	{
		ret	= _dio_read(dev, arg);
	}
	else if ((arg[0] >= 0) && (arg[0] <= 0xFF))
	{
		v	= (arg[0] & 0x0F) | ((arg[0] & 0xF0) << 4);
		os_reg_mem_mx_u32(dev, dev->vaddr.gsc_diopr_32, v, 0x0F0F);
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static int _gen_a_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_GEN_ENABLE_NO,
		AISS6C_GEN_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 18, 18);
	return(ret);
}



//*****************************************************************************
static int _gen_a_ndiv(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(
			dev,
			arg,
			dev->cache.rate_gen_ndiv_min,
			dev->cache.rate_gen_ndiv_max,
			dev->vaddr.gsc_rgar_32,
			23,
			0);
	return(ret);
}



//*****************************************************************************
static int _gen_b_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS6C_GEN_ENABLE_NO,
		AISS6C_GEN_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 19, 19);
	return(ret);
}



//*****************************************************************************
static int _gen_b_ndiv(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(
			dev,
			arg,
			dev->cache.rate_gen_ndiv_min,
			dev->cache.rate_gen_ndiv_max,
			dev->vaddr.gsc_rgbr_32,
			23,
			0);
	return(ret);
}



//*****************************************************************************
static int _irq_enable(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFF, dev->vaddr.gsc_psr_32, 7, 0);
	return(ret);
}



//*****************************************************************************
static int _rx_io_abort(dev_data_t* dev, s32* arg)
{
	arg[0]	= gsc_read_abort_active_xfer(dev, &dev->io.rx);
	return(0);
}



//*****************************************************************************
static int _rx_io_mode(dev_data_t* dev, s32* arg)
{
	static const s32	list[]	=
	{
		GSC_IO_MODE_PIO,
		GSC_IO_MODE_BMDMA,
		GSC_IO_MODE_DMDMA,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.rx.io_mode);
	return(ret);
}



//*****************************************************************************
static int _rx_io_overflow(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS6C_IO_OVERFLOW_IGNORE,
		AISS6C_IO_OVERFLOW_CHECK,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, options, &dev->io.rx.overflow_check);
	return(ret);
}



//*****************************************************************************
static int _rx_io_timeout(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_var(
			arg,
			AISS6C_IO_TIMEOUT_MIN,
			AISS6C_IO_TIMEOUT_INFINITE,
			&dev->io.rx.timeout_s);
	return(ret);
}



//*****************************************************************************
static int _rx_io_underflow(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS6C_IO_UNDERFLOW_IGNORE,
		AISS6C_IO_UNDERFLOW_CHECK,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, options, &dev->io.rx.underflow_check);
	return(ret);
}



// variables ******************************************************************

const gsc_ioctl_t	dev_ioctl_list[]	=
{
	{ AISS6C_IOCTL_REG_READ,			(void*) gsc_reg_read_ioctl		},
	{ AISS6C_IOCTL_REG_WRITE,			(void*) gsc_reg_write_ioctl		},
	{ AISS6C_IOCTL_REG_MOD,				(void*) gsc_reg_mod_ioctl		},
	{ AISS6C_IOCTL_QUERY,				(void*) _query					},
	{ AISS6C_IOCTL_INITIALIZE,			(void*) initialize_ioctl		},
	{ AISS6C_IOCTL_AUTOCAL,				(void*) _autocal				},
	{ AISS6C_IOCTL_AUTOCAL_STATUS,		(void*) _autocal_status			},
	{ AISS6C_IOCTL_AI_BUF_CLEAR,		(void*) _ai_buf_clear			},
	{ AISS6C_IOCTL_AI_BUF_LEVEL,		(void*) _ai_buf_level			},
	{ AISS6C_IOCTL_AI_BUF_OVERFLOW,		(void*) _ai_buf_overflow		},
	{ AISS6C_IOCTL_AI_BUF_UNDERFLOW,	(void*) _ai_buf_underflow		},
	{ AISS6C_IOCTL_AI_BURST_ENABLE,		(void*) _ai_burst_enable		},
	{ AISS6C_IOCTL_AI_BURST_SIZE,		(void*) _ai_burst_size			},
	{ AISS6C_IOCTL_AI_BURST_STATUS,		(void*) _ai_burst_status		},
	{ AISS6C_IOCTL_AI_CHAN_SEL,			(void*) _ai_chan_sel			},
	{ AISS6C_IOCTL_AI_CLK_SRC,			(void*) _ai_clk_src				},
	{ AISS6C_IOCTL_AI_CONFIG,			(void*) _ai_config				},
	{ AISS6C_IOCTL_AI_ENABLE,			(void*) _ai_enable				},
	{ AISS6C_IOCTL_AI_FILTER,			(void*) _ai_filter				},
	{ AISS6C_IOCTL_AI_RANGE,			(void*) _ai_range				},
	{ AISS6C_IOCTL_AI_SW_CLOCK,			(void*) _ai_sw_clock			},
	{ AISS6C_IOCTL_AI_SW_TRIGGER,		(void*) _ai_sw_trigger			},
	{ AISS6C_IOCTL_AI_THRESH_LVL,		(void*) _ai_thresh_lvl			},
	{ AISS6C_IOCTL_AI_THRESH_STS,		(void*) _ai_thresh_sts			},
	{ AISS6C_IOCTL_AI_TRIG_SRC,			(void*) _ai_trig_src			},
	{ AISS6C_IOCTL_AI_WARP_MODE,		(void*) _ai_warp_mode			},
	{ AISS6C_IOCTL_AUX_CLK_MODE,		(void*) _aux_clk_mode			},
	{ AISS6C_IOCTL_AUX_IN_POL,			(void*) _aux_in_pol				},
	{ AISS6C_IOCTL_AUX_NOISE,			(void*) _aux_noise				},
	{ AISS6C_IOCTL_AUX_OUT_POL,			(void*) _aux_out_pol			},
	{ AISS6C_IOCTL_AUX_SYNC_MODE,		(void*) _aux_sync_mode			},
	{ AISS6C_IOCTL_DATA_FORMAT,			(void*) _data_format			},
	{ AISS6C_IOCTL_DIO_DIR_OUT,			(void*) _dio_dir_out			},
	{ AISS6C_IOCTL_DIO_READ,			(void*) _dio_read				},
	{ AISS6C_IOCTL_DIO_WRITE,			(void*) _dio_write				},
	{ AISS6C_IOCTL_GEN_A_ENABLE,		(void*) _gen_a_enable			},
	{ AISS6C_IOCTL_GEN_A_NDIV,			(void*) _gen_a_ndiv				},
	{ AISS6C_IOCTL_GEN_B_ENABLE,		(void*) _gen_b_enable			},
	{ AISS6C_IOCTL_GEN_B_NDIV,			(void*) _gen_b_ndiv				},
	{ AISS6C_IOCTL_IRQ_ENABLE,			(void*) _irq_enable				},
	{ AISS6C_IOCTL_RX_IO_ABORT,			(void*) _rx_io_abort			},
	{ AISS6C_IOCTL_RX_IO_MODE,			(void*) _rx_io_mode				},
	{ AISS6C_IOCTL_RX_IO_OVERFLOW,		(void*) _rx_io_overflow			},
	{ AISS6C_IOCTL_RX_IO_TIMEOUT,		(void*) _rx_io_timeout			},
	{ AISS6C_IOCTL_RX_IO_UNDERFLOW,		(void*) _rx_io_underflow		},
	{ AISS6C_IOCTL_WAIT_EVENT,			(void*) gsc_wait_event_ioctl	},
	{ AISS6C_IOCTL_WAIT_CANCEL,			(void*) gsc_wait_cancel_ioctl	},
	{ AISS6C_IOCTL_WAIT_STATUS,			(void*) gsc_wait_status_ioctl	},
	{ -1, NULL }
};


