// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/18AISS6C/driver/device.c $
// $Rev: 53515 $
// $Date: 2023-08-03 10:20:05 -0500 (Thu, 03 Aug 2023) $

// 18AISS6C: Device Driver: source file

#include "main.h"



// variables ******************************************************************

const gsc_dev_id_t	dev_id_list[]	=
{
	// model		Vendor	Device	SubVen	SubDev	type

	{ "18AISS6C",	0x10B5, 0x9056, 0x10B5, 0x3467,	GSC_DEV_TYPE_18AISS6C	},
	{ NULL }
};



//*****************************************************************************
static void _channels_compute(dev_data_t* dev)
{
	switch (dev->cache.gsc_acr_32 & 0x30000)
	{
		default:
		case 0x00000:	dev->cache.chan_qty	= 2;	break;
		case 0x10000:	dev->cache.chan_qty	= 4;	break;
		case 0x20000:	dev->cache.chan_qty	= 6;	break;
	}

	dev->cache.chans_max	= 6;
}



//*****************************************************************************
static void _filter_compute(dev_data_t* dev)
{
	switch (dev->cache.gsc_acr_32 & 0xC0000)
	{
		default:
		case 0x00000:	dev->cache.ai_filter	= AISS6C_AI_FILTER_F1;	break;
	}
}



//*****************************************************************************
static void _master_clock_compute(dev_data_t* dev)
{
	switch (dev->cache.gsc_acr_32 & 0x300000)
	{
		default:
		case 0x000000:	dev->cache.master_clock	= 45000000L;	break;
	}
}



//*****************************************************************************
static void _range_compute(dev_data_t* dev)
{
	switch (dev->cache.gsc_acr_32 & 0xC00000)
	{
		default:
		case 0x000000:	dev->cache.ai_range	= AISS6C_AI_RANGE_20_10;	break;
		case 0x400000:	dev->cache.ai_range	= AISS6C_AI_RANGE_10_5;		break;
	}
}



/******************************************************************************
*
*	Function:	dev_device_create
*
*	Purpose:
*
*		Do everything needed to setup and use the given device.
*
*	Arguments:
*
*		dev		The structure to initialize.
*
*	Returned:
*
*		0		All is well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int dev_device_create(dev_data_t* dev)
{
	static const gsc_bar_maps_t	bar_map	=
	{
		{
			// mem	io	rw
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 0: PLX registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 1: PLX registers, I/O mapped
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RW },	// BAR 2: GSC registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 3: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 4: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO }	// BAR 5: unused
		}
	};

	u32	dma;
	int	ret;

	for (;;)	// A convenience loop.
	{
		// Verify some macro contents.
		ret	= gsc_macro_test_base_name(AISS6C_BASE_NAME);
		if (ret)	break;

		ret	= gsc_macro_test_model();
		if (ret)	break;

		// PCI setup.
		ret	= os_pci_dev_enable(&dev->pci);
		if (ret)	break;

		ret	= os_pci_master_set(&dev->pci);
		if (ret)	break;

		// Control ISR access to the device and data structure.
		ret	= os_spinlock_create(&dev->spinlock);
		if (ret)	break;

		// Control access to the device and data structure.
		ret	= os_sem_create(&dev->sem);
		if (ret)	break;

		// Access the BAR regions.
		ret	= gsc_bar_create(dev, &dev->bar, &bar_map);
		if (ret)	break;

		// Firmware access.
		dev->vaddr.gsc_acr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS6C_GSC_ACR));
		dev->vaddr.gsc_asior_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS6C_GSC_ASIOR));
		dev->vaddr.gsc_bcr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS6C_GSC_BCR));
		dev->vaddr.gsc_diopr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS6C_GSC_DIOPR));
		dev->vaddr.gsc_ibbsr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS6C_GSC_IBBSR));
		dev->vaddr.gsc_ibsr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS6C_GSC_IBSR));
		dev->vaddr.gsc_ibtr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS6C_GSC_IBTR));
		dev->vaddr.gsc_icr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS6C_GSC_ICR));
		dev->vaddr.gsc_idbr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS6C_GSC_IDBR));
		dev->vaddr.gsc_psr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS6C_GSC_PSR));
		dev->vaddr.gsc_rgar_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS6C_GSC_RGAR));
		dev->vaddr.gsc_rgbr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS6C_GSC_RGBR));

		// Data cache initialization.
		dev->cache.gsc_acr_32			= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_acr_32);

		dev->cache.autocal_ms			= 7000;
		dev->cache.fifo_size			= _256K;
		dev->cache.fsamp_max			= 555556L;
		dev->cache.initialize_ms		= 1;
		dev->cache.rate_gen_fgen_max	= 550000L;
		dev->cache.rate_gen_ndiv_max	= 0xFFFFFF;
		dev->cache.rate_gen_ndiv_min	= 81;
		dev->cache.rate_gen_qty			= 2;

		_channels_compute(dev);
		_master_clock_compute(dev);
		_filter_compute(dev);
		_range_compute(dev);

		dev->cache.fsamp_min			= dev->cache.master_clock / 0xFFFFFF;
		dev->cache.rate_gen_fgen_min	= dev->cache.master_clock / 0xFFFFFF;

		// Initialize additional resources.
		ret	= dev_irq_create(dev);
		if (ret)	break;

		ret	= dev_io_create(dev);
		if (ret)	break;

		dma	= GSC_DMA_SEL_STATIC
			| GSC_DMA_CAP_BMDMA_READ
			| GSC_DMA_CAP_DMDMA_READ;
		ret	= gsc_dma_create(dev, dma, dma);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	dev_device_destroy
*
*	Purpose:
*
*		Do everything needed to release the referenced device.
*
*	Arguments:
*
*		dev		The partial data for the device of interest.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void dev_device_destroy(dev_data_t* dev)
{
	if (dev)
	{
		gsc_dma_destroy(dev);
		dev_io_destroy(dev);
		dev_irq_destroy(dev);
		gsc_bar_destroy(&dev->bar);
		os_sem_destroy(&dev->sem);
		os_spinlock_destroy(&dev->spinlock);
		os_pci_master_clear(&dev->pci);
		os_pci_dev_disable(&dev->pci);
	}
}


