// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AO16/16AO16_Linux_2.x.x.x_DN/utils/reg.c $
// $Rev: 56206 $
// $Date: 2025-02-05 08:48:14 -0600 (Wed, 05 Feb 2025) $

// 16AO16: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	_GSC_REG(a)				"GSC " #a, AO16_GSC_##a, 0, 0
#define	WIDTH					19

#define	SHOW_FIELD(high,low,eol,list,name)	\
			gsc_reg_field_show(	\
				width + 10,		\
				WIDTH,			\
				value,			\
				(high),			\
				(low),			\
				(eol),			\
				(void*)(list),	\
				(name))



//*****************************************************************************
static int _bcr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	filter[]	=
	{
		"No Filter",
		"No Filter",
		"Filter-A",
		"Filter-B"
	};

	static const char*	irq[]	=
	{
		"Initialize Done",
		"Autocalibration Done",
		"Output Buffer Empty",
		"Output Buffer < 1/4 Full",
		"Output Buffer 3/4 Full",
		"Burst Trigger Ready",
		"Load Ready (low-to-high transition)",
		"End Load Ready (high-to-low transition)"
	};

	static const char*	range_high[]	=
	{
		"RESERVED",
		"+-5 Volts",
		"+-10 Volts",
		"+-20 Volts"
	};

	static const char*	range_low[]	=
	{
		"+-1.25 Volts",
		"+-2.5 Volts",
		"+-5 Volts",
		"+-10 Volts"
	};

	static const char*	disable[]	= { "Disable",			"Enable"		};
	static const char*	enable[]	= { "Enable",			"Disable"		};
	static const char*	hi_curent[]	= { "+-1.5 Volts",		"+-2.5 Volts"	};
	static const char*	hi_level[]	= { "+-5 Volts",		"+-10 Volts"	};
	static const char*	idle[]		= { "Idle",				"Active"		};
	static const char*	invert[]	= { "Uninverted",		"Inverted"		};
	static const char*	no[]		= { "No",				"Yes"			};
	static const char*	pass[]		= { "Pass",				"Fail"			};
	static const char*	seq[]		= { "Sequential",		"Simultaneous"	};
	static const char*	ttl[]		= { "TTL",				"LVDS"			};
	static const char*	twos[]		= { "Twos Compliment",	"Offset Binary"	};

	s32	capacity;
	int	errs	= 0;
	s32	inv;
	s32	model;
	s32	range;
	s32	watchdog;

	errs	+= ao16_query(fd, -1, 0, AO16_QUERY_MODEL, &model);
	errs	+= ao16_query(fd, -1, 0, AO16_QUERY_OUTPUT_CAPACITY, &capacity);
	errs	+= ao16_query(fd, -1, 0, AO16_QUERY_CABLE_INVERT_4, &inv);
	errs	+= ao16_query(fd, -1, 0, AO16_QUERY_VOLT_RANGE, &range);
	errs	+= ao16_query(fd, -1, 0, AO16_QUERY_WATCHDOG, &watchdog);

	SHOW_FIELD( 0,  0, 1, disable,	"Burst Enabled"		);
	SHOW_FIELD( 1,  1, 1, no,		"Burst Ready"		);
	SHOW_FIELD( 2,  2, 1, idle,		"Burst Trigger"		);
	SHOW_FIELD( 3,  3, 1, disable,	"Ground Sense"		);
	SHOW_FIELD( 4,  4, 1, twos,		"Data Format"		);
	SHOW_FIELD( 5,  5, 1, ttl,		"Diff. Sync I/O"	);
	SHOW_FIELD( 6,  6, 1, enable,	"Ext Burst Trig"	);
	SHOW_FIELD( 7,  7, 1, seq,		"Simul. Output"		);
	SHOW_FIELD(10,  8, 1, irq,		"IRQ"				);
	SHOW_FIELD(11, 11, 1, idle,		"IRQ Request Flag"	);
	SHOW_FIELD(12, 12, 1, NULL,		"Reserved"			);
	SHOW_FIELD(13, 13, 1, idle,		"Autocal"			);
	SHOW_FIELD(14, 14, 1, pass,		"Autocal Status"	);
	SHOW_FIELD(15, 15, 1, idle,		"Initialize"		);

	if (model == AO16_MODEL_16AO16)
	{
		if (range == AO16_VOLT_RANGE_LOW)
			SHOW_FIELD(17, 16, 1, range_low,	"Output Range"	);
		else
			SHOW_FIELD(17, 16, 1, range_high,	"Output Range"	);

		if (inv)
		{
			SHOW_FIELD(18, 18, 1, invert,	"Invert Trigger Input"	);
			SHOW_FIELD(19, 19, 1, invert,	"Invert Trigger Output"	);
			SHOW_FIELD(20, 20, 1, invert,	"Invert DAC Clock Out"	);
			SHOW_FIELD(21, 21, 1, invert,	"Invert Clock I/O"		);
		}
		else
		{
			SHOW_FIELD(21, 18, 1, NULL,		"Reserved"	);
		}

		if (watchdog)
		{
			SHOW_FIELD(22, 22, 1, disable,	"Watchdog Enable"	);
			SHOW_FIELD(23, 23, 1, disable,	"Watchdog Output"	);
		}
		else
		{
			SHOW_FIELD(23, 22, 1, NULL,	"Reserved"	);
		}

		SHOW_FIELD(31, 24, 1, NULL,	"Reserved"	);
	}
	else
	{
		if (capacity == AO16_OUTPUT_CAPACITY_HI_LEVEL)
			SHOW_FIELD(16, 16, 1, hi_level,		"Output Range"	);
		else
			SHOW_FIELD(16, 16, 1, hi_curent,	"Output Range"	);

		SHOW_FIELD(17, 17, 1, NULL,		"Reserved"		);
		SHOW_FIELD(19, 18, 1, filter,	"Output Filter"	);
		SHOW_FIELD(31, 20, 1, NULL,		"Reserved"		);
	}

	return(errs);
}



//*****************************************************************************
static int _csr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	disable[]	= { "Disable",	"Enable"	};

	SHOW_FIELD( 0,  0, 1, disable,	"Channel 0"		);
	SHOW_FIELD( 1,  1, 1, disable,	"Channel 1"		);
	SHOW_FIELD( 2,  2, 1, disable,	"Channel 2"		);
	SHOW_FIELD( 3,  3, 1, disable,	"Channel 3"		);
	SHOW_FIELD( 4,  4, 1, disable,	"Channel 4"		);
	SHOW_FIELD( 5,  5, 1, disable,	"Channel 5"		);
	SHOW_FIELD( 6,  6, 1, disable,	"Channel 6"		);
	SHOW_FIELD( 7,  7, 1, disable,	"Channel 7"		);
	SHOW_FIELD( 8,  8, 1, disable,	"Channel 8"		);
	SHOW_FIELD( 9,  9, 1, disable,	"Channel 9"		);
	SHOW_FIELD(10, 10, 1, disable,	"Channel 10"	);
	SHOW_FIELD(11, 11, 1, disable,	"Channel 11"	);
	SHOW_FIELD(12, 12, 1, disable,	"Channel 12"	);
	SHOW_FIELD(13, 13, 1, disable,	"Channel 13"	);
	SHOW_FIELD(14, 14, 1, disable,	"Channel 14"	);
	SHOW_FIELD(15, 15, 1, disable,	"Channel 15"	);
	SHOW_FIELD(31, 16, 1, NULL,		"Reserved"		);

	return(0);
}



//*****************************************************************************
static int _srr_decode(int fd, int supported, u32 value, int width)
{
	long	v;

	SHOW_FIELD(17,  0, 0, NULL,	"Nrate"		);
	v	= value & 0x3FFFF;
	printf("Fref / %ld\n", v);

	SHOW_FIELD(31, 18, 1, NULL,	"Reserved"	);

	return(0);
}



//*****************************************************************************
static int _bor_decode(int fd, int supported, u32 value, int width)
{
	static const char*	size[]	=
	{
		"8 samples",
		"16 samples",
		"32 samples",
		"64 samples",
		"128 samples",
		"256 samples",
		"512 samples",
		"1024 samples",
		"2048 samples",
		"4096 samples",
		"8192 samples",
		"16384 samples",
		"32768 samples",
		"65536 samples",
		"131072 samples",
		"262144 samples"
	};

	static const char*	clock[]		= { "Rate Gen",	"Extern/SW"	};
	static const char*	disable[]	= { "Disable",	"Enable"	};
	static const char*	idle[]		= { "Idle",		"Active"	};
	static const char*	no[]		= { "No",		"Yes"		};

	int	errs	= 0;
	s32	isolate;
	s32	model;

	errs	+= ao16_query(fd, -1, 0, AO16_QUERY_MODEL, &model);
	errs	+= ao16_query(fd, -1, 0, AO16_QUERY_CABLE_PASSIVE_3, &isolate);

	SHOW_FIELD( 3,  0, 1, size,		"Buffer Size"		);
	SHOW_FIELD( 4,  4, 1, clock,	"External. Clock"	);
	SHOW_FIELD( 5,  5, 1, disable,	"Enable Clock"		);
	SHOW_FIELD( 6,  6, 1, no,		"Clock Ready"		);
	SHOW_FIELD( 7,  7, 1, idle,		"SW Clock"			);
	SHOW_FIELD( 8,  8, 1, no,		"Circ. Buffer"		);
	SHOW_FIELD( 9,  9, 1, idle,		"Load Request"		);
	SHOW_FIELD(10, 10, 1, no,		"Load Ready"		);
	SHOW_FIELD(11, 11, 1, idle,		"Clear Buffer"		);
	SHOW_FIELD(12, 12, 1, no,		"Buffer Empty"		);
	SHOW_FIELD(13, 13, 1, no,		"Buffer < 1/4 Full"	);
	SHOW_FIELD(14, 14, 1, no,		"Buffer 3/4 Full"	);
	SHOW_FIELD(15, 15, 1, no,		"Buffer Full"		);
	SHOW_FIELD(16, 16, 1, no,		"Buffer Overflow"	);
	SHOW_FIELD(17, 17, 1, no,		"Frame Overflow"	);

	if (model == AO16_MODEL_16AO16)
	{
		if (isolate)
		{
			SHOW_FIELD(18, 18, 1, no,	"isolate Trigger Out"	);
			SHOW_FIELD(19, 19, 1, no,	"Isolate DAC Clock Out"	);
			SHOW_FIELD(20, 20, 1, no,	"Isolate Clock I/O"		);
			SHOW_FIELD(31, 21, 1, NULL,	"Reserved"				);
		}
		else
		{
			SHOW_FIELD(31, 18, 1, NULL,	"Reserved"	);
		}
	}
	else
	{
		SHOW_FIELD(31, 18, 1, NULL,		"Reserved"	);
	}

	return(errs);
}



//*****************************************************************************
static int _for_decode(int fd, int supported, u32 value, int width)
{
	static const char*	chans[]	=
	{
		"RESERVED",
		"8 Channels",
		"12 Channels",
		"16 Channels"
	};

	static const char*	filter_1[]	=
	{
		"None (>300 KHz)",
		"10 KHz",
		"100 KHz",
		"RESERVED"
	};

	static const char*	filter_2[]	=
	{
		"F1 Filters",
		"F2 Filters",
		"F3 Filters",
		"F4 Filters"
	};

	static const char*	capacity[]	= { "High Current",	"High Level"	};
	static const char*	master[]	= { "45MHz",		"49.152MHz"		};
	static const char*	no[]		= { "No",			"Yes"			};
	static const char*	outputs[]	= { "Single Ended",	"Differential"	};
	static const char*	product[]	= { "16AO16",		"16AO16FLV"		};

	int		errs;
	long	fw		= value & 0xFFF;
	s32		model;

	errs	= ao16_query(fd, -1, 0, AO16_QUERY_MODEL, &model);

	SHOW_FIELD(11,  0, 0, NULL,		"Firmware Revision"	);
	printf("%03lX\n", fw);

	SHOW_FIELD(15, 12, 1, NULL,		"Reserved"			);
	SHOW_FIELD(17, 16, 1, chans,	"Output Channels"	);

	if (model == AO16_MODEL_16AO16)
	{
		SHOW_FIELD(19, 18, 1, filter_1,	"Output Filter"		);
		SHOW_FIELD(20, 20, 1, outputs,	"Output Mode"		);
		SHOW_FIELD(21, 21, 1, master,	"Master Oscillator"	);

		if (fw >= 0x5)
			SHOW_FIELD(22, 22, 1, product,	"Model"			);
		else
			SHOW_FIELD(22, 22, 1, NULL,		"Reserved"		);

		SHOW_FIELD(23, 23, 1, no,		"High-V Outputs"	);
		SHOW_FIELD(31, 24, 1, NULL,		"Reserved"			);
	}
	else
	{
		SHOW_FIELD(19, 18, 1, filter_2,	"Output Filter"		);
		SHOW_FIELD(20, 20, 1, capacity,	"Output Capacity"	);
		SHOW_FIELD(21, 21, 1, outputs,	"Output Mode"		);
		SHOW_FIELD(31, 22, 1, NULL,		"Reserved"			);
	}

	return(errs);
}



//*****************************************************************************
static int _acr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	source[]	= { "Primary",	"Alternate"	};

	long	v;

	SHOW_FIELD( 8,  0, 0, NULL,		"Nclk"			);
	v	= value & 0x1FF;
	gsc_label_long_comma(v);
	printf(": Fref-alt = 16,000,000 * (1 + ");
	gsc_label_long_comma(v);
	printf(" / 511)\n");

	SHOW_FIELD( 9,  9, 1, source,	"Ref. Source"	);
	SHOW_FIELD(31, 10, 1, NULL,		"Reserved"		);
	return(0);
}



// variables ******************************************************************

static gsc_reg_def_t	_gsc[]	=
{
	{ _GSC_REG(BCR),		0,	_bcr_decode,	"Board Control Register"		},
	{ _GSC_REG(CSR),		0,	_csr_decode,	"Channel Selection Register"	},
	{ _GSC_REG(SRR),		0,	_srr_decode,	"Sample Rate Register"			},
	{ _GSC_REG(BOR),		0,	_bor_decode,	"Buffer Operations Register"	},
	{ _GSC_REG(FOR),		0,	_for_decode,	"Firmware Options Register"		},
	{ _GSC_REG(AVR),		0,	NULL,			"Autocal Values Register"		},
	{ _GSC_REG(ODBR),		0,	NULL,			"Output Data Buffer Register"	},
	{ _GSC_REG(ACR),		0,	_acr_decode,	"Adjustable Clock Register"		},

	{ NULL, 0, 0, 0, 0,		NULL	}
};



//*****************************************************************************
static const gsc_reg_def_t* _find_reg(u32 reg, const gsc_reg_def_t* list)
{
	const gsc_reg_def_t*	def	= NULL;
	int						i;

	for (i = 0; list[i].name; i++)
	{
		if (reg == list[i].reg)
		{
			def	= &list[i];
			break;
		}
	}

	return(def);
}



/******************************************************************************
*
*	Function:	ao16_reg_get_def_id
*
*	Purpose:
*
*		Retrieve the register definition structure given the register id.
*
*	Arguments:
*
*		reg		The id of the register to access.
*
*	Returned:
*
*		NULL	The register id wasn't found.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* ao16_reg_get_def_id(u32 reg)
{
	const gsc_reg_def_t*	def;

	def	= _find_reg(reg, _gsc);
	return(def);
}



/******************************************************************************
*
*	Function:	ao16_reg_get_def_index
*
*	Purpose:
*
*		Retrieve the register definition structure based on an index.
*
*	Arguments:
*
*		index	The index of the register to access.
*
*	Returned:
*
*		NULL	The index doesn't correspond to a known register.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* ao16_reg_get_def_index(int index)
{
	const gsc_reg_def_t*	def;

	if (index < 0)
		def	= NULL;
	else if (index >= (SIZEOF_ARRAY(_gsc) - 1))
		def	= NULL;
	else
		def	= &_gsc[index];

	return(def);
}



/******************************************************************************
*
*	Function:	ao16_reg_get_desc
*
*	Purpose:
*
*		Retrieve the description of the specified register.
*
*	Arguments:
*
*		reg		The register whose description is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* ao16_reg_get_desc(u32 reg)
{
	const gsc_reg_def_t*	def;
	const char*				desc;

	def	= _find_reg(reg, _gsc);

	if (def)
		desc	= def->desc;
	else
		desc	= "UNKNOWN";

	return(desc);
}



/******************************************************************************
*
*	Function:	ao16_reg_get_name
*
*	Purpose:
*
*		Retrieve the name of the specified register.
*
*	Arguments:
*
*		reg		The register whose name is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* ao16_reg_get_name(u32 reg)
{
	const gsc_reg_def_t*	def;
	const char*				name;

	def	= _find_reg(reg, _gsc);

	if (def)
		name	= def->name;
	else
		name	= "UNKNOWN";

	return(name);
}



/******************************************************************************
*
*	Function:	ao16_reg_list
*
*	Purpose:
*
*		List the GSC registers and their values.
*
*	Arguments:
*
*		fd		The handle to access the device.
*
*		detail	List the register details?
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int ao16_reg_list(int fd, int detail)
{
	int	errs;

	errs	= gsc_reg_list(fd, _gsc, detail, ao16_reg_read);
	return(errs);
}



