// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AO16/16AO16_Linux_2.x.x.x_DN/utils/query.c $
// $Rev: 56222 $
// $Date: 2025-02-06 13:45:16 -0600 (Thu, 06 Feb 2025) $

// 16AO16: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	ao16_query
*
*	Purpose:
*
*		Provide a visual wrapper for the AO16_IOCTL_QUERY service.
*
*	Arguments:
*
*		fd		Use this handle to access the device.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely? 0 = no, !0 = yes
*
*		set		This is the query option to access.
*
*		get		The results are reported here, if not NULL.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int	ao16_query(int fd, int index, int verbose, s32 set, s32* get)
{
	char		buf[128];
	int			errs	= 0;
	const char*	ptr;
	s32			query	= set;
	int			ret;
	s32			tmp;

	switch (query)
	{
		default:

			errs++;
			ptr	= buf;
			sprintf(buf, "Query Error (#%ld)", (long) query);
			break;

		case AO16_QUERY_AUTOCAL_MS:			ptr	= "Autocal Duration";	break;
		case AO16_QUERY_CABLE_INVERT_4:		ptr	= "Invert 4 Signals";	break;
		case AO16_QUERY_CABLE_PASSIVE_3:	ptr	= "3 Passive Signals";	break;
		case AO16_QUERY_CHANNEL_MASK:		ptr	= "Channel Mask";		break;
		case AO16_QUERY_CHANNEL_MAX:		ptr	= "Max Channels";		break;
		case AO16_QUERY_CHANNEL_QTY:		ptr	= "Installed Channels";	break;
		case AO16_QUERY_COUNT:				ptr	= "Query Options";		break;
		case AO16_QUERY_DEVICE_TYPE:		ptr	= "Device Type";		break;
		case AO16_QUERY_DIFFERENTIAL:		ptr	= "Diff Outputs";		break;
		case AO16_QUERY_DMDMA:				ptr	= "DMDMA Support";		break;
		case AO16_QUERY_FIFO_SIZE:			ptr	= "FIFO Size";			break;
		case AO16_QUERY_FILTER:				ptr	= "Filter";				break;
		case AO16_QUERY_FREF_DEFAULT:		ptr	= "Fref Default";		break;
		case AO16_QUERY_FSAMP_MAX:			ptr	= "Fsamp Max";			break;
		case AO16_QUERY_FSAMP_MIN:			ptr	= "Fsamp Min";			break;
		case AO16_QUERY_INIT_MS:			ptr	= "Init. Duration";		break;
		case AO16_QUERY_MODEL:				ptr	= "Model";				break;
		case AO16_QUERY_NCLK_MASK:			ptr	= "Nclk Mask";			break;
		case AO16_QUERY_NCLK_MAX:			ptr	= "Nclk Max";			break;
		case AO16_QUERY_NCLK_MIN:			ptr	= "Nclk Min";			break;
		case AO16_QUERY_NRATE_MASK:			ptr	= "Nrate Mask";			break;
		case AO16_QUERY_NRATE_MAX:			ptr	= "Nrate Max";			break;
		case AO16_QUERY_NRATE_MIN:			ptr	= "Nrate Min";			break;
		case AO16_QUERY_OUTPUT_CAPACITY:	ptr	= "Output Capacity";	break;
		case AO16_QUERY_OUTPUT_FILTER:		ptr	= "Output Filter Sel";	break;
		case AO16_QUERY_VOLT_RANGE:			ptr	= "Voltage Range";		break;
		case AO16_QUERY_WATCHDOG:			ptr	= "Watchdog";			break;
	}

	if (verbose)
		gsc_label_index(ptr, index);

	ret		= ao16_ioctl(fd, AO16_IOCTL_QUERY, &set);
	errs	+= ret ? 1 : 0;

	switch (query)
	{
		default:

			errs++;
			gsc_label_long_comma_buf(set, buf);
			break;

		case AO16_QUERY_CHANNEL_MAX:
		case AO16_QUERY_CHANNEL_QTY:
		case AO16_QUERY_COUNT:
		case AO16_QUERY_NCLK_MAX:
		case AO16_QUERY_NCLK_MIN:
		case AO16_QUERY_NRATE_MAX:
		case AO16_QUERY_NRATE_MIN:

			gsc_label_long_comma_buf(set, buf);
			break;

		case AO16_QUERY_CHANNEL_MASK:
		case AO16_QUERY_NCLK_MASK:
		case AO16_QUERY_NRATE_MASK:

			sprintf(buf, "0x%lX", (long) set);
			break;

		case AO16_QUERY_FREF_DEFAULT:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " Hz");
			break;

		case AO16_QUERY_AUTOCAL_MS:
		case AO16_QUERY_INIT_MS:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " ms");
			break;

		case AO16_QUERY_FSAMP_MAX:
		case AO16_QUERY_FSAMP_MIN:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " S/S");
			break;

		case AO16_QUERY_DIFFERENTIAL:
		case AO16_QUERY_DMDMA:
		case AO16_QUERY_CABLE_INVERT_4:
		case AO16_QUERY_CABLE_PASSIVE_3:
		case AO16_QUERY_OUTPUT_FILTER:
		case AO16_QUERY_WATCHDOG:

			strcpy(buf, set ? "Supported" : "Not Supported");
			break;

		case AO16_QUERY_FIFO_SIZE:

			ptr	= "";
			tmp	= set;

			if ((tmp) && ((tmp % 1024) == 0))
			{
				ptr	= "K";
				tmp	/= 1024;
			}

			gsc_label_long_comma_buf(tmp, buf);
			strcat(buf, ptr);
			strcat(buf, " Words");
			break;

		case AO16_QUERY_DEVICE_TYPE:

			if (set == GSC_DEV_TYPE_16AO16)
			{
				strcpy(buf, "16AO16");
			}
			else
			{
				errs++;
				sprintf(buf, "INVALID: %ld", (long) set);
			}

			break;

		case AO16_QUERY_FILTER:

			switch (set)
			{
				default:

					errs++;
					sprintf(buf, "Invalid option: %ld", (long) set);
					break;

				case AO16_FILTER_NONE:		strcpy(buf, "None (>300KHz)");	break;
				case AO16_FILTER_10KHZ:		strcpy(buf, "10K Kz");			break;
				case AO16_FILTER_100KHZ:	strcpy(buf, "100K Hz");			break;
				case AO16_FILTER_F1:		strcpy(buf, "F1");				break;
				case AO16_FILTER_F2:		strcpy(buf, "F2");				break;
				case AO16_FILTER_F3:		strcpy(buf, "F3");				break;
				case AO16_FILTER_F4:		strcpy(buf, "F4");				break;
			}

			break;

		case AO16_QUERY_MODEL:

			switch (set)
			{
				default:

					errs++;
					sprintf(buf, "Invalid option: %ld", (long) set);
					break;

				case AO16_MODEL_16AO16:		strcpy(buf, "16AO16");		break;
				case AO16_MODEL_16AO16FLV:	strcpy(buf, "16AO16FLV");	break;
			}

			break;

		case AO16_QUERY_OUTPUT_CAPACITY:

			switch (set)
			{
				default:

					errs++;
					sprintf(buf, "Invalid option: %ld", (long) set);
					break;

				case -1:								strcpy(buf, "Not Supported");					break;
				case AO16_OUTPUT_CAPACITY_HI_LEVEL:		strcpy(buf, "High Level (+-10V, +-5V)");		break;
				case AO16_OUTPUT_CAPACITY_HI_CURRENT:	strcpy(buf, "High Current (+-2.5V, +-1.5V)");	break;
			}

			break;

		case AO16_QUERY_VOLT_RANGE:

			switch (set)
			{
				default:

					errs++;
					sprintf(buf, "Invalid option: %ld", (long) set);
					break;

				case -1:					strcpy(buf, "Not Supported");	break;
				case AO16_VOLT_RANGE_LOW:	strcpy(buf, "Low");				break;
				case AO16_VOLT_RANGE_HIGH:	strcpy(buf, "High");			break;
			}

			break;
	}

	if (verbose)
	{
		if (errs)
			printf("FAIL <---  (%s)\n", buf);
		else
			printf("%s\n", buf);
	}

	if (get)
		get[0]	= set;

	return(errs);
}


