// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AO16/16AO16_Linux_2.x.x.x_DN/samples/mcao/main.c $
// $Rev: 56214 $
// $Date: 2025-02-06 13:34:07 -0600 (Thu, 06 Feb 2025) $

// 16AO16: Sample Application: source file

#include "main.h"



//*****************************************************************************
static int _parse_args(int argc, char** argv, args_t* args)
{
	const gsc_arg_item_t	list[]	=
	{
		//	type				var						arg			values					desc
		{	GSC_ARG_S32_FLAG,	&args->show_stats,		"-bs",		{ 1 },					"Show buffer statistics when done."								},
		{	GSC_ARG_S32_FLAG,	&args->io_mode,			"-bmdma",	{GSC_IO_MODE_BMDMA },	"Use Block Mode DMA mode for write requests."					},
		{	GSC_ARG_S32_FLAG,	&args->continuous,		"-c",		{ -1 },					"Continue testing until an error occurs."						},
		{	GSC_ARG_S32_FLAG,	&args->continuous,		"-C",		{ +1 },					"Continue testing even if errors occur."						},
		{	GSC_ARG_S32_FLAG,	&args->io_mode,			"-dmdma",	{ GSC_IO_MODE_DMDMA },	"Use Demand Mode DMA mode for write requests."					},
		{	GSC_ARG_S32_RANGE,	&args->ao_channels,		"-e",		{ 1, 0xFFFF },			"Enable these output channels (default: 0xFFFF)."				},
		{	GSC_ARG_S32_RANGE,	&args->wave_cps,		"-hz",		{ 1, CPS_MIN },			"Waveform cycles per second (default: 100)."					},
		{	GSC_ARG_S32_MIN,	&args->minute_limit,	"-m",		{ 1 },					"Limit continuous testing to # minutes."						},
		{	GSC_ARG_S32_MIN,	&args->test_limit,		"-n",		{ 1 },					"Limit continuous testing to # iterations."						},
		{	GSC_ARG_S32_FLAG,	&args->io_mode,			"-pio",		{ GSC_IO_MODE_PIO },	"Use PIO mode for write requests."								},
		{	GSC_ARG_S32_MASK,	&args->wave_ramp,		"-rmp",		{ 0xFFFF },				"Output a ramp wave on these channels (mask: 0xFFFF)(default)."	},
		{	GSC_ARG_S32_MASK,	&args->wave_saw,		"-saw",		{ 0xFFFF },				"Output a saw tooth wave on these channels (mask: 0xFFFF)."		},
		{	GSC_ARG_S32_MASK,	&args->wave_sin,		"-sin",		{ 0xFFFF },				"Output a sine wave on these channels (mask: 0xFFFF)."			},
		{	GSC_ARG_S32_RANGE,	&args->fsamp,			"-sps",		{ 172, 450000 },		"Output data at this sample rate."								},
		{	GSC_ARG_S32_MASK,	&args->wave_square,		"-sqr",		{ 0xFFFF },				"Output a square wave on these channels (mask: 0xFFFF)."		},
		{	GSC_ARG_S32_RANGE,	&args->seconds,			"-s",		{ 1, 3600 },			"Output data for # seconds (default: 15)."						},
		{	GSC_ARG_DEV_INDEX,	&args->index,			"index",	{ 1 },					"The zero based index of the device to access."					}
	};

	const gsc_arg_set_t	set	=
	{
		/* name	*/	"mcao",
		/* desc	*/	"Multi-Channel Analog Output Sample Application",
		/* qty	*/	SIZEOF_ARRAY(list),
		/* list	*/	list
	};

	int	bit;
	int	errs;
	int	i;

	memset(args, 0, sizeof(args_t));
	args->ao_channels	= 0xFFFF;
	args->fsamp			= 10000;
	args->io_mode		= GSC_IO_MODE_PIO;
	args->seconds		= 15;
	args->wave_ramp		= 0xFFFF;
	args->wave_cps		= 100;

	errs	= gsc_args_parse(argc, argv, &set);

	// Make sure every channel has a pattern.

	for (i = 0; i < 16; i++)
	{
		bit	= 1 << i;

		if (bit & args->wave_ramp)
			continue;

		if (bit & args->wave_saw)
			continue;

		if (bit & args->wave_sin)
			continue;

		if (bit & args->wave_square)
			continue;

		args->wave_ramp	|= bit;
	}

	return(errs);
}



//*****************************************************************************
static void _show_access_index(const args_t* args)
{
	gsc_label("Accessing Device");
	printf("%d\n", args->index);
}



//*****************************************************************************
static void _show_time_stamp(void)
{
	const char*	psz;
	struct tm*	stm;
	time_t		tt;

	time(&tt);
	stm	= localtime(&tt);
	psz	= (char*) asctime(stm);
	gsc_label("Time Stamp");
	printf("%s", psz);
}



/******************************************************************************
*
*	Function:	main
*
*	Purpose:
*
*		Control the overall flow of the application.
*
*	Arguments:
*
*		argc			The number of command line arguments.
*
*		argv			The list of command line arguments.
*
*	Returned:
*
*		EXIT_SUCCESS	No errors were encounterred.
*		EXIT_FAILURE	One or more problems were encounterred.
*
******************************************************************************/

int main(int argc, char** argv)
{
	args_t	args;
	int		errs;
	time_t	exec		= time(NULL);
	int		failures	= 0;
	long	hours;
	long	mins;
	time_t	now;
	int		passes		= 0;
	int		qty			= 0;
	int		ret;
	long	secs;
	time_t	t_limit;
	time_t	test;
	int		tests		= 0;

	for (;;)	// A convenience loop.
	{
		gsc_label_init(30);
		test	= time(NULL);
		errs	= _parse_args(argc, argv, &args);

		// Introduction

		t_limit	= exec + (args.minute_limit * 60);
		os_id_host();
		_show_time_stamp();
		_show_access_index(&args);
		errs	+= ao16_init_util(1);
		errs	+= os_id_driver(ao16_open, ao16_read, ao16_close);
		errs	+= ao16_count_boards(1, &qty);

		if (qty <= 0)
			errs++;

		// Open access to device

		if (errs == 0)
			errs	= ao16_open_util(args.index, 0, -1, 1, &args.fd);

		if (errs)
			args.continuous	= 0;

		if (errs == 0)
		{
			// Device identification

			errs	= ao16_id_device(args.fd, -1, 1);

			// Perform testing

			errs	+= perform_tests(&args);

			// Close device access

			errs	+= ao16_close_util(args.fd, -1, 1);
		}

		// End of processing.

		now	= time(NULL);
		tests++;

		if (errs)
		{
			failures++;
			printf(	"\nRESULTS: FAIL <---  (%d error%s)",
					errs,
					(errs == 1) ? "" : "s");
		}
		else
		{
			passes++;
			printf("\nRESULTS: PASS");
		}

		secs	= (long) (now - test);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(" (duration %ld:%ld:%02ld)\n", hours, mins, secs);

		secs	= (long) (now - exec);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(	"SUMMARY: tests %d, pass %d, fail %d"
				" (duration %ld:%ld:%02ld)\n\n",
				tests,
				passes,
				failures,
				hours,
				mins,
				secs);

		if (args.continuous == 0)
			break;

		if ((args.continuous < 0) && (errs))
			break;

		if ((args.minute_limit > 0) && (now >= t_limit))
			break;

		if ((args.test_limit > 0) && (tests >= args.test_limit))
			break;
	}

	ret	= ((qty <= 0) || (errs) || (failures)) ? EXIT_FAILURE : EXIT_SUCCESS;
	return(ret);
}


