// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AO16/16AO16_Linux_2.x.x.x_DN/samples/aout/perform.c $
// $Rev: 56210 $
// $Date: 2025-02-06 13:30:01 -0600 (Thu, 06 Feb 2025) $

// 16AO16: Sample Application: source file

#include "main.h"



// macros *********************************************************************

#define	_1M				(1024 * 1024)

#define	DELTA_NS(t0,t1)	(((1000000000LL * ((t1).tv_sec - (t0).tv_sec)) + (t1).tv_nsec - (t0).tv_nsec))

#define	EMPTY_THREAD	"Empty Event Thread"

#define	HDR_1			"# aout log file\n"
#define	HDR_2			"# BOR Rx Time: duration of call to read the Buffer Operations Register\n"
#define	HDR_3			"# Tx Time:     duration of the ao16_write() call for 'Tx Bytes'\n"
#define	HDR_4			"# FIFO Est:    gross estimate of FIFO fill level based on status from BOR\n"
#define	HDR_T			"# Index     t0               FIFO Est  BOR Rx Time    Tx Bytes  Tx Time        Emptied\n"
#define	HDR_U			"# ========  ===============  ========  =============  ========  =============  ==========\n"



// data types *****************************************************************

typedef struct
{
	int				index;
	os_time_ns_t	t0;			// time immediately before reading FIFO fill level
	u32				fifo;		// FIFO fill level immediately after "t0"
	os_time_ns_t	t1;			// time immediately before write call
	os_time_ns_t	t2;			// time immediately after write call
	int				sent;		// return value from write call
	int				empty_qty;	// number of times the output buffer went empty
} log_t;

typedef struct
{
	struct
	{
		int				stop;
	} ctrl_c;

	struct
	{
		long long		errors;		// errors encountered inside the empty detection thread
		long long		qty;		// Number of times the output buffer went empty
		int				stop;		// stop operation: we reached the command line limit
		int				stop_thread;// Stop thread when requested
		long long		sum;		// used to compute averag and per log entry count
		os_thread_t		thread;		// used to count empty events
	} empty;

	struct
	{
		// Entries are always added to the log when requested.
		// When the log gets full, newer entries overwrite the oldest entries.
		log_t			data[_1M];	// monitor content go here
		long long		lost;		// number of log entries overwritten by newer entries
		long long		prefill;	// Number of log entries made during prefill
		long long		qty;		// number of entries added to log
		int				stop;		// stop operation: we've reached the command line limit
		os_time_ns_t	t0;			// Time of very first action
	} log;

	struct
	{
		int				stop;		// stop operation: write was too fast per command line limit
		long long		limit;		// stop if write time takes this long or less
	} ns;

	struct
	{
		u32				buffer[_1M];	// source for output data
		long long		samples;		// Number of samples to send per write
	} output;

} data_t;



// variables ******************************************************************

static	data_t	_data;



//*****************************************************************************
static int _empty_event_thread(void* arg)
{
	const args_t*	args	= arg;
	gsc_wait_t		wait;

	for (;;)
	{
		_data.empty.errors	+= ao16_irq_sel(args->fd, -1, 0, AO16_IRQ_BUF_EMPTY, NULL);

		memset(&wait, 0, sizeof(wait));
		wait.gsc			= AO16_WAIT_GSC_BUF_EMPTY;
		wait.timeout_ms		= 1000;
		_data.empty.errors	+= ao16_wait_event(args->fd, -1, 0, &wait);

		if (_data.empty.stop_thread)
			break;

		if (wait.flags & GSC_WAIT_FLAG_DONE)
			_data.empty.qty++;
	}

	return(0);
}



//*****************************************************************************
static int _empty_thread_start(const args_t* args)
{
	int	errs;

	gsc_label(EMPTY_THREAD);

	errs	= os_thread_create(&_data.empty.thread, EMPTY_THREAD, _empty_event_thread, (void*) args);

	printf("%s  (Start)\n", errs ? "FAIL <---" : "PASS");
	return(errs);
}



//*****************************************************************************
static int _empty_thread_stop(const args_t* args)
{
	int			errs	= 0;
	gsc_wait_t	wait;

	gsc_label(EMPTY_THREAD);

	_data.empty.stop_thread	= 1;

	memset(&wait, 0, sizeof(wait));
	wait.gsc	= AO16_WAIT_GSC_ALL;
	errs		+= ao16_wait_cancel(args->fd, -1, 0, &wait);

	errs		+= os_thread_destroy(&_data.empty.thread);

	printf("%s  (Stop)\n", errs ? "FAIL <---" : "PASS");
	return(errs);
}



//*****************************************************************************
static int _init_data(const args_t* args, s32 chans)
{
	#define	PI		(3.14159265359)

	int			chan;		// 0, 1, 2 ...
	u32*		dest;
	int			i;
	int			pattern;
	long		period_len;	// The length of the pattern in samples.
	long		period_tot;	// Pattern samples total for all channels.
	long		period;		// Pattern length in x units.
	int			samp;		// 0, 1, 2, 3
	long		samp_dat;	// Total sample size of the data buffer.
	float		samp_sec;	// Samples Per Second for all active channels
	double		samp_tot;	// Samples Total for all active channels
	int			seconds;
	const char*	units;

	gsc_label("Initializing Data");

	if (args->minutes)
		seconds	= args->minutes * 60;
	else
		seconds	= args->seconds;

	samp_sec	= ((float) args->fsamp) * chans;
	samp_tot	= ((float) args->fsamp) * chans * seconds;
	samp_dat	= sizeof(_data.output.buffer) / 4;

	// Assign a default pattern length in samples.
	period_len	= (args->fsamp <= 100)		? args->fsamp			//      10-      100 ->    10-   100
				: (args->fsamp <= 1000)		? (args->fsamp / 2)		//     101-    1,000 ->    50-   500
				: (args->fsamp <= 10000)	? (args->fsamp / 5)		//   1,001-   10,000 ->   200- 2,000
				: (args->fsamp <= 100000)	? (args->fsamp / 10)	//  10,001-  100,000 -> 1,000-10,000
				: (args->fsamp / 100);								// 100,001-1,000,000 -> 1,000-10,000
	period_len	= (period_len <= 1) ? 2 : period_len;
	period_tot	= period_len * chans;

	// Compute the period in ns/us/ms

	if (args->fsamp)
		period	= (long) (1000000000. / args->fsamp * period_len);
	else
		period	= 0;

	units		= "ns";

	if ((period % 1000) == 0)
	{
		period	/= 1000;
		units	= "us";
	}

	if ((period % 1000) == 0)
	{
		period	/= 1000;
		units	= "ms";
	}

	if (samp_tot <= SIZEOF_ARRAY(_data.output.buffer))
	{
		// We can get the entire image in the buffer.
		_data.output.samples	= (long) samp_tot;
	}
	else if (samp_sec <= SIZEOF_ARRAY(_data.output.buffer))
	{
		// We can get one second of the image in the buffer.
		_data.output.samples	= (long) samp_sec;
	}
	else if (period_tot <= SIZEOF_ARRAY(_data.output.buffer))
	{
		// We can get one period of the image in the buffer.
		_data.output.samples	= period_tot;
	}
	else
	{
		// We can get only a fraction of a period of the image in the buffer.
		_data.output.samples	= (long) ((samp_dat / chans) * chans);
	}

	// Initialize the data buffer.
	// The data is encoded in OFFSET BINARY format.
	dest	= _data.output.buffer;
	#define	MAX		0xFFFFU

	for (i = 0, samp = 0; i < _data.output.samples; i += chans, samp++)
	{
		samp	= (samp < period_len) ? samp : 0;

		for (chan = 0; chan < chans; chan++, dest++)
		{
			pattern	= chan % 4;

			switch (pattern)
			{
				default:
				case 0:	// ramp down wave (sharp rise, slow fall)

					dest[0]	= (u32) (MAX - (u32) ((float) samp / (period_len - 1) * MAX));
					continue;

				case 1:	// ramp up wave (slow rise, sharp fall)

					dest[0]	= (u32) ((float) samp / (period_len - 1) * MAX);
					continue;

				case 2:	// square wave

					dest[0]	= (u32) ((samp < (period_len / 2)) ? MAX : 0);
					continue;

				case 3: // sine wave

					dest[0]	= (u32) ((float) (MAX >> 1)) * (1 + sin((2 * PI / period_len) * samp));
					continue;
			}
		}
	}

	printf("DONE  (pattern length: ");
	gsc_label_long_comma(period_len);
	printf(" samples, ");
	gsc_label_long_comma(period);
	printf(" %s)\n", units);
	return(0);
}



//*****************************************************************************
static int _save_eol(const char* name, FILE* file)
{
	int	errs	= 0;
	int	i;

	i	= fprintf(file, "\r\n");

	if (i != 2)
	{
		printf("FAIL <---  (fprintf() failure to %s)\n", name);
		errs	= 1;
	}

	return(errs);
}



//*****************************************************************************
static int _save_value(const args_t* args, const char* name, FILE* file, u32 value)
{
	char	buf1[40];
	char	buf2[80];
	int		errs	= 0;
	int		i;
	int		len;

	if (args->decimal)
		sprintf(buf1, "%ld", (long) value);
	else
		sprintf(buf1, "%08lX", (long) value);

	sprintf(buf2, "  %8s", buf1);
	len	= strlen(buf2);
	i	= fprintf(file, "%8s", buf2);

	if (i != len)
	{
		printf("FAIL <---  (fprintf() failure to %s)\n", name);
		errs	= 1;
	}

	return(errs);
}



//*****************************************************************************
static int _save_data(const args_t* args, s32 chans, int errs)
{
	char		buf[64];
	int			chan;
	FILE*		file;
	long		l;
	const char*	name	= DATA_FILE;
	long		samples	= _data.output.samples;

	strcpy(buf, "Saving");

	if (args->save_force)
	{
		errs	= 0;
		strcat(buf, " (Forced)");
	}

	gsc_label(buf);

	for (;;)
	{

		if ((args->save_force == 0) && (errs))
		{
			printf("FAIL <---  (Aborting due to errors.)\n");
			break;
		}

		file	= fopen(name, "w+b");

		if (file == NULL)
		{
			printf("FAIL <---  (unable to create %s)\n", name);
			errs	= 1;
			break;
		}

		for (l = 0, chan = 0; l < samples; l++)
		{
			errs	+= _save_value(args, name, file, _data.output.buffer[l]);

			if ((l > 0) && (chan == (chans - 1)))
				errs	+= _save_eol(name, file);

			chan	= (chan + 1) % chans;
		}

		fclose(file);

		if (errs == 0)
			printf("PASS  (%s)\n", name);

		break;
	}

	return(errs);
}



//*****************************************************************************
static void _log_append(
	const os_time_ns_t*	t0,
	const os_time_ns_t* t1,
	const os_time_ns_t* t2,
	u32					fifo,	// estimated FIFO fill level
	int					sent,
	int					empty_qty)
{
	int	i;

	// The log array captures only the most recent entries. When the log array
	// gets full, new entries overwrite the oldest entries present.

	if (_data.log.qty == 0)
		_data.log.t0	= t0[0];

	if (_data.log.qty >= SIZEOF_ARRAY(_data.log.data))
		_data.log.lost++;

	i	= _data.log.qty % SIZEOF_ARRAY(_data.log.data);

	_data.log.data[i].index		= _data.log.qty;
	_data.log.data[i].t0		= t0[0];
	_data.log.data[i].t1		= t1[0];
	_data.log.data[i].t2		= t2[0];
	_data.log.data[i].fifo		= fifo;
	_data.log.data[i].sent		= sent;
	_data.log.data[i].empty_qty	= empty_qty - _data.empty.sum;
	_data.empty.sum				+= _data.log.data[i].empty_qty;
	_data.log.qty++;
}



//*****************************************************************************
static int _write_string(FILE* file, int errs, const char* src)
{
	long	l;
	int		tmp	= errs;

	errs	= 0;

	if (tmp == 0)
	{
		l	= file_write_str(file, src);

		if (l < 0)
		{
			errs	= 1;
			printf("FAIL <---  (write to file)\n");
		}
	}

	return(errs);
}



//*****************************************************************************
static int _log_entry_prefill(FILE* file, int errs, int prefill)
{
	char	buf[80];

	sprintf(buf, "# Prefill:     %d entrie%s", prefill, (prefill == 1) ? "" : "s");

	if (prefill)
	{
		strcat(buf, ", #0");

		if (prefill > 1)
			sprintf(buf + strlen(buf), "-#%d", prefill - 1);
	}

	strcat(buf, "\n");
	errs	= _write_string(file, errs, buf);
	return(errs);
}



//*****************************************************************************
static int _log_entry_us_stop(FILE* file, int errs, int us_stop)
{
	char	buf[80];

	strcpy(buf, "# us Stop:     ");

	if (us_stop <= 0)
	{
		strcat(buf, "Not Enabled\n");
	}
	else
	{
		gsc_label_long_comma_buf(us_stop, buf + strlen(buf));
		strcat(buf, "us\n");
	}

	errs	= _write_string(file, errs, buf);
	return(errs);
}



//*****************************************************************************
static int _log_entry_index(FILE* file, int errs, int comment, int index)
{
	char	buf[80];

	if (comment)
		strcpy(buf, "#         ");
	else
		sprintf(buf, "  %8d", _data.log.data[index].index);

	errs	= _write_string(file, errs, buf);
	return(errs);
}



//*****************************************************************************
static int _log_entry_t0(FILE* file, int errs, int comment, int index)
{
	char		buf[80];
	long long	ns;		// nano seconds

	if (comment)
	{
		strcpy(buf, "  Averages       ");
	}
	else
	{
		ns	= DELTA_NS(_data.log.t0, _data.log.data[index].t0);
		sprintf(buf, "  %5lld.%09lld", ns / 1000000000, ns % 1000000000);
	}

	errs	= _write_string(file, errs, buf);
	return(errs);
}



//*****************************************************************************
static int _log_entry_fifo(
	FILE*				file,
	int					errs,
	int					average,	// compute average? If not, then add to sum
	int					index,
	unsigned long long*	sum,
	int					qty)
{
	long long	ave;
	char		buf[80];

	if (average)
	{
		ave	= sum[0] / qty;
		sprintf(buf, "  %8llu", ave);
	}
	else
	{
		sum[0]	+= _data.log.data[index].fifo;
		sprintf(buf, "  %8d", (int) _data.log.data[index].fifo);
	}

	errs	= _write_string(file, errs, buf);
	return(errs);
}



//*****************************************************************************
static int _log_entry_fifo_t(
	FILE*				file,
	int					errs,
	int					average,	// compute average? If not, then add to sum
	int					index,
	unsigned long long*	sum,
	int					qty)
{
	char		buf[80];
	long long	ns;			// nano seconds

	if (average)
	{
		ns	= sum[0] / qty;
	}
	else
	{
		ns		= DELTA_NS(_data.log.data[index].t0, _data.log.data[index].t1);
		sum[0]	+= ns;
	}

	sprintf(buf, "  %3lld.%09lld", ns / 1000000000, ns % 1000000000);
	errs	= _write_string(file, errs, buf);
	return(errs);
}



//*****************************************************************************
static int _log_entry_tx(
	FILE*				file,
	int					errs,
	int					average,	// compute average? If not, then add to sum
	int					index,
	unsigned long long*	sum,
	int					qty)
{
	long long	ave;
	char		buf[80];

	if (average)
	{
		ave	= sum[0] / qty;
		sprintf(buf, "  %8lld", ave);
	}
	else
	{
		sum[0]	+= _data.log.data[index].sent;
		sprintf(buf, "  %8d", _data.log.data[index].sent);
	}

	errs	= _write_string(file, errs, buf);
	return(errs);
}



//*****************************************************************************
static int _log_entry_tx_t(
	FILE*				file,
	int					errs,
	int					average,	// compute average? If not, then add to sum
	int					index,
	unsigned long long*	sum,
	int					qty)
{
	char		buf[80];
	long long	ns;			// nano seconds

	if (average)
	{
		ns	= sum[0] / qty;
	}
	else
	{
		ns		= DELTA_NS(_data.log.data[index].t1, _data.log.data[index].t2);
		sum[0]	+= ns;
	}

	sprintf(buf, "  %3lld.%09lld", ns / 1000000000, ns % 1000000000);
	errs	= _write_string(file, errs, buf);
	return(errs);
}



//*****************************************************************************
static int _log_entry_empty(
	FILE*	file,
	int		errs,
	int		average,	// compute average? If not, then add to sum
	int		index,
	float*	sum,
	int		qty)
{
	float	ave;
	char	buf[80];

	if (average)
	{
		ave	= sum[0] / qty;
		sprintf(buf, "  %10.6f", ave);
	}
	else
	{
		sum[0]	+= _data.log.data[index].empty_qty;
		sprintf(buf, "  %3d", _data.log.data[index].empty_qty);
	}

	errs	= _write_string(file, errs, buf);
	return(errs);
}



//*****************************************************************************
static int _log_save(const args_t* args)
{
	float				empty_sum	= 0;
	long long			entries;			// the number of log entries in the array
	int					errs		= 0;
	unsigned long long	fifo_sum	= 0;
	FILE*				file;
	long long			index		= 0;	// Which log entry we access and log to the file
	long long			ll;
	long long			offset;				// offset of the oldest entry in the array
	unsigned long long	t1_sum		= 0;
	unsigned long long	t2_sum		= 0;
	unsigned long long	tx_sum		= 0;

	gsc_label("Log Data");
	printf("\n");
	gsc_label_level_inc();

	//========================================================
	gsc_label("Empty Event Errors");

	if (_data.empty.errors)
	{
		printf("FAIL <---  (");
		gsc_label_long_comma(_data.empty.errors);
		printf(")\n");
		errs	+= _data.empty.errors;
	}
	else
	{
		printf("PASS  (none)\n");
	}

	//========================================================
	gsc_label("Empty Events");

	if (_data.empty.qty)
	{
		printf("Output Buffer Ran Empty ");
		gsc_label_long_comma(_data.empty.qty);
		printf(" time%s. <---\n", (_data.empty.qty == 1) ? "" : "s");
	}
	else
	{
		printf("Output Buffer did not run empty.\n");
	}

	//========================================================
	gsc_label("Write Entries Logged");
	gsc_label_long_comma(_data.log.qty);
	printf("\n");

	//========================================================
	gsc_label("Write Entries Overwritten");
	gsc_label_long_comma(_data.log.lost);
	printf("\n");

	//========================================================
	gsc_label("Write Prefill Entries");
	gsc_label_long_comma(_data.log.prefill);
	printf("\n");

	//========================================================
	gsc_label("Saving");

	for (;;)	// A convenience loop.
	{
		file	= file_create(LOG_FILE);

		if (file == NULL)
		{
			errs++;
			printf("FAIL <---  (can't create " LOG_FILE ")\n");
			break;
		}

		errs	+= _write_string(file, errs, HDR_1);
		errs	+= _write_string(file, errs, HDR_2);
		errs	+= _write_string(file, errs, HDR_3);
		errs	+= _write_string(file, errs, HDR_4);
		errs	+= _log_entry_prefill(file, errs, _data.log.prefill);
		errs	+= _log_entry_us_stop(file, errs, args->us_stop);
		errs	+= _write_string(file, errs, HDR_T);	// column titles
		errs	+= _write_string(file, errs, HDR_U);	// underlines

		if (_data.log.qty <= SIZEOF_ARRAY(_data.log.data))
		{
			entries	= _data.log.qty;
			offset	= 0;
		}
		else
		{
			entries	= SIZEOF_ARRAY(_data.log.data);
			offset	= _data.log.qty % SIZEOF_ARRAY(_data.log.data);
		}

		for (ll = 0; (errs == 0) && (ll < entries); ll++)
		{
			index	= (ll + offset) % SIZEOF_ARRAY(_data.log.data);
			errs	+= _log_entry_index	(file, errs, 0, index);
			errs	+= _log_entry_t0	(file, errs, 0, index);
			errs	+= _log_entry_fifo	(file, errs, 0, index, &fifo_sum, entries);
			errs	+= _log_entry_fifo_t(file, errs, 0, index, &t1_sum, entries);
			errs	+= _log_entry_tx	(file, errs, 0, index, &tx_sum, entries);
			errs	+= _log_entry_tx_t	(file, errs, 0, index, &t2_sum, entries);
			errs	+= _log_entry_empty	(file, errs, 0, index, &empty_sum, entries);
			errs	+= _write_string	(file, errs, "\n");
		}

		// Now report the averages ==================================
		errs	+= _write_string(file, errs, HDR_U);	// underlines

		if (_data.log.qty <= 0)
			_data.log.qty	= 1;

		errs	+= _log_entry_index	(file, errs, 1, index);
		errs	+= _log_entry_t0	(file, errs, 1, index);
		errs	+= _log_entry_fifo	(file, errs, 1, index, &fifo_sum, entries);
		errs	+= _log_entry_fifo_t(file, errs, 1, index, &t1_sum, entries);
		errs	+= _log_entry_tx	(file, errs, 1, index, &tx_sum, entries);
		errs	+= _log_entry_tx_t	(file, errs, 1, index, &t2_sum, entries);
		errs	+= _log_entry_empty	(file, errs, 1, index, &empty_sum, entries);
		errs	+= _write_string	(file, errs, "\r");

		file_close(file);

		if (errs == 0)
			printf("PASS  (%s)\n", LOG_FILE);

		break;
	}

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
static void _signal_handler(int sig)
{
	_data.ctrl_c.stop	= 1;
	fflush(stdout);
}



//*****************************************************************************
static void _fifo_estimate(s32 size, u32 bor, s32* fifo)
{
	if (bor & D15)
	{
		// Full
		fifo[0]	= size;
	}
	else if (bor & D14)
	{
		// High Quarter
		fifo[0]	= size * 3 / 4;
	}
	else if (bor & D13)
	{
		// Low Quarter
		fifo[0]	= size / 4;
	}
	else if (bor & D12)
	{
		// Empty
		fifo[0]	= 0;
	}
	else
	{
		// Half
		fifo[0]	= size / 2;
	}
}



//*****************************************************************************
static int _io_write(const args_t* args, s32 chans, int errs)
{
	os_time_ns_t	begin;
	u32				bor;		// contains FIFO fill level status
	os_time_ns_t	end;
	s32				fifo	= 0;	// estimated FIFO fill level
	os_time_ns_t	now;
	long long		ns;				// calculate duration of calls
	char*			ptr;
	double			rate;
	int				scans;
	int				seconds;
	int				send;
	int				send_pf;		// Send this amount for prefill writes
	int				sent;
	s32				size;			// size of FIFO in words, then bytes
	os_time_ns_t	t0;
	os_time_ns_t	t1;
	os_time_ns_t	t2;
	s32				timeout;
	long long		total	= 0;

	errs	+= ao16_query(args->fd, -1, 1, AO16_QUERY_FIFO_SIZE, &size);
	size	*= 4;

	os_time_get_ns(&t0);	// lint
	os_time_get_ns(&t1);	// lint

	signal(SIGINT, _signal_handler);	// Allow us to stop via Ctrl-C

	if (args->minutes)
		seconds	= args->minutes * 60;
	else
		seconds	= args->seconds;

	for (;;)	// A convenience loop.
	{
		if (errs)
		{
			errs	= 0;
			gsc_label("Writing Data");
			printf("SKIPPED  (due to errors)\n");
			break;
		}

		//=================================================
		gsc_label("Write Size");
		scans	= _data.output.samples / chans;
		send_pf	= _data.output.samples * 4;

		if (args->scan_limit)
		{
			if (scans > args->scan_limit)
				scans	= args->scan_limit;
		}

		_data.output.samples	= scans * chans;
		send	= _data.output.samples * 4;
		gsc_label_long_comma(scans);
		printf(" Scans => ");
		gsc_label_long_comma(send);
		printf(" Bytes\n");

		// Prefill the output buffer ======================
		gsc_label("Prefill Buffer");
		errs	+= ao16_tx_io_timeout(args->fd, -1, 0, -1, &timeout);
		errs	+= ao16_tx_io_timeout(args->fd, -1, 0, 0, NULL);

		for (;;)
		{
			if ((args->monitor) && (args->log_prefill))
			{
				os_time_get_ns(&t0);
				errs	+= ao16_reg_read(args->fd, -1, 0, AO16_GSC_BOR, &bor);
				os_time_get_ns(&t1);
				_fifo_estimate(size, bor, &fifo);
			}

			sent	= ao16_write(args->fd, _data.output.buffer, send_pf);

			if ((args->monitor) && (args->log_prefill))
			{
				// The us_stop option does not apply during prefill.
				os_time_get_ns(&t2);
				_log_append(&t0, &t1, &t2, fifo, sent, 0);
			}

			if (sent < 0)
				errs++;

			if ((errs) || (_data.ctrl_c.stop))
				break;

			if (sent < (send_pf / 10))
				break;

			total	+= sent;

			if (total >= (size * 2))	// make extra large, fsamp may be very high
				break;
		}

		_data.log.prefill	= _data.log.qty;
		errs	+= ao16_tx_io_timeout(args->fd, -1, 0, timeout, NULL);
		printf("%s\n", errs ? "FAIL <---" : "PASS");

		if ((errs) || (_data.ctrl_c.stop))
			break;

		// Write data to the device. ======================
		gsc_label("Writing Data");
		printf("Press Control-C to stop.\n");
		gsc_label(NULL);

		os_time_get_ns(&begin);
		end				= begin;
		end.tv_sec		+= seconds;
		_data.empty.qty	= 0;	// We ignore those seen during prefill.
		total			= 0;

		for (;;)
		{
			if ((errs) || (_data.ctrl_c.stop))
			{
				break;
			}
			else
			{
				os_time_get_ns(&now);

				if ((now.tv_sec > end.tv_sec) ||
					((now.tv_sec >= end.tv_sec) && (now.tv_nsec >= end.tv_nsec)))
				{
					if (args->indefinate == 0)
						break;
				}
			}

			send	= _data.output.samples * 4;
			ptr		= (char*) _data.output.buffer;

			for (;;)
			{
				if (args->monitor)
				{
					os_time_get_ns(&t0);
					errs	+= ao16_reg_read(args->fd, -1, 0, AO16_GSC_BOR, &bor);
					os_time_get_ns(&t1);
					_fifo_estimate(size, bor, &fifo);
				}

				sent	= ao16_write(args->fd, ptr, send);

				if (sent < 0)
					errs++;

				if (sent > 0)
				{
					total	+= sent;
					ptr		+= sent;
					send	-= sent;
				}

				if (args->monitor)
				{
					os_time_get_ns(&t2);

					if (errs == 0)
						_log_append(&t0, &t1, &t2, fifo, sent, _data.empty.qty);

					if (args->us_stop)
					{
						ns	= DELTA_NS(t1, t2);

						if (ns <= _data.ns.limit)
						{
							_data.ns.stop	= 1;
							break;
						}
					}

					if ((args->empty_qty) && (_data.empty.qty >= args->empty_qty))
					{
						_data.empty.stop	= 1;
						break;
					}

					if ((args->log_qty) && (_data.log.qty >= args->log_qty))
					{
						_data.log.stop	= 1;
						break;
					}
				}

				if (send <= 0)
					break;		// go back to send another block of data
			}

			if ((errs) || (sent <= 0) || (_data.empty.stop) || (_data.log.stop) || (_data.ns.stop))
				break;
		}

		// Compute the results. ===========================
		os_time_get_ns(&end);

		if (errs)
		{
			printf("FAIL <---  (write error, ");
		}
		else if (_data.empty.stop)
		{
			printf(	"STOPPED <---  (Output Buffer ran empty %lld time%s)\n",
					_data.empty.qty,
					(_data.empty.qty == 1) ? "" : "s");
			gsc_label(NULL);
			printf("PASS  (");
		}
		else if (_data.log.stop)
		{
			printf(	"STOPPED <---  (Logged %lld entr%s.)\n",
					_data.log.qty,
					(_data.log.qty == 1) ? "y" : "ies");
			gsc_label(NULL);
			printf("PASS  (");
		}
		else if (_data.ns.stop)
		{
			printf(	"STOPPED <---  (write took less than %dus.)\n",
					(int) args->us_stop);
			gsc_label(NULL);
			printf("PASS  (");
		}
		else if (sent == 0)
		{
			errs	= 1;
			printf("FAIL <---  (I/O timeout, ");
		}
		else
		{
			printf("PASS  (");
		}

		// Samples
		gsc_label_long_comma(total / 4);
		printf(" Samples, ");

		// Period

		ns	= ((long long) end.tv_sec   * 1000000000 + end.tv_nsec)
			- ((long long) begin.tv_sec * 1000000000 + begin.tv_nsec);
		gsc_label_float_comma(((long double) ns) / 1000000000.0, -1, 9);
		printf(" Seconds, ");

		if (ns > 0)
		{
			// Rate
			rate	= (long double) total / 4 / ns * 1000000000;
			gsc_label_float_comma(rate, -1, 3);
			printf(" S/S");
		}

		printf(", excludes prefill)\n");

		if (((_data.ns.stop) || (_data.empty.stop)) && (args->reg_dump))
			ao16_reg_list(args->fd, 1);

		break;
	}

	signal(SIGINT, NULL);
	return(errs);
}



//*****************************************************************************
int perform_tests(const args_t* args)
{
	s32	chans	= args->chan_qty;
	s32	dmdma;
	int	errs	= 0;
	s32	io_mode	= args->io_mode;
	u32	map;

	memset(&_data, 0, sizeof(_data));
	_data.ns.limit	= (long long) args->us_stop * 1000;

	errs	+= ao16_query(args->fd, -1, 1, AO16_QUERY_DMDMA, &dmdma);

	if ((dmdma == 0) && (io_mode == GSC_IO_MODE_DMDMA))
	{
		io_mode	= GSC_IO_MODE_BMDMA;
		gsc_label(NULL);
		printf("ALTERATION  (Using BMDMA since DMDMA is not supported.)\n");
	}

	errs	+= ao16_query(args->fd, -1, 1, AO16_QUERY_CHANNEL_QTY, &chans);

	if (chans > args->chan_qty)
		chans	= args->chan_qty;

	errs	+= ao16_config_ao		(args->fd, -1, 1, args->fsamp);
	errs	+= ao16_tx_io_mode		(args->fd, -1, 1, io_mode, NULL);
	map		= (u32) 0xFFFF >> (16 - chans);
	map		&= (u32) 0xFFFF >> (16 - args->chan_qty);
	errs	+= ao16_channel_sel		(args->fd, -1, 1, map, NULL);
	errs	+= ao16_fsamp_report_all(args->fd, -1, 1, NULL);

	if (args->monitor)
		errs	+= _empty_thread_start(args);

	gsc_label("Analog Output");
	printf("\n");
	gsc_label_level_inc();

	errs	+= _init_data				(args, chans);

	errs	+= ao16_buffer_over_frame	(args->fd, -1, 1, AO16_BUFFER_OVER_FRAME_CLR, NULL);
	errs	+= ao16_buffer_over_data	(args->fd, -1, 1, AO16_BUFFER_OVER_DATA_CLR, NULL);
	errs	+= _io_write				(args, chans, errs);
	errs	+= ao16_buffer_over_frame	(args->fd, -1, 1, AO16_BUFFER_OVER_FRAME_CHK, NULL);
	errs	+= ao16_buffer_over_data	(args->fd, -1, 1, AO16_BUFFER_OVER_DATA_CHK, NULL);

	if (args->monitor)
	{
		errs	+= _log_save(args);
		errs	+= _empty_thread_stop(args);
	}

	if (args->save)
		errs	+= _save_data(args, chans, errs);

	gsc_label_level_dec();
	return(errs);
}


