// $URL: http://subversion:8080/svn/gsc/trunk/drivers/gsc_common/driver/gsc_vpd.c $
// $Rev: 53839 $
// $Date: 2023-11-15 13:53:05 -0600 (Wed, 15 Nov 2023) $

// OS & Device Independent: Device Driver: source file

#include "main.h"



//*****************************************************************************
#if defined(DEV_SUPPORTS_VPD)
static int _vpd_read_dword(dev_data_t* dev, int adrs, u8* ptr)
{
	int	adrs_off	= dev->vpd.pci_reg_offset;
	int	data_off	= dev->vpd.pci_reg_offset + 4;
	int	i;
	int	ret			= 0;
	u16	status;

	os_reg_pci_tx_u16(dev, 1, adrs_off, adrs);

	for (i = 1;; i++)
	{
		status	= os_reg_pci_rx_u16(dev, 1, adrs_off);

		if (status & 0x8000)
			break;

		if (i > 100)
		{
			ret	= -ETIMEDOUT;
			break;
		}

		// Wait one timer tick before checking again.
		ret	= os_time_tick_sleep(1);

		if (ret)
		{
			// The sleep returned prematurely; it was signaled.
			// We've essentially been told to quit.
			ret	= -ECANCELED;
			break;
		}
	}

	if (ret == 0)
	{
		ptr[0]	= os_reg_pci_rx_u8(dev, 1, data_off + 0);
		ptr[1]	= os_reg_pci_rx_u8(dev, 1, data_off + 1);
		ptr[2]	= os_reg_pci_rx_u8(dev, 1, data_off + 2);
		ptr[3]	= os_reg_pci_rx_u8(dev, 1, data_off + 3);
	}

	return(ret);
}
#endif



//*****************************************************************************
#if defined(DEV_SUPPORTS_VPD)
static int _vpd_auto_locate(dev_data_t* dev)
{
	int			i;
	const u8*	ptr		= (void*) dev->vpd.image;
	int			size	= sizeof(dev->vpd.image) - 10;

	dev->vpd.image_offset	= -1;

	for (i = 0; i < size; i++, ptr++)
	{
		if ((ptr[0] == 0x82) &&
			(ptr[3] == 'G')  &&
			(ptr[4] == 'S')  &&
			(ptr[5] == 'C'))
		{
			dev->vpd.image_offset	= i;
			break;
		}
	}

	return(0);
}
#endif



//*****************************************************************************
#if defined(DEV_SUPPORTS_VPD)
static int _vpd_load(dev_data_t* dev)
{
	int		i;
	u32*	ptr;
	int		ret		= 0;
	int		size	= sizeof(dev->vpd.image) / 4;

	for (;;)	// A convenience loop.
	{
		if (dev->vpd.loaded)
			break;

		for (i = 0; (ret == 0) && (i < size); i += 4)
			ret	= _vpd_read_dword(dev, i, dev->vpd.image + i);

		dev->vpd.loaded	= 1;
		ptr				= (void*) dev->vpd.image;

		if (ptr[0] == 0xFFFFFFFF)
			dev->vpd.available	= 0;
		else
			_vpd_auto_locate(dev);

		break;
	}

	return(ret);
}
#endif



//*****************************************************************************
#if defined(DEV_SUPPORTS_VPD)
static void _long_string_copy(const u8* src, gsc_vpd_t* vpd)
{
	long	len;
	int		limit	= sizeof(vpd->data);

	len	= src[1] + 256L * src[2];
	len	= (len >= (limit - 1)) ? (limit - 1) : len;
	memcpy(vpd->data, src + 3, len);
	vpd->data[len]	= 0;
}
#endif



//*****************************************************************************
#if defined(DEV_SUPPORTS_VPD)
static void _long_string_skip(const u8** src)
{
	long	len;

	len		= src[0][1] + 256L * src[0][2];
	src[0]	+= len + 3;
}
#endif



//*****************************************************************************
#if defined(DEV_SUPPORTS_VPD)
static int _retrieve_vpd_r_field(const u8* src, u8 c1, u8 c2, gsc_vpd_t* vpd)
{
	int	len;
	int	limit	= sizeof(vpd->data);

	if (src[0] == 0x90)
	{
		// This is a VPD-R record.
		src	+= 3;	// Skip the header.

		for (;;)
		{
			if (src[0] == 0x78)	// End Tag
				break;

			len	= src[2];

			if ((src[0] == c1) && (src[1] == c2))
			{
				len	= (len >= (limit - 1)) ? (limit - 1) : len;
				memcpy(vpd->data, src + 3, len);
				vpd->data[len]	= 0;
			}

			src	+= 3 + len;	// Skip the header and the data.
		}
	}

	return(0);
}
#endif



//*****************************************************************************
#if defined(DEV_SUPPORTS_VPD)
static int _retrieve_model_number(dev_data_t* dev, gsc_vpd_t* vpd)
{
	const u8*	ptr;
	int			ret	= 0;

	vpd->reserved	= 0;
	vpd->data[0]	= 0;

	if (dev->vpd.image_offset >= 0)
		ptr	= dev->vpd.image + dev->vpd.image_offset;
	else
		ptr	= NULL;

	if (ptr)
	{
		_long_string_skip(&ptr);
		ret	= _retrieve_vpd_r_field(ptr, 'P', 'N', vpd);
	}

	return(ret);
}
#endif



//*****************************************************************************
#if defined(DEV_SUPPORTS_VPD)
static int _retrieve_name(dev_data_t* dev, gsc_vpd_t* vpd)
{
	const u8*	ptr;
	int			ret	= 0;

	vpd->reserved	= 0;
	vpd->data[0]	= 0;

	if (dev->vpd.image_offset >= 0)
		ptr	= dev->vpd.image + dev->vpd.image_offset;
	else
		ptr	= NULL;

	if (ptr)
	{
		_long_string_skip(&ptr);
		_long_string_copy(ptr, vpd);
	}

	return(ret);
}
#endif



//*****************************************************************************
#if defined(DEV_SUPPORTS_VPD)
static int _retrieve_serial_number(dev_data_t* dev, gsc_vpd_t* vpd)
{
	const u8*	ptr;
	int			ret	= 0;

	vpd->reserved	= 0;
	vpd->data[0]	= 0;

	if (dev->vpd.image_offset >= 0)
		ptr	= dev->vpd.image + dev->vpd.image_offset;
	else
		ptr	= NULL;

	if (ptr)
	{
		_long_string_skip(&ptr);
		ret	= _retrieve_vpd_r_field(ptr, 'S', 'N', vpd);
	}

	return(ret);
}
#endif



//*****************************************************************************
#if defined(DEV_SUPPORTS_VPD)
static int _vpd_initialize(dev_data_t* dev)
{
	u8	id;
	u8	next;
	u16	reg;

	for (;;)	// Convenience loop
	{
		if (dev->vpd.initialized)
			break;

		memset(&dev->vpd, 0, sizeof(dev->vpd));
		memset(dev->vpd.image, 0xFF, sizeof(dev->vpd.image));
		dev->vpd.initialized	= 1;

		// Locate the VPD optional feature.
		reg		= os_reg_pci_rx_u16(dev, 1, 0x34);	// per PCI spec
		next	= reg & 0xFF;

		for (;;)
		{
			if (next == 0)
				break;

			reg		= os_reg_pci_rx_u16(dev, 1, next);
			id		= reg & 0xFF;

			if (id == 0x03)
			{
				dev->vpd.available		= 1;
				dev->vpd.pci_reg_offset	= next;
				break;
			}

			next	= reg >> 8;
		}

		break;
	}

	return(0);
}
#endif



/******************************************************************************
*
*	Function:	gsc_vpd_retrieve
*
*	Purpose:
*
*		Retrieve vital product data from the board.
*
*	Arguments:
*
*		dev		The device data structure.
*
*		vpd		This indicates the specific data desired. This is also where
*				the retrieved data is stored.
*
*	Returned:
*
*		>= 0	The number of bytes transferred.
*		< 0		There was a problem and this is the error status.
*
******************************************************************************/

#if defined(DEV_SUPPORTS_VPD)
int gsc_vpd_read_ioctl(dev_data_t* dev, gsc_vpd_t* vpd)
{
	int	ret;

	for (;;)	// A convenience loop.
	{
		memset(vpd->data, 0, sizeof(vpd->data));

		ret	= _vpd_initialize(dev);

		if ((ret) || (dev->vpd.available == 0))
			break;

		ret	= _vpd_load(dev);

		if (ret)
			break;

		if (dev->vpd.available == 0)
		{
			vpd->data[0]	= 0;
			break;
		}

		switch (vpd->type)
		{
			default:

				vpd->data[0]	= 0;
				ret				= -EINVAL;
				break;

			case GSC_VPD_TYPE_MODEL_NUMBER:

				ret	= _retrieve_model_number(dev, vpd);
				break;

			case GSC_VPD_TYPE_NAME:

				ret	= _retrieve_name(dev, vpd);
				break;

			case GSC_VPD_TYPE_SERIAL_NUMBER:

				ret	= _retrieve_serial_number(dev, vpd);
				break;
		}

		break;
	}

	return(ret);
}
#endif


