// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AISS2AO2A2M/utils/fsamp_bdo.c $
// $Rev: 45184 $
// $Date: 2019-07-01 16:59:04 -0500 (Mon, 01 Jul 2019) $

// 16AISS2AO2A2M: Utilities: source file

#include "main.h"



// #defines *******************************************************************

#define	ABS(v)					(((v) < 0) ? (-(v)) : (v))
#define	LIMIT_MAX(v,max)		(((v) > (max)) ? (max) : (v))
#define	LIMIT_MIN(v,min)		(((v) < (min)) ? (min) : (v))
#define	LIMIT_RANGE(v,min,max)	(LIMIT_MIN((LIMIT_MAX((v),(max))),(min)))



/******************************************************************************
*
*	Function:	aiss2ao2a2m_fsamp_bdo_compute
*
*	Purpose:
*
*		Determine the best configuration for the specified output sample rate.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the board to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fsamp	The desired sample rate.
*
*		ndiv	The Ndiv value is put here. MUST NOT BE NULL!
*
*		rate	Report the sample rate achieved here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int aiss2ao2a2m_fsamp_bdo_compute(int fd, int index, int verbose, s32 fsamp, s32* ndiv, double* rate)
{
	double	delta;
	double	delta_best	= FLT_MAX;
	int		errs		= 0;
	double	fsamp_best	= 0;
	double	fsamp_got;
	s32		fsamp_max;			// Queried value.
	s32		fsamp_min;			// Queried value.
	double	fsamp_want;
	int		init		= 0;	// Have we performed the first run initialization?
	s32		mast;
	double	master;
	s32		nd;
	s32		nd_max;
	s32		nd_min;
	s32		ndiv_best	= 0;
	s32		ndiv_max;			// Queried value.
	s32		ndiv_min;			// Queried value.

	if (verbose)
	{
		gsc_label_index("BDO Sample Rate", index);
		gsc_label_long_comma(fsamp);
		printf(" S/S");
	}

	errs	+= aiss2ao2a2m_query(fd, -1, 0, AISS2AO2A2M_QUERY_BDO_FSAMP_MAX, &fsamp_max);
	errs	+= aiss2ao2a2m_query(fd, -1, 0, AISS2AO2A2M_QUERY_BDO_FSAMP_MIN, &fsamp_min);
	errs	+= aiss2ao2a2m_query(fd, -1, 0, AISS2AO2A2M_QUERY_BDO_NDIV_MAX, &ndiv_max);
	errs	+= aiss2ao2a2m_query(fd, -1, 0, AISS2AO2A2M_QUERY_BDO_NDIV_MIN, &ndiv_min);
	errs	+= aiss2ao2a2m_query(fd, -1, 0, AISS2AO2A2M_QUERY_MASTER_CLOCK, &mast);
	master	= (double) mast;

	fsamp_want	= (double) LIMIT_RANGE(fsamp, fsamp_min, fsamp_max);
	nd			= (s32) (master / fsamp_want);
	nd_min		= nd - 1;
	nd_max		= nd + 1;

	if (nd_max > ndiv_max)
	{
		nd_max	= ndiv_max;
		nd_min	= ndiv_max - 2;
	}

	if (nd_min < ndiv_min)
	{
		nd_min	= ndiv_min;
		nd_max	= ndiv_min + 2;
	}

	for (nd = nd_min; nd <= nd_max; nd++)
	{
		fsamp_got	= master / nd;
		delta		= fsamp_got - fsamp_want;

		if ((init == 0) || ((ABS(delta)) < (ABS(delta_best))))
		{
			init		= 1;
			delta_best	= delta;
			fsamp_best	= fsamp_got;
			ndiv_best	= nd;
		}
	}

	ndiv[0]		= ndiv_best;

	if (rate)
		rate[0]		= fsamp_best;

	if (verbose)
	{
		printf("%s  (", errs ? " FAIL <---" : "");
		printf("Fref ");
		gsc_label_long_comma((long) master);
		printf(", Ndiv ");
		gsc_label_long_comma((long) ndiv_best);
		printf(", Fsamp ");
		gsc_label_float_comma(fsamp_best, 1, 3);
		printf(")\n");
	}

	return(errs);
}


