// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AISS2AO2A2M/utils/fsamp_ai.c $
// $Rev: 43419 $
// $Date: 2018-08-24 17:33:36 -0500 (Fri, 24 Aug 2018) $

// 16AISS2AO2A2M: Utilities: source file

#include "main.h"



// #defines *******************************************************************

#define	ABS(v)					(((v) < 0) ? (-(v)) : (v))
#define	LIMIT_MAX(v,max)		(((v) > (max)) ? (max) : (v))
#define	LIMIT_MIN(v,min)		(((v) < (min)) ? (min) : (v))
#define	LIMIT_RANGE(v,min,max)	(LIMIT_MIN((LIMIT_MAX((v),(max))),(min)))



/******************************************************************************
*
*	Function:	_fsamp_ai_report
*
*	Purpose:
*
*		Determine the input sample rate.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		verbose	Work verbosely?
*
*		chan	The channel index of interest.
*
*		get		Store the sample rate here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

static int _fsamp_ai_report(int fd, int verbose, int chan, s32* get)
{
	s32		ai_enable;
	char	buf[80]		= "";
	s32		chan_sel;
	s32		chans;
	s32		clk_dir;
	int		errs		= 0;
	s32		fsamp		= 0;
	s32		gen_enable;
	s32		master;
	s32		ndiv;

	errs	+= aiss2ao2a2m_query(fd, -1, 0, AISS2AO2A2M_QUERY_AI_CHANS_QTY, &chans);
	errs	+= aiss2ao2a2m_query(fd, -1, 0, AISS2AO2A2M_QUERY_MASTER_CLOCK, &master);

	errs	+= aiss2ao2a2m_ai_buf_enable	(fd, -1, 0, -1, &ai_enable);
	errs	+= aiss2ao2a2m_ai_chan_sel		(fd, -1, 0, -1, &chan_sel);
	errs	+= aiss2ao2a2m_cbl_io_clk_dir	(fd, -1, 0, -1, &clk_dir);
	errs	+= aiss2ao2a2m_rate_a_gen_enable(fd, -1, 0, -1, &gen_enable);
	errs	+= aiss2ao2a2m_rate_a_gen_ndiv	(fd, -1, 0, -1, &ndiv);

	if (ndiv == 0)
		strcpy(buf, "0 S/S  (Rate-A Gen. Ndiv is zero)");

	if (gen_enable == AISS2AO2A2M_RATE_A_GEN_ENABLE_NO)
		strcpy(buf, "0 S/S  (Rate-A Gen. disabled)");

	if (clk_dir == AISS2AO2A2M_CBL_IO_CLK_DIR_IN)
		strcpy(buf, "0 S/S  (Using externl clock)");

	if (ai_enable == AISS2AO2A2M_AI_BUF_ENABLE_NO)
		strcpy(buf, "0 S/S  (Input disabled)");

	if ((chan >= 0) && ((chan_sel & (0x1 << chan)) == 0))
		strcpy(buf, "0 S/S  (Channel Disabled)");

	if (chan >= chans)
		strcpy(buf, "0 S/S  (Invalid channel index.)");

	if (errs)
		strcpy(buf, "0 S/S  FAIL <---  (service errors)");

	if (buf[0] == 0)
		fsamp	= master / ndiv;

	if (verbose == 0)
	{
	}
	else if (buf[0])
	{
		printf("%s\n", buf);
	}
	else
	{
		gsc_label_long_comma(fsamp);
		printf(" S/S  (Fref ");
		gsc_label_long_comma(master);
		printf(", Gen-A Ndiv ");
		gsc_label_long_comma(ndiv);
		printf(")\n");
	}

	if (get)
		get[0]	= fsamp;

	return(errs);
}



/******************************************************************************
*
*	Function:	aiss2ao2a2m_fsamp_ai_compute
*
*	Purpose:
*
*		Determine the best configuration for the specified input sample rate.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the board to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fsamp	The desired sample rate.
*
*		ndiv	The Ndiv value is put here. MUST NOT BE NULL!
*
*		rate	Report the sample rate achieved here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int aiss2ao2a2m_fsamp_ai_compute(int fd, int index, int verbose, s32 fsamp, s32* ndiv, double* rate)
{
	double	delta;
	double	delta_best	= FLT_MAX;
	int		errs		= 0;
	double	fsamp_best	= 0;
	double	fsamp_got;
	s32		fsamp_max;			// Queried value.
	s32		fsamp_min;			// Queried value.
	double	fsamp_want;
	int		init		= 0;	// Have we performed the first run initialization?
	s32		mast;
	double	master;
	s32		nd;
	s32		nd_max;
	s32		nd_min;
	s32		ndiv_best	= 0;
	s32		ndiv_max;			// Queried value.
	s32		ndiv_min;			// Queried value.

	if (verbose)
	{
		gsc_label_index("AI Sample Rate", index);
		gsc_label_long_comma(fsamp);
		printf(" S/S");
	}

	errs	+= aiss2ao2a2m_query(fd, -1, 0, AISS2AO2A2M_QUERY_AI_FSAMP_MAX, &fsamp_max);
	errs	+= aiss2ao2a2m_query(fd, -1, 0, AISS2AO2A2M_QUERY_AI_FSAMP_MIN, &fsamp_min);
	errs	+= aiss2ao2a2m_query(fd, -1, 0, AISS2AO2A2M_QUERY_MASTER_CLOCK, &mast);
	errs	+= aiss2ao2a2m_query(fd, -1, 0, AISS2AO2A2M_QUERY_AI_NDIV_MAX, &ndiv_max);
	errs	+= aiss2ao2a2m_query(fd, -1, 0, AISS2AO2A2M_QUERY_AI_NDIV_MIN, &ndiv_min);
	master	= (double) mast;

	fsamp_want	= (double) LIMIT_RANGE(fsamp, fsamp_min, fsamp_max);
	nd			= (s32) (master / fsamp_want);
	nd_min		= nd - 1;
	nd_max		= nd + 1;

	if (nd_max > ndiv_max)
	{
		nd_max	= ndiv_max;
		nd_min	= ndiv_max - 2;
	}

	if (nd_min < ndiv_min)
	{
		nd_min	= ndiv_min;
		nd_max	= ndiv_min + 2;
	}

	for (nd = nd_min; nd <= nd_max; nd++)
	{
		fsamp_got	= master / nd;
		delta		= fsamp_got - fsamp_want;

		if ((init == 0) || ((ABS(delta)) < (ABS(delta_best))))
		{
			init		= 1;
			delta_best	= delta;
			fsamp_best	= fsamp_got;
			ndiv_best	= nd;
		}
	}

	ndiv[0]		= ndiv_best;

	if (rate)
		rate[0]		= fsamp_best;

	if (verbose)
	{
		printf("%s  (", errs ? " FAIL <---" : "");
		printf("Fref ");
		gsc_label_long_comma((long) master);
		printf(", Ndiv ");
		gsc_label_long_comma((long) ndiv_best);
		printf(", Fsamp ");
		gsc_label_float_comma(fsamp_best, 1, 3);
		printf(")\n");
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	aiss2ao2a2m_fsamp_ai_report
*
*	Purpose:
*
*		Determine and report the input sample rate for the specified channel.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the board to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		chan	The index of the channel of interest.
*
*		get		Store the sample rate here, if non-NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int aiss2ao2a2m_fsamp_ai_report(int fd, int index, int verbose, int chan, s32* get)
{
	char	buf[64];
	int		errs;
	s32		fsamp;

	if (verbose)
	{
		if (chan >= 0)
			sprintf(buf, "AI Channel %d", chan);
		else
			strcpy(buf, "Sample Rate");

		gsc_label_index(buf, index);
	}

	errs	= _fsamp_ai_report(fd, verbose, chan, &fsamp);

	if (get)
		get[0]	= fsamp;

	return(errs);
}



/******************************************************************************
*
*	Function:	aiss2ao2a2m_fsamp_ai_report_all
*
*	Purpose:
*
*		Determine and report the input sample rate for all channels.
*
*	Arguments:
*
*		fd		The handle to use to access the driver.
*
*		index	The index of the board to access. Ignore if < 0.
*
*		verbose	Work verbosely?
*
*		fsamp	Report the overall sample rate here.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int aiss2ao2a2m_fsamp_ai_report_all(int fd, int index, int verbose, s32* fsamp)
{
	s32	chans;
	int	chan;
	int	errs	= 0;
	s32	sps;
	s32	total	= 0;

	if (verbose)
		gsc_label_index("AI Sample Rates", index);

	errs	+= aiss2ao2a2m_query(fd, -1, 0, AISS2AO2A2M_QUERY_AI_CHANS_QTY, &chans);

	if (errs)
	{
	}
	else if (chans == 0)
	{
		if (verbose)
			printf("(This board has no input channels.)\n");
	}
	else
	{
		if (verbose)
		{
			printf("\n");
			gsc_label_level_inc();
		}

		for (chan = 0; chan < chans; chan++)
		{
			sps		= 0;
			errs	+= aiss2ao2a2m_fsamp_ai_report(fd, -1, verbose, chan, &sps);
			total	+= sps;
		}

		if (verbose)
		{
			gsc_label("Overall AI Rate");
			gsc_label_long_comma(total);
			printf(" S/S\n");
			gsc_label_level_dec();
		}
	}

	if (fsamp)
		fsamp[0]	= total;

	return(errs);
}


