// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AISS2AO2A2M/samples/aout/main.c $
// $Rev: 45169 $
// $Date: 2019-07-01 14:07:47 -0500 (Mon, 01 Jul 2019) $

// 16AISS2AO2A2M: Sample Application: source file

#include "main.h"



// #defines *******************************************************************

#define	DEFAULT_PERIOD	15
#define	DEFAULT_RATE	100000L



// variables ******************************************************************

static	s32	_continuous		= 0;
static	s32	_def_index		= 0;
static	s32	_io_mode		= GSC_IO_MODE_PIO;
static	s32	_minute_limit	= 0;
static	s32	_period			= DEFAULT_PERIOD;
static	s32	_rate			= DEFAULT_RATE;
static	s32	_save			= 0;
static	s32	_test_limit		= 0;



//*****************************************************************************
static int _parse_args(int argc, char** argv)
{
	static const gsc_arg_item_t	list[]	=
	{
		//	type				var				arg			values					desc
		{	GSC_ARG_S32_FLAG,	&_io_mode,		"-bmdma",	{ GSC_IO_MODE_BMDMA },	"Transfer data using Block Mode DMA."					},
		{	GSC_ARG_S32_FLAG,	&_continuous,	"-c",		{ -1 },					"Continue testing until an error occurs."				},
		{	GSC_ARG_S32_FLAG,	&_continuous,	"-C",		{ +1 },					"Continue testing even if errors occur."				},
		{	GSC_ARG_S32_FLAG,	&_io_mode,		"-dmdma",	{ GSC_IO_MODE_DMDMA },	"Transfer data using Demand Mode DMA."					},
		{	GSC_ARG_S32_MIN,	&_rate,			"-f",		{ 1000000L },			"This specifies the master clock frequency in hertz."	},
		{	GSC_ARG_S32_MIN,	&_minute_limit,	"-m",		{ 1 },					"Continue testing for up to # minutes."					},
		{	GSC_ARG_S32_MIN,	&_test_limit,	"-n",		{ 1 },					"Continue testing for up to # iterations."				},
		{	GSC_ARG_S32_FLAG,	&_io_mode,		"-pio",		{ GSC_IO_MODE_PIO },	"Transfer data using PIO."								},
		{	GSC_ARG_S32_RANGE,	&_period,		"-p",		{ 1, 3600 },			"Select the data output period to # seconds."			},
		{	GSC_ARG_S32_RANGE,	&_rate,			"-r"	,	{ 10, 1000000L },		"Output data at # samples per second."					},
		{	GSC_ARG_S32_FLAG,	&_save,			"-s",		{ 1 },					"Save the pattern to aout.txt."							},
		{	GSC_ARG_DEV_INDEX,	&_def_index,	"index",	{ 0 },					"The zero based index of the device to access."			}
	};

	static const gsc_arg_set_t	set	=
	{
		/* name	*/	"aout",
		/* desc	*/	"Analog output sample application.",
		/* qty	*/	SIZEOF_ARRAY(list),
		/* list	*/	list
	};

	int	errs;

	errs	= gsc_args_parse(argc, argv, &set);

	return(errs);
}



//*****************************************************************************
static int _perform_tests(int fd)
{
	int	errs;

	errs	= aio_out(fd, _io_mode, _rate, _period, _save);

	return(errs);
}



//*****************************************************************************
static void _show_access_index(int index)
{
	gsc_label("Accessing Device");
	printf("%d\n", index);
}



//*****************************************************************************
static void _show_time_stamp(void)
{
	const char*	psz;
	struct tm*	stm;
	time_t		tt;

	time(&tt);
	stm	= localtime(&tt);
	psz	= (char*) asctime(stm);
	gsc_label("Time Stamp");
	printf("%s", psz);
}



/******************************************************************************
*
*	Function:	main
*
*	Purpose:
*
*		Control the overall flow of the application.
*
*	Arguments:
*
*		argc			The number of command line arguments.
*
*		argv			The list of command line arguments.
*
*	Returned:
*
*		EXIT_SUCCESS	No errors were encounterred.
*		EXIT_FAILURE	One or more problems were encounterred.
*
******************************************************************************/

int main(int argc, char** argv)
{
	int		errs	= 0;
	time_t	exec		= time(NULL);
	long	failures	= 0;
	int		fd			= 0;
	long	hours;
	long	mins;
	time_t	now;
	long	passes		= 0;
	int		qty			= 0;
	int		ret;
	long	secs;
	time_t	t_limit;
	time_t	test;
	long	tests		= 0;

	for (;;)	// This is a convenience loop.
	{
		gsc_label_init(29);
		test	= time(NULL);
		errs	= _parse_args(argc, argv);

		if (errs)
			break;

		// Introduction

		t_limit	= exec + (_minute_limit * 60);
		os_id_host();
		_show_time_stamp();
		_show_access_index(_def_index);
		errs	+= aiss2ao2a2m_init_util(1);
		errs	+= os_id_driver(aiss2ao2a2m_open, aiss2ao2a2m_read_ai, aiss2ao2a2m_close);
		errs	+= aiss2ao2a2m_count_boards(1, &qty);

		if ((errs) || (qty <= 0))
			break;

		// Open access to device

		errs	= aiss2ao2a2m_open_util(_def_index, 0, -1, 1, &fd);

		if (errs == 0)
		{
			// Device identification

			errs	= aiss2ao2a2m_id_device(fd, -1, 1);

			// Perform testing

			errs	+= _perform_tests(fd);

			// Close device access

			errs	+= aiss2ao2a2m_close_util(fd, -1, 1);
		}

		// End of processing.

		now	= time(NULL);
		tests++;

		if (errs)
		{
			failures++;
			printf(	"\nRESULTS: FAIL <---  (%d error%s)",
					errs,
					(errs == 1) ? "" : "s");
		}
		else
		{
			passes++;
			printf("\nRESULTS: PASS");
		}

		secs	= (long) now - test;
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(" (duration %ld:%ld:%02ld)\n", hours, mins, secs);

		secs	= now - exec;
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(	"SUMMARY: tests %ld, pass %ld, fail %ld"
				" (duration %ld:%ld:%02ld)\n\n",
				tests,
				passes,
				failures,
				hours,
				mins,
				secs);

		if (_continuous == 0)
			break;

		if ((_continuous < 0) && (errs))
			break;

		if ((_minute_limit > 0) && (now >= t_limit))
			break;

		if ((_test_limit > 0) && (tests >= _test_limit))
			break;
	}

	ret	= ((qty <= 0) || (errs) || (failures)) ? EXIT_FAILURE : EXIT_SUCCESS;
	return(ret);
}


