// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AISS2AO2A2M/samples/aout/aio.c $
// $Rev: 45169 $
// $Date: 2019-07-01 14:07:47 -0500 (Mon, 01 Jul 2019) $

// 16AISS2AO2A2M: Sample Application: source file

#include "main.h"



// #defines *******************************************************************

#define	SIZEOF_ARRAY(a)		(sizeof((a)) / sizeof((a)[0]))



// variables ******************************************************************

static	u32		_data[128 * 1024];
static	long	_loops;	// Number of times to send a block of data.
static	long	_qty;	// Number of samples to send per block.



//*****************************************************************************
static int _init_data(int chans, long rate, int seconds)
{
	u32*		dest;
	int			errs	= 0;
	int			i;
	long		period_len;	// The length of the pattern in samples.
	long		period_tot;	// Pattern samples total for all channels.
	long		period;		// Pattern length in x units.
	int			samp;
	float		samp_dat;	// Total sample size of the data buffer.
	float		samp_sec;	// Samples Per Second for all active channels
	float		samp_tot;	// Samples Total for all active channels
	const char*	units;

	gsc_label("Initializing Data");

	memset(_data, 0, sizeof(_data));
	samp_sec	= ((float) rate) * chans;
	samp_tot	= ((float) rate) * chans * seconds;
	samp_dat	= sizeof(_data) / 4;

	period_len	= (rate <= 100) ? rate				//      10-      100 ->    10-   100
				: (rate <= 1000) ? (rate / 2)		//     101-    1,000 ->    50-   500
				: (rate <= 10000) ? (rate / 5)		//   1,001-   10,000 ->   200- 2,000
				: (rate <= 100000) ? (rate / 10)	//  10,001-  100,000 -> 1,000-10,000
				: (rate / 100);						// 100,001-1,000,000 -> 1,000-10,000
	period_tot	= period_len * chans;
	period		= (long) (1000000000. / rate * period_len);
	units		= "ns";

	if ((period % 1000) == 0)
	{
		period	/= 1000;
		units	= "us";
	}

	if ((period % 1000) == 0)
	{
		period	/= 1000;
		units	= "ms";
	}

	if (samp_tot <= SIZEOF_ARRAY(_data))
	{
		// We can get the entire image in the buffer.
		_loops	= 1;
		_qty	= (long) samp_tot;
	}
	else if (samp_sec <= SIZEOF_ARRAY(_data))
	{
		// We can get one second of the image in the buffer.
		_loops	= seconds;
		_qty	= (long) samp_sec;
	}
	else if (period_tot <= SIZEOF_ARRAY(_data))
	{
		// We can get one period of the image in the buffer.
		_loops	= seconds * (samp_sec / period_tot);
		_qty	= period_tot;
	}
	else
	{
		// We can get only a fraction of a period of the image in the buffer.
		_qty	= (samp_dat / chans) * chans;
		_loops	= (samp_tot + _qty - 1) / _qty;
	}

	// Initialize the data buffer.
	// The data is encoded in OFFSET BINARY format.
	dest	= _data;;

	for (i = 0, samp = 0; i < _qty; i += chans, samp++)
	{
		samp	= (samp < period_len) ? samp : 0;

		// 1st channel: sawtooth wave
		*dest++	= (u16) (0xFFFF - (0xFFFF * samp / (period_len - 1)));

		// 2nd channel: ramp wave
		if (chans >= 2)
			*dest++	= (u16) (0xFFFF * samp / (period_len - 1));
	}

	printf("Done  (pattern length: %ld samples, %ld %s)\n", period_len, period, units);
	return(errs);
}



//*****************************************************************************
static int _save_data(int chans, int rate, int save)
{
	char		buf[64];
	int			errs	= 0;
	FILE*		file;
	int			i;
	const char*	name	= "aout.txt";

	for (;;)
	{

		if (save == 0)
			break;

		gsc_label("Saving Data");
		file	= fopen(name, "w+b");

		if (file == NULL)
		{
			printf("FAIL <---  (unable to create %s)\n", name);
			errs	= 1;
			break;
		}

		gsc_label_long_comma_buf(rate, buf);
		fprintf(file, "# Fsamp = %s S/S\n", buf);

		gsc_label_long_comma_buf(_qty, buf);
		fprintf(file, "# Pattern = %s S/C\n", buf);

		gsc_label_long_comma_buf(_loops, buf);
		fprintf(file, "# Output = %s pattern iterations\n", buf);

		gsc_label_long_comma_buf(_loops * _qty, buf);
		fprintf(file, "# Output = %s samples total\n", buf);

		fprintf(file, "# %s  %s\n", (chans > 0) ? "Chan 0 \\" : "", (chans > 1) ? "Chan 1 /" : "");
		fprintf(file, "# %s  %s\n", (chans > 0) ? "========"  : "", (chans > 1) ? "========" : "");

		for (i = 0; i < _qty; i++)
		{
			if ((i) && ((i % chans) == 0))
				fprintf(file, "\n");

			fprintf(file, "  %08lX", (long) _data[i]);
		}

		fprintf(file, "\n");
		fclose(file);

		if (errs == 0)
			printf("PASS  (%s)\n", name);

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _io_write(int fd, int errs, int chans)
{
	int			i;
	long		loop;
	s32			lvl;
	s32			lvl_last;
	char*		ptr;
	int			ret;
	int			send;
	int			sent;
	long long	total		= 0;

	gsc_label("Writing Data");

	for (;;)	// A convenience loop.
	{
		if (errs)
		{
			errs	= 0;
			printf("SKIPPED   (due to errors)\n");
			break;
		}

		errs	= 0;

		// Write the data to the device.

		for (loop = 0; loop < _loops; loop++)
		{
			send	= _qty * 4;
			ptr		= (char*) &_data;

			for (;;)
			{
				sent	= aiss2ao2a2m_write_ao(fd, ptr, send);

				if (sent <= 0)
					break;

				total	+= sent;

				if (sent == send)
					break;

				ptr		+= sent;
				send	-= sent;
			}

			if (sent <= 0)
				break;
		}

		if (sent < 0)
		{
			errs	= 1;
			printf("FAIL <---  (write error)\n");
		}
		else if (sent == 0)
		{
			printf("FAIL <---  (I/O timeout, ");
			gsc_label_long_comma(total / 4);
			printf(" samples)\n");
		}
		else
		{
			printf("PASS  (");
			gsc_label_long_comma(total / 4);
			printf(" samples)\n");
		}

		gsc_label("Wait For Completion");
		lvl_last	= -1;

		for (i = 0; send > 0; i++)
		{
			ret	= aiss2ao2a2m_ioctl(fd, AISS2AO2A2M_IOCTL_AO_BUF_LEVEL, &lvl);

			if (ret)
			{
				errs	= 1;
				printf("FAIL <---  (access error)\n");
				break;
			}

			if (lvl == 0)
			{
				printf("PASS\n");
				break;
			}

			if (lvl != lvl_last)
				i	= 0;

			if (i >= 100)
			{
				errs	= 1;
				printf(	"FAIL <---  (AO FIFO didn't empty. %ld sample%s left.)\n",
						(long) lvl,
						(lvl == 1) ? "" : "s");
				break;
			}

			os_sleep_ms(50);
			lvl_last	= lvl;
		}

		break;
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	aio_out
*
*	Purpose:
*
*		Ganerate Analog Output.
*
*	Arguments:
*
*		fd		The file descriptor for the board to access.
*
*		io_mode	Transfer data using this I/O Mode.
*
*		rate	Configure the board for this sample rate.
*
*		seconds	Generate data for this period of time.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

int aio_out(int fd, int io_mode, int rate, int seconds, int save)
{
	s32	chans	= 2;
	int	errs	= 0;
	s32	fsamp	= 0;

	errs	+= aiss2ao2a2m_query(fd, -1, 1, AISS2AO2A2M_QUERY_AO_CHANS_QTY, &chans);

	if (errs)
	{
	}
	else if (chans == 0)
	{
		gsc_label("Analog Output");
		printf("SKIPPED  (No output channels)\n");
	}
	else
	{
		errs	+= aiss2ao2a2m_config_ao		(fd, -1, 1, rate);
		errs	+= aiss2ao2a2m_ao_io_mode		(fd, -1, 1, io_mode, NULL);
		errs	+= aiss2ao2a2m_fsamp_ao_report	(fd, -1, 1, -1, &fsamp);

		if (errs)
		{
			gsc_label("Status");
			printf("ABORTING  (Errors reported)\n");
		}
		else if (fsamp < 1)
		{
			gsc_label("Status");
			printf("ABORTING  (Fsamp is invalid.)\n");
		}
		else
		{
			errs	+= _init_data(chans, rate, seconds);
			errs	+= _save_data(chans, rate, save);
			errs	+= aiss2ao2a2m_ao_buf_over_data	(fd, -1, 1, AISS2AO2A2M_AO_BUF_OVER_DATA_CLEAR, NULL);
			errs	+= aiss2ao2a2m_ao_buf_over_frame(fd, -1, 1, AISS2AO2A2M_AO_BUF_OVER_FRAME_CLEAR, NULL);
			errs	+= _io_write(fd, errs, chans);
			errs	+= aiss2ao2a2m_ao_buf_over_data	(fd, -1, 1, AISS2AO2A2M_AO_BUF_OVER_DATA_CHECK, NULL);
			errs	+= aiss2ao2a2m_ao_buf_over_frame(fd, -1, 1, AISS2AO2A2M_AO_BUF_OVER_FRAME_CHECK, NULL);
		}
	}

	return(errs);
}


