// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AISS2AO2A2M/driver/write_bdo.c $
// $Rev: 45168 $
// $Date: 2019-07-01 13:41:01 -0500 (Mon, 01 Jul 2019) $

// 16AISS2AO2A2M: Device Driver: source file

#include "main.h"



//*****************************************************************************
static void _dev_io_sw_init(dev_data_t* dev, dev_io_t* io)
{
	io->io_mode			= AISS2AO2A2M_BDO_IO_MODE_DEFAULT;
	io->overflow_check	= AISS2AO2A2M_BDO_IO_OVERFLOW_DEFAULT;
	io->timeout_s		= AISS2AO2A2M_BDO_IO_TIMEOUT_DEFAULT;
	io->underflow_check	= AISS2AO2A2M_BDO_IO_UNDERFLOW_DEFAULT;
}



//*****************************************************************************
static void _dev_io_close(dev_data_t* dev, dev_io_t* io)
{
	io->dma_channel	= NULL;
}



//*****************************************************************************
static void _dev_io_open(dev_data_t* dev, dev_io_t* io)
{
	_dev_io_sw_init(dev, io);
}



//*****************************************************************************
static int _dev_io_startup(dev_data_t* dev, dev_io_t* io)
{
	u32		bdobcr	= 0;
	long	ret		= 0;

	if ((io->overflow_check) || (io->underflow_check))
	{
		bdobcr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bdobcr_32);

		if ((io->overflow_check) && (bdobcr & D29))
			ret	= -EIO;

		if ((io->underflow_check) && (bdobcr & D28))
			ret	= -EIO;
	}

	return(ret);
}



//*****************************************************************************
static long _dev_pio_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	u32	bdobsr;
	u32	samples;
	u32	space;

	bdobsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bdobsr_32);
	samples	= bdobsr & 0x3FFFFF;
	space	= dev->cache.bdo_fifo_size - samples;
	bytes	= space * 4;
	return(bytes);
}



//*****************************************************************************
static long _dev_pio_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	long	qty;

	qty	= gsc_write_pio_work_32_bit(dev, io, mem, bytes, st_end);
	return(qty);
}



//*****************************************************************************
static long _dev_bmdma_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	// DMA is not supported
	return(-EACCES);
}



//*****************************************************************************
static long _dev_bmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	// DMA is not supported
	return(-EACCES);
}



//*****************************************************************************
static long _dev_dmdma_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	// DMA is not supported
	return(-EACCES);
}



//*****************************************************************************
static long _dev_dmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	// DMA is not supported
	return(-EACCES);
}



/******************************************************************************
*
*	Function:	dev_write_bdo_create
*
*	Purpose:
*
*		Perform a one-tine setup of the Write BDO streaming I/O structure.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*		io		The I/O structure for this stream.
*
*	Returned:
*
*		None.
*
******************************************************************************/

int dev_write_bdo_create(dev_data_t* dev, dev_io_t* io)
{
	int	ret;

	io->bytes_per_sample	= 4;
	io->io_reg_offset		= GSC_REG_OFFSET(AISS2AO2A2M_GSC_BDOBR);
	io->io_reg_vaddr		= dev->vaddr.gsc_bdobr_32;
	io->pio_threshold		= 32;

	io->dev_io_sw_init		= _dev_io_sw_init;
	io->dev_io_close		= _dev_io_close;
	io->dev_io_open			= _dev_io_open;
	io->dev_io_startup		= _dev_io_startup;
	io->dev_pio_available	= _dev_pio_available;
	io->dev_bmdma_available	= _dev_bmdma_available;
	io->dev_dmdma_available	= _dev_dmdma_available;
	io->dev_pio_xfer		= _dev_pio_xfer;
	io->dev_bmdma_xfer		= _dev_bmdma_xfer;
	io->dev_dmdma_xfer		= _dev_dmdma_xfer;

	io->wait.abort			= AISS2AO2A2M_WAIT_IO_BDO_TX_ABORT;
	io->wait.done			= AISS2AO2A2M_WAIT_IO_BDO_TX_DONE;
	io->wait.error			= AISS2AO2A2M_WAIT_IO_BDO_TX_ERROR;
	io->wait.timeout		= AISS2AO2A2M_WAIT_IO_BDO_TX_TIMEOUT;

	ret	= gsc_io_create(dev, io, 256L * 1024 * 4);
	return(ret);
}


