// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AISS2AO2A2M/driver/read_ai.c $
// $Rev: 45168 $
// $Date: 2019-07-01 13:41:01 -0500 (Mon, 01 Jul 2019) $

// 16AISS2AO2A2M: Device Driver: source file

#include "main.h"



//*****************************************************************************
static void _dev_io_sw_init(dev_data_t* dev, dev_io_t* io)
{
	io->io_mode			= AISS2AO2A2M_AI_IO_MODE_DEFAULT;
	io->overflow_check	= AISS2AO2A2M_AI_IO_OVERFLOW_DEFAULT;
	io->timeout_s		= AISS2AO2A2M_AI_IO_TIMEOUT_DEFAULT;
	io->underflow_check	= AISS2AO2A2M_AI_IO_UNDERFLOW_DEFAULT;
}



//*****************************************************************************
static void _dev_io_close(dev_data_t* dev, dev_io_t* io)
{
	io->dma_channel	= NULL;
}



//*****************************************************************************
static void _dev_io_open(dev_data_t* dev, dev_io_t* io)
{
	_dev_io_sw_init(dev, io);
}



//*****************************************************************************
static int _dev_io_startup(dev_data_t* dev, dev_io_t* io)
{
	u32		bcr;
	long	ret		= 0;

	if ((io->overflow_check) || (io->underflow_check))
	{
		bcr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bcr_32);

		if ((io->overflow_check) && (bcr & D15))
			ret	= -EIO;

		if ((io->underflow_check) && (bcr & D23))
			ret	= -EIO;
	}

	return(ret);
}



//*****************************************************************************
static long _dev_pio_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	u32	ibsr;
	u32	samples;

	ibsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ibsr_32);
	samples	= ibsr & 0x3FFFFF;
	bytes	= samples * 4;
	return(bytes);
}



//*****************************************************************************
static long _dev_pio_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	long	qty;

	qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	return(qty);
}



//*****************************************************************************
static long _dev_bmdma_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	u32	available;
	u32	ibsr;
	u32	ibtr;
	u32	threshold;

	ibsr		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ibsr_32);
	available	= ibsr & 0x3FFFFF;
	available	*= 4;

	if (available >= bytes)
	{
		// The desired amount is available.
	}
	else
	{
		ibtr		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ibtr_32);
		threshold	= (ibtr & 0x3FFFFF) * 4;

		if (available >= threshold)
			bytes	= available;
		else
			bytes	= 0;
	}

	return(bytes);
}



//*****************************************************************************
static long _dev_bmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	u32		dpr;
	u32		mode;
	long	qty;
	long	samples	= bytes / 4;

	if (samples < io->pio_threshold)
	{
		qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	}
	else
	{
		mode	= GSC_DMA_MODE_BLOCK_DMA
				| GSC_DMA_MODE_SIZE_32_BITS
				| GSC_DMA_MODE_INPUT_ENABLE
				| GSC_DMA_MODE_BURSTING_LOCAL
				| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
				| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
				| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		dpr		= GSC_DMA_DPR_BOARD_TO_HOST
				| GSC_DMA_DPR_END_OF_CHAIN
				| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty		= gsc_dma_perform(	dev,
									io,
									st_end,
									GSC_DMA_CAP_BMDMA_READ,
									mode,
									dpr,
									mem,
									bytes);
	}

	return(qty);
}



//*****************************************************************************
static long _dev_dmdma_available(dev_data_t* dev, dev_io_t* io, size_t bytes)
{
	return(bytes);
}



//*****************************************************************************
static long _dev_dmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			bytes,
	os_time_tick_t	st_end)
{
	u32		dpr;
	u32		mode;
	long	qty;
	long	samples	= bytes / 4;

	if (samples < io->pio_threshold)
	{
		qty	= gsc_read_pio_work_32_bit(dev, io, mem, bytes, st_end);
	}
	else
	{
		mode	= GSC_DMA_MODE_DM_DMA
				| GSC_DMA_MODE_SIZE_32_BITS
				| GSC_DMA_MODE_INPUT_ENABLE
				| GSC_DMA_MODE_BURSTING_LOCAL
				| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
				| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
				| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		dpr		= GSC_DMA_DPR_BOARD_TO_HOST
				| GSC_DMA_DPR_END_OF_CHAIN
				| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty		= gsc_dma_perform(	dev,
									io,
									st_end,
									GSC_DMA_CAP_DMDMA_READ,
									mode,
									dpr,
									mem,
									bytes);
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	dev_read_ai_create
*
*	Purpose:
*
*		Perform a one-tine setup of the Read Analog Input streaming I/O structure.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*		io		The I/O structure for this stream.
*
*	Returned:
*
*		None.
*
******************************************************************************/

int dev_read_ai_create(dev_data_t* dev, dev_io_t* io)
{
	int	ret;

	io->bytes_per_sample	= 4;
	io->io_reg_offset		= GSC_REG_OFFSET(AISS2AO2A2M_GSC_AIBR);
	io->io_reg_vaddr		= dev->vaddr.gsc_aibr_32;
	io->pio_threshold		= 32;

	io->dev_io_sw_init		= _dev_io_sw_init;
	io->dev_io_close		= _dev_io_close;
	io->dev_io_open			= _dev_io_open;
	io->dev_io_startup		= _dev_io_startup;
	io->dev_pio_available	= _dev_pio_available;
	io->dev_bmdma_available	= _dev_bmdma_available;
	io->dev_dmdma_available	= _dev_dmdma_available;
	io->dev_pio_xfer		= _dev_pio_xfer;
	io->dev_bmdma_xfer		= _dev_bmdma_xfer;
	io->dev_dmdma_xfer		= _dev_dmdma_xfer;

	io->wait.abort			= AISS2AO2A2M_WAIT_IO_AI_RX_ABORT;
	io->wait.done			= AISS2AO2A2M_WAIT_IO_AI_RX_DONE;
	io->wait.error			= AISS2AO2A2M_WAIT_IO_AI_RX_ERROR;
	io->wait.timeout		= AISS2AO2A2M_WAIT_IO_AI_RX_TIMEOUT;

	ret	= gsc_io_create(dev, io, 2L * 1024 * 1024 * 4);

	return(ret);
}


