// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AISS2AO2A2M/driver/ioctl.c $
// $Rev: 45168 $
// $Date: 2019-07-01 13:41:01 -0500 (Mon, 01 Jul 2019) $

// 16AISS2AO2A2M: Device Driver: source file

#include "main.h"



// #defines *******************************************************************

#define	AUTOCAL_AI_START	D3
#define	AUTOCAL_ALL_START	D28
#define	AUTOCAL_PASS		D29



//*****************************************************************************
static int _query(dev_data_t* dev, s32* arg)
{
	switch (arg[0])
	{
		default:								arg[0]	= AISS2AO2A2M_IOCTL_QUERY_ERROR;	break;
		case AISS2AO2A2M_QUERY_AI_CHANS_MAX:	arg[0]	= dev->cache.ai_chans_max;			break;
		case AISS2AO2A2M_QUERY_AI_CHANS_QTY:	arg[0]	= dev->cache.ai_chans_qty;			break;
		case AISS2AO2A2M_QUERY_AI_FIFO_SIZE:	arg[0]	= dev->cache.ai_fifo_size;			break;
		case AISS2AO2A2M_QUERY_AI_FSAMP_MAX:	arg[0]	= dev->cache.ai_fsamp_max;			break;
		case AISS2AO2A2M_QUERY_AI_FSAMP_MIN:	arg[0]	= dev->cache.ai_fsamp_min;			break;
		case AISS2AO2A2M_QUERY_AI_NDIV_MAX:		arg[0]	= dev->cache.ai_ndiv_max;			break;
		case AISS2AO2A2M_QUERY_AI_NDIV_MIN:		arg[0]	= dev->cache.ai_ndiv_min;			break;
		case AISS2AO2A2M_QUERY_AO_CHANS_MAX:	arg[0]	= dev->cache.ao_chans_max;			break;
		case AISS2AO2A2M_QUERY_AO_CHANS_QTY:	arg[0]	= dev->cache.ao_chans_qty;			break;
		case AISS2AO2A2M_QUERY_AO_FIFO_SIZE:	arg[0]	= dev->cache.ao_fifo_size;			break;
		case AISS2AO2A2M_QUERY_AO_FSAMP_MAX:	arg[0]	= dev->cache.ao_fsamp_max;			break;
		case AISS2AO2A2M_QUERY_AO_FSAMP_MIN:	arg[0]	= dev->cache.ao_fsamp_min;			break;
		case AISS2AO2A2M_QUERY_AO_NDIV_MAX:		arg[0]	= dev->cache.ao_ndiv_max;			break;
		case AISS2AO2A2M_QUERY_AO_NDIV_MIN:		arg[0]	= dev->cache.ao_ndiv_min;			break;
		case AISS2AO2A2M_QUERY_AUTOCAL_MS:		arg[0]	= dev->cache.autocal_ms;			break;
		case AISS2AO2A2M_QUERY_BDO_FIFO_SIZE:	arg[0]	= dev->cache.bdo_fifo_size;			break;
		case AISS2AO2A2M_QUERY_BDO_FSAMP_MAX:	arg[0]	= dev->cache.bdo_fsamp_max;			break;
		case AISS2AO2A2M_QUERY_BDO_FSAMP_MIN:	arg[0]	= dev->cache.bdo_fsamp_min;			break;
		case AISS2AO2A2M_QUERY_BDO_NDIV_MAX:	arg[0]	= dev->cache.bdo_ndiv_max;			break;
		case AISS2AO2A2M_QUERY_BDO_NDIV_MIN:	arg[0]	= dev->cache.bdo_ndiv_min;			break;
		case AISS2AO2A2M_QUERY_COUNT:			arg[0]	= AISS2AO2A2M_IOCTL_QUERY_LAST;		break;
		case AISS2AO2A2M_QUERY_DEVICE_TYPE:		arg[0]	= dev->board_type;					break;
		case AISS2AO2A2M_QUERY_INITIALIZE_MS:	arg[0]	= dev->cache.initialize_ms;			break;
		case AISS2AO2A2M_QUERY_MASTER_CLOCK:	arg[0]	= dev->cache.master_clock;			break;
		case AISS2AO2A2M_QUERY_RATE_GEN_QTY:	arg[0]	= dev->cache.rate_gen_qty;			break;
	}

	return(0);
}



//*****************************************************************************
int initialize_ioctl(dev_data_t* dev, void* arg)
{
	#define	INIT_START	D31

	int		i;
	long	ms;
	u32		reg;
	int		ret;

	// Manually wait for completion to insure that initialization completes
	// even if we get a signal, such as ^c.
	os_reg_mem_tx_u32(dev, dev->vaddr.gsc_bcr_32, INIT_START);
	ms	= dev->cache.initialize_ms + 5000;
	ret	= gsc_poll_u32(dev, ms, dev->vaddr.gsc_bcr_32, INIT_START, 0);
	reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bcr_32);

	if (ret)
	{
		printf(	"%s: INITIALIZATION TIMED OUT (%ld ms)\n",
				dev->model,
				(long) ms);
	}

	if (reg & INIT_START)
	{
		ret	= -EIO;
		printf(	"%s: INITIALIZATION DID NOT COMPLETE (%ld ms)\n",
				dev->model,
				(long) ms);
	}

	// Initialize the software settings.

	for (i = 0; i < DEV_IO_STREAM_QTY; i++)
	{
		if (dev->io.io_streams[i])
		{
			if (dev->io.io_streams[i]->dev_io_sw_init)
			{
				(dev->io.io_streams[i]->dev_io_sw_init)(dev, dev->io.io_streams[i]);
			}
		}
	}

	return(ret);
}



//*****************************************************************************
static int _autocal_all_start(dev_data_t* dev, void* arg)
{
	// Initiate auto-calibration.
	os_reg_mem_mx_u32(NULL, dev->vaddr.gsc_bcr_32, AUTOCAL_ALL_START, AUTOCAL_ALL_START);
	return(0);
}



//*****************************************************************************
static int _autocal_all(dev_data_t* dev, void* arg)
{
	long		ms;
	u32			reg;
	int			ret;
	os_sem_t	sem;
	gsc_wait_t	wt;

	// Enable the Autocal Done interrupt.
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_psr_32, D0, D0);

	// Wait for the local interrupt.
	os_sem_create(&sem);	// dummy, required for wait operations.
	ms				= dev->cache.autocal_ms + 5000;
	memset(&wt, 0, sizeof(wt));
	wt.flags		= GSC_WAIT_FLAG_INTERNAL;
	wt.gsc			= AISS2AO2A2M_WAIT_GSC_AUTO_CAL_DONE;
	wt.timeout_ms	= ms;
	ret				= gsc_wait_event(dev, &wt, _autocal_all_start, NULL, &sem);
	os_sem_destroy(&sem);

	if (wt.flags & GSC_WAIT_FLAG_TIMEOUT)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATE (ALL) TIMED OUT (%ld ms).\n",
				dev->model,
				(long) ms);
	}

	// Check the completion status.
	reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bcr_32);

	if (ret)
	{
	}
	else if (reg & AUTOCAL_ALL_START)
	{
		ret	= -EIO;
		printf(	"%s: AUTOCALIBRATE (ALL) DID NOT COMPLETE (%ld ms)\n",
				dev->model,
				(long) ms);
	}
	else if ((reg & AUTOCAL_PASS) == 0)
	{
		ret	= ret ? ret : -EIO;
		printf(	"%s: AUTOCALIBRATE (ALL) FAILED (%ld ms)\n",
				dev->model,
				(long) ms);
	}
	else
	{
		// Wait for settling.
		os_time_sleep_ms(100);
	}

	return(ret);
}



//*****************************************************************************
static int _autocal_ai_start(dev_data_t* dev, void* arg)
{
	// Initiate auto-calibration.
	os_reg_mem_mx_u32(NULL, dev->vaddr.gsc_bcr_32, AUTOCAL_AI_START, AUTOCAL_AI_START);
	return(0);
}



//*****************************************************************************
static int _autocal_ai(dev_data_t* dev, void* arg)
{
	long		ms;
	u32			reg;
	int			ret;
	os_sem_t	sem;
	gsc_wait_t	wt;

	// Enable the Autocal Done interrupt.
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_psr_32, D0, D0);

	// Wait for the local interrupt.
	os_sem_create(&sem);	// dummy, required for wait operations.
	ms				= dev->cache.autocal_ms + 5000;
	memset(&wt, 0, sizeof(wt));
	wt.flags		= GSC_WAIT_FLAG_INTERNAL;
	wt.gsc			= AISS2AO2A2M_WAIT_GSC_AUTO_CAL_DONE;
	wt.timeout_ms	= ms;
	ret				= gsc_wait_event(dev, &wt, _autocal_ai_start, NULL, &sem);
	os_sem_destroy(&sem);

	if (wt.flags & GSC_WAIT_FLAG_TIMEOUT)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATE (AI) TIMED OUT (%ld ms).\n",
				dev->model,
				(long) ms);
	}

	// Check the completion status.
	reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bcr_32);

	if (ret)
	{
	}
	else if (reg & AUTOCAL_AI_START)
	{
		ret	= -EIO;
		printf(	"%s: AUTOCALIBRATE (AI) DID NOT COMPLETE (%ld ms)\n",
				dev->model,
				(long) ms);
	}
	else if ((reg & AUTOCAL_PASS) == 0)
	{
		ret	= ret ? ret : -EIO;
		printf(	"%s: AUTOCALIBRATE (AI) FAILED (%ld ms)\n",
				dev->model,
				(long) ms);
	}
	else
	{
		// Wait for settling.
		os_time_sleep_ms(100);
	}

	return(ret);
}



//*****************************************************************************
static int _autocal_sts(dev_data_t* dev, s32* arg)
{
	u32	reg;

	reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bcr_32);

	if (reg & (AUTOCAL_ALL_START | AUTOCAL_AI_START))
		arg[0]	= AISS2AO2A2M_AUTOCAL_STS_ACTIVE;
	else if ((reg & AUTOCAL_PASS) == 0)
		arg[0]	= AISS2AO2A2M_AUTOCAL_STS_FAIL;
	else
		arg[0]	= AISS2AO2A2M_AUTOCAL_STS_PASS;

	return(0);
}



//*****************************************************************************
static int _ai_buf_clear(dev_data_t* dev, void* arg)
{
	long	ms		= (dev->io.ai_rx.timeout_s ? dev->io.ai_rx.timeout_s : 1) * 1000;
	int		ret;

	// Clear the buffer.
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bcr_32, D13, D13);

	// Wait for the bit to clear.
	ret	= gsc_poll_u32(dev, ms, dev->vaddr.gsc_bcr_32, D13, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_enable(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AI_BUF_ENABLE_NO,
		AISS2AO2A2M_AI_BUF_ENABLE_YES,
		-1	// terminate list
	};

	int	ret	= 0;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 12, 12);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_level(dev_data_t* dev, s32* arg)
{
	arg[0]	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ibsr_32);
	arg[0]	&= 0x3FFFFF;
	return(0);
}



//*****************************************************************************
static int _ai_buf_overflow(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AI_BUF_OVERFLOW_NO,
		AISS2AO2A2M_AI_BUF_OVERFLOW_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 15, 15);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_underflow(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AI_BUF_UNDERFLOW_NO,
		AISS2AO2A2M_AI_BUF_UNDERFLOW_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 23, 23);
	return(ret);
}



//*****************************************************************************
static int _ai_burst_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AI_BURST_ENABLE_NO,
		AISS2AO2A2M_AI_BURST_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 9, 9);
	return(ret);
}



//*****************************************************************************
static int _ai_burst_size(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFFFF, dev->vaddr.gsc_icr_32, 23, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_burst_status(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AI_BURST_STATUS_IDLE,
		AISS2AO2A2M_AI_BURST_STATUS_BUSY,
		-1	// terminate list
	};

	int		ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 10, 10);
	return(ret);
}



//*****************************************************************************
static int _ai_chan_0_range(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AI_CHAN_0_RANGE_2_5V,
		AISS2AO2A2M_AI_CHAN_0_RANGE_5V,
		AISS2AO2A2M_AI_CHAN_0_RANGE_10V,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 5, 4);
	return(ret);
}



//*****************************************************************************
static int _ai_chan_1_range(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AI_CHAN_1_RANGE_2_5V,
		AISS2AO2A2M_AI_CHAN_1_RANGE_5V,
		AISS2AO2A2M_AI_CHAN_1_RANGE_10V,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 7, 6);
	return(ret);
}



//*****************************************************************************
static int _ai_chan_sel(dev_data_t* dev, void* arg)
{
	int	ret;

	if (dev->cache.ai_chans_qty == 1)
		ret	= gsc_s32_range_reg(dev, arg, 0, 0x1, dev->vaddr.gsc_icr_32, 24, 24);
	else
		ret	= gsc_s32_range_reg(dev, arg, 0, 0x3, dev->vaddr.gsc_icr_32, 25, 24);

	return(ret);
}



//*****************************************************************************
static int _ai_io_abort(dev_data_t* dev, s32* arg)
{
	arg[0]	= gsc_read_abort_active_xfer(dev, &dev->io.ai_rx);
	return(0);
}



//*****************************************************************************
static int _ai_io_mode(dev_data_t* dev, s32* arg)
{
	static const s32	list[]	=
	{
		GSC_IO_MODE_PIO,
		GSC_IO_MODE_BMDMA,
		GSC_IO_MODE_DMDMA,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.ai_rx.io_mode);
	return(ret);
}



//*****************************************************************************
static int _ai_io_overflow(dev_data_t* dev, void* arg)
{
	static const s32	list[]	=
	{
		AISS2AO2A2M_AI_IO_OVERFLOW_IGNORE,
		AISS2AO2A2M_AI_IO_OVERFLOW_CHECK,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.ai_rx.overflow_check);
	return(ret);
}



//*****************************************************************************
static int _ai_io_timeout(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_var(
			arg,
			AISS2AO2A2M_AI_IO_TIMEOUT_MIN,
			AISS2AO2A2M_AI_IO_TIMEOUT_INFINITE,
			&dev->io.ai_rx.timeout_s);
	return(ret);
}



//*****************************************************************************
static int _ai_io_underflow(dev_data_t* dev, void* arg)
{
	static const s32	list[]	=
	{
		AISS2AO2A2M_AI_IO_UNDERFLOW_IGNORE,
		AISS2AO2A2M_AI_IO_UNDERFLOW_CHECK,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.ai_rx.underflow_check);
	return(ret);
}



//*****************************************************************************
static int _ai_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AI_MODE_DIFF,
		AISS2AO2A2M_AI_MODE_SINGLE,
		AISS2AO2A2M_AI_MODE_ZERO,
		AISS2AO2A2M_AI_MODE_VREF,
		AISS2AO2A2M_AI_MODE_AO_0,
		AISS2AO2A2M_AI_MODE_AO_1,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 2, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_sw_clock(dev_data_t* dev, void* arg)
{
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bcr_32, D8, D8);
	return(0);
}



//*****************************************************************************
static int _ai_sw_trigger(dev_data_t* dev, void* arg)
{
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bcr_32, D11, D11);
	return(0);
}



//*****************************************************************************
static int _ai_thresh_lvl(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0x3FFFFF, dev->vaddr.gsc_ibtr_32, 21, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_thresh_sts(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AI_THRESH_STS_CLEAR,
		AISS2AO2A2M_AI_THRESH_STS_SET,
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 14, 14);
	return(ret);
}



//*****************************************************************************
static int _ao_access_mode(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_ACCESS_MODE_REG,
		AISS2AO2A2M_AO_ACCESS_MODE_FIFO,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 19, 19);
	return(ret);
}



//*****************************************************************************
static int _ao_buf_clear(dev_data_t* dev, void* arg)
{
	int	ret;

	// This action clears the buffer.
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_boor_32, D11, D11);

	// The bit should clear quickly.
	ret	= gsc_poll_u32(dev, 1, dev->vaddr.gsc_boor_32, D11, 0);

	// This action clears the buffer overflow and frame overflow bits.
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_boor_32, 0, D16 | D17);
	return(ret);
}



//*****************************************************************************
static int _ao_buf_level(dev_data_t* dev, s32* arg)
{
	arg[0]	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_obsr_32);
	arg[0]	&= 0x3FFFFF;
	return(0);
}



//*****************************************************************************
static int _ao_buf_load_req(dev_data_t* dev, s32* arg)
{
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_boor_32, D9, D9);
	return(0);
}



//*****************************************************************************
static int _ao_buf_load_sts(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_BUF_LOAD_STS_BUSY,
		AISS2AO2A2M_AO_BUF_LOAD_STS_READY,
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_boor_32, 10, 10);
	return(ret);
}



//*****************************************************************************
static int _ao_buf_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_BUF_MODE_OPEN,
		AISS2AO2A2M_AO_BUF_MODE_CIRC,
		-1	// terminate list
	};

	int		ret;

	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_boor_32, 8, 8);
	return(ret);
}



//*****************************************************************************
static int _ao_buf_over_data(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_BUF_OVER_DATA_NO,
		AISS2AO2A2M_AO_BUF_OVER_DATA_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_boor_32, 16, 16);
	return(ret);
}



//*****************************************************************************
static int _ao_buf_over_frame(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_BUF_OVER_FRAME_NO,
		AISS2AO2A2M_AO_BUF_OVER_FRAME_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_boor_32, 17, 17);
	return(ret);
}



//*****************************************************************************
static int _ao_buf_status(dev_data_t* dev, s32* arg)
{
	u32	boor;

	boor	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_boor_32);

	if (boor & D12)
		arg[0]	= AISS2AO2A2M_AO_BUF_STATUS_EMPTY;
	else if (boor & D15)
		arg[0]	= AISS2AO2A2M_AO_BUF_STATUS_FULL;
	else if (boor & D13)
		arg[0]	= AISS2AO2A2M_AO_BUF_STATUS_ABOVE;
	else
		arg[0]	= AISS2AO2A2M_AO_BUF_STATUS_AT_LOW;

	return(0);
}



//*****************************************************************************
static int _ao_burst_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_BURST_ENABLE_NO,
		AISS2AO2A2M_AO_BURST_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_boor_32, 19, 19);
	return(ret);
}



//*****************************************************************************
static int _ao_burst_status(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_BURST_STATUS_BUSY,
		AISS2AO2A2M_AO_BURST_STATUS_READY,
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_boor_32, 18, 18);
	return(ret);
}



//*****************************************************************************
static int _ao_burst_sync_ai(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_BURST_SYNC_AI_NO,
		AISS2AO2A2M_AO_BURST_SYNC_AI_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_boor_32, 21, 21);
	return(ret);
}



//*****************************************************************************
static int _ao_chan_sel(dev_data_t* dev, void* arg)
{
	int	ret;

	if (dev->cache.ao_chans_qty == 2)
		ret	= gsc_s32_range_reg(dev, arg, 0, 0x3, dev->vaddr.gsc_boor_32, 1, 0);
	else
		ret	= -EINVAL;

	return(ret);
}



//*****************************************************************************
static int _ao_clock_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_CLOCK_ENABLE_NO,
		AISS2AO2A2M_AO_CLOCK_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_boor_32, 5, 5);
	return(ret);
}



//*****************************************************************************
static int _ao_clock_source(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_CLOCK_SOURCE_EXT,
		AISS2AO2A2M_AO_CLOCK_SOURCE_INT,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_boor_32, 4, 4);
	return(ret);
}



//*****************************************************************************
static int _ao_clock_status(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_CLOCK_STATUS_BUSY,
		AISS2AO2A2M_AO_CLOCK_STATUS_READY,
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_boor_32, 6, 6);
	return(ret);
}



//*****************************************************************************
static int _ao_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_ENABLE_NO,
		AISS2AO2A2M_AO_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_boor_32, 14, 14);
	return(ret);
}



//*****************************************************************************
static int _ao_io_abort(dev_data_t* dev, s32* arg)
{
	arg[0]	= gsc_write_abort_active_xfer(dev, &dev->io.ao_tx);
	return(0);
}



//*****************************************************************************
static int _ao_io_mode(dev_data_t* dev, s32* arg)
{
	static const s32	list[]	=
	{
		GSC_IO_MODE_PIO,
		GSC_IO_MODE_BMDMA,
		GSC_IO_MODE_DMDMA,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.ao_tx.io_mode);
	return(ret);
}



//*****************************************************************************
static int _ao_io_overflow(dev_data_t* dev, void* arg)
{
	static const s32	list[]	=
	{
		AISS2AO2A2M_AO_IO_OVERFLOW_IGNORE,
		AISS2AO2A2M_AO_IO_OVERFLOW_CHECK,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.ao_tx.overflow_check);
	return(ret);
}



//*****************************************************************************
static int _ao_io_timeout(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_var(
			arg,
			AISS2AO2A2M_AO_IO_TIMEOUT_MIN,
			AISS2AO2A2M_AO_IO_TIMEOUT_INFINITE,
			&dev->io.ao_tx.timeout_s);
	return(ret);
}



//*****************************************************************************
static int _ao_output_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_OUTPUT_MODE_SEQ,
		AISS2AO2A2M_AO_OUTPUT_MODE_SIMUL,
		-1	// terminate list
	};

	int	ret;

	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 18, 18);
	return(ret);
}



//*****************************************************************************
static int _ao_range(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_RANGE_2_5V,
		AISS2AO2A2M_AO_RANGE_5V,
		AISS2AO2A2M_AO_RANGE_10V,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 17, 16);
	return(ret);
}



//*****************************************************************************
static int _ao_sw_clock(dev_data_t* dev, void* arg)
{
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bcr_32, D20, D20);
	return(0);
}



//*****************************************************************************
static int _ao_sw_trigger(dev_data_t* dev, void* arg)
{
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_boor_32, D20, D20);
	return(0);
}



//*****************************************************************************
static int _ao_thresh_lvl(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0x3FFFFF, dev->vaddr.gsc_obtr_32, 21, 0);
	return(ret);
}



//*****************************************************************************
static int _ao_thresh_sts(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_AO_THRESH_STS_CLEAR,
		AISS2AO2A2M_AO_THRESH_STS_SET,
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_obtr_32, 24, 24);
	return(ret);
}



//*****************************************************************************
static int _bdo_access_mode(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_BDO_ACCESS_MODE_FIFO,
		AISS2AO2A2M_BDO_ACCESS_MODE_REG,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bdobcr_32, 27, 27);
	return(ret);
}



//*****************************************************************************
static int _bdo_buf_clear(dev_data_t* dev, void* arg)
{
	long	ms		= (dev->io.ai_rx.timeout_s ? dev->io.ai_rx.timeout_s : 1) * 1000;
	int		ret;

	// Clear the buffer.
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bdobcr_32, D20, D20);

	// Wait for the bit to clear.
	ret	= gsc_poll_u32(dev, ms, dev->vaddr.gsc_bdobcr_32, D20, 0);
	return(ret);
}



//*****************************************************************************
static int _bdo_buf_enable(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_BDO_BUF_ENABLE_NO,
		AISS2AO2A2M_BDO_BUF_ENABLE_YES,
		-1	// terminate list
	};

	int	ret	= 0;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bdobcr_32, 21, 21);
	return(ret);
}



//*****************************************************************************
static int _bdo_buf_level(dev_data_t* dev, s32* arg)
{
	arg[0]	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bdobsr_32);
	arg[0]	&= 0x3FFFF;
	return(0);
}



//*****************************************************************************
static int _bdo_buf_overflow(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_BDO_BUF_OVERFLOW_NO,
		AISS2AO2A2M_BDO_BUF_OVERFLOW_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bdobcr_32, 29, 29);
	return(ret);
}



//*****************************************************************************
static int _bdo_buf_underflow(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_BDO_BUF_UNDERFLOW_NO,
		AISS2AO2A2M_BDO_BUF_UNDERFLOW_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bdobcr_32, 28, 28);
	return(ret);
}



//*****************************************************************************
static int _bdo_clock_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_BDO_CLOCK_ENABLE_NO,
		AISS2AO2A2M_BDO_CLOCK_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bdobcr_32, 22, 22);
	return(ret);
}



//*****************************************************************************
static int _bdo_clock_source(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_BDO_CLOCK_SOURCE_EXT,
		AISS2AO2A2M_BDO_CLOCK_SOURCE_INT,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bdobcr_32, 25, 25);
	return(ret);
}



//*****************************************************************************
static int _bdo_io_abort(dev_data_t* dev, s32* arg)
{
	arg[0]	= gsc_write_abort_active_xfer(dev, &dev->io.bdo_tx);
	return(-1);
}



//*****************************************************************************
static int _bdo_io_mode(dev_data_t* dev, s32* arg)
{
	static const s32	list[]	=
	{
		GSC_IO_MODE_PIO,
		// BDO doesn't support DMA.
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.bdo_tx.io_mode);
	return(ret);
}



//*****************************************************************************
static int _bdo_io_overflow(dev_data_t* dev, void* arg)
{
	static const s32	list[]	=
	{
		AISS2AO2A2M_BDO_IO_OVERFLOW_IGNORE,
		AISS2AO2A2M_BDO_IO_OVERFLOW_CHECK,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.bdo_tx.overflow_check);
	return(ret);
}



//*****************************************************************************
static int _bdo_io_timeout(dev_data_t* dev, void* arg)
{
	int	ret;

	ret		= gsc_s32_range_var(
				arg,
				AISS2AO2A2M_BDO_IO_TIMEOUT_MIN,
				AISS2AO2A2M_BDO_IO_TIMEOUT_INFINITE,
				&dev->io.bdo_tx.timeout_s);
	return(ret);
}



//*****************************************************************************
static int _bdo_io_underflow(dev_data_t* dev, void* arg)
{
	static const s32	list[]	=
	{
		AISS2AO2A2M_BDO_IO_UNDERFLOW_IGNORE,
		AISS2AO2A2M_BDO_IO_UNDERFLOW_CHECK,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.bdo_tx.underflow_check);
	return(ret);
}



//*****************************************************************************
static int _bdo_rate_gen_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_BDO_RATE_GEN_ENABLE_NO,
		AISS2AO2A2M_BDO_RATE_GEN_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bdobcr_32, 26, 26);
	return(ret);
}



//*****************************************************************************
static int _bdo_rate_gen_ndiv(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(
			dev,
			arg,
			dev->cache.bdo_ndiv_min,
			dev->cache.bdo_ndiv_max,
			dev->vaddr.gsc_bdorgr_32,
			23,
			0);
	return(ret);
}



//*****************************************************************************
static int _bdo_sw_clock(dev_data_t* dev, void* arg)
{
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bdobcr_32, D23, D23);
	return(0);
}



//*****************************************************************************
static int _bdo_thresh_lvl(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFFF, dev->vaddr.gsc_bdobcr_32, 19, 0);
	return(ret);
}



//*****************************************************************************
static int _bdo_thresh_sts(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_BDO_THRESH_STS_CLEAR,
		AISS2AO2A2M_BDO_THRESH_STS_SET,
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bdobcr_32, 24, 24);
	return(ret);
}



//*****************************************************************************
static int _cbl_in_clk_io(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_CBL_IN_CLK_IO_FUNC,
		AISS2AO2A2M_CBL_IN_CLK_IO_OUT_0,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_acr_32, 0, 0);
	return(ret);
}



//*****************************************************************************
static int _cbl_io_clk_dir(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_CBL_IO_CLK_DIR_IN,
		AISS2AO2A2M_CBL_IO_CLK_DIR_OUT,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 24, 24);
	return(ret);
}



//*****************************************************************************
static int _cbl_out_clk_io(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_CBL_OUT_CLK_IO_FUNC,
		AISS2AO2A2M_CBL_OUT_CLK_IO_OUT_0,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_acr_32, 2, 2);
	return(ret);
}



//*****************************************************************************
static int _cbl_trig_io(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_CBL_TRIG_IO_FUNC,
		AISS2AO2A2M_CBL_TRIG_IO_OUT_0,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_acr_32, 1, 1);
	return(ret);
}



//*****************************************************************************
static int _data_format(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_DATA_FORMAT_2S_COMP,
		AISS2AO2A2M_DATA_FORMAT_OFF_BIN,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 25, 25);
	return(ret);
}



//*****************************************************************************
static int _dio_dir(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_DIO_DIR_IN,
		AISS2AO2A2M_DIO_DIR_OUT,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_diopr_32, 8, 8);
	return(ret);
}



//*****************************************************************************
static int _dio_read(dev_data_t* dev, s32* arg)
{
	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_range_reg(dev, arg, 0, 0x3F, dev->vaddr.gsc_diopr_32, 7, 0);
	return(ret);
}



//*****************************************************************************
static int _dio_write(dev_data_t* dev, s32* arg)
{
	u32	bdobcr;
	u32	lb		= 0xFF & arg[0];
	u32	ub		= 0xFF & (arg[0] >> 8);

	bdobcr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bdobcr_32);

	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_diopr_32, lb, 0xFF);

	if (bdobcr & D27)
		os_reg_mem_tx_u32(dev, dev->vaddr.gsc_bdobr_32, ub);

	return(0);
}



//*****************************************************************************
static int _irq_enable(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0x7FFF, dev->vaddr.gsc_psr_32, 14, 0);
	return(ret);
}



//*****************************************************************************
static int _rate_a_gen_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_RATE_A_GEN_ENABLE_NO,
		AISS2AO2A2M_RATE_A_GEN_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 26, 26);
	return(ret);
}



//*****************************************************************************
static int _rate_a_gen_ndiv(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(
			dev,
			arg,
			dev->cache.ai_ndiv_min,
			dev->cache.ai_ndiv_max,
			dev->vaddr.gsc_ragr_32,
			23,
			0);
	return(ret);
}



//*****************************************************************************
static int _rate_b_gen_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_RATE_B_GEN_ENABLE_NO,
		AISS2AO2A2M_RATE_B_GEN_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 27, 27);
	return(ret);
}



//*****************************************************************************
static int _rate_b_gen_ndiv(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(
			dev,
			arg,
			dev->cache.ai_ndiv_min,
			dev->cache.ai_ndiv_max,
			dev->vaddr.gsc_rbgr_32,
			23,
			0);
	return(ret);
}



//*****************************************************************************
static int _rate_c_gen_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_RATE_C_GEN_ENABLE_NO,
		AISS2AO2A2M_RATE_C_GEN_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 22, 22);
	return(ret);
}



//*****************************************************************************
static int _rate_c_gen_ndiv(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(
			dev,
			arg,
			dev->cache.ao_ndiv_min,
			dev->cache.ao_ndiv_max,
			dev->vaddr.gsc_rcgr_32,
			23,
			0);
	return(ret);
}



//*****************************************************************************
static int _trigger_dir(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AISS2AO2A2M_TRIGGER_DIR_IN,
		AISS2AO2A2M_TRIGGER_DIR_OUT,
		-1	// terminate list
	};

	int	ret;

	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bcr_32, 21, 21);
	return(ret);
}



// variables *******************************************************************

const gsc_ioctl_t	dev_ioctl_list[]	=
{
	{ AISS2AO2A2M_IOCTL_REG_READ,				(void*) gsc_reg_read_ioctl		},
	{ AISS2AO2A2M_IOCTL_REG_WRITE,				(void*) gsc_reg_write_ioctl		},
	{ AISS2AO2A2M_IOCTL_REG_MOD,				(void*) gsc_reg_mod_ioctl		},
	{ AISS2AO2A2M_IOCTL_QUERY,					(void*) _query					},
	{ AISS2AO2A2M_IOCTL_INITIALIZE,				(void*) initialize_ioctl		},
	{ AISS2AO2A2M_IOCTL_AUTOCAL_ALL,			(void*) _autocal_all			},
	{ AISS2AO2A2M_IOCTL_AUTOCAL_AI,				(void*) _autocal_ai				},
	{ AISS2AO2A2M_IOCTL_AUTOCAL_STS,			(void*) _autocal_sts			},

	{ AISS2AO2A2M_IOCTL_AI_BUF_CLEAR,			(void*) _ai_buf_clear			},
	{ AISS2AO2A2M_IOCTL_AI_BUF_ENABLE,			(void*) _ai_buf_enable			},
	{ AISS2AO2A2M_IOCTL_AI_BUF_LEVEL,			(void*) _ai_buf_level			},
	{ AISS2AO2A2M_IOCTL_AI_BUF_OVERFLOW,		(void*) _ai_buf_overflow		},
	{ AISS2AO2A2M_IOCTL_AI_BUF_UNDERFLOW,		(void*) _ai_buf_underflow		},
	{ AISS2AO2A2M_IOCTL_AI_BURST_ENABLE,		(void*) _ai_burst_enable		},
	{ AISS2AO2A2M_IOCTL_AI_BURST_SIZE,			(void*) _ai_burst_size			},
	{ AISS2AO2A2M_IOCTL_AI_BURST_STATUS,		(void*) _ai_burst_status		},
	{ AISS2AO2A2M_IOCTL_AI_CHAN_0_RANGE,		(void*) _ai_chan_0_range		},
	{ AISS2AO2A2M_IOCTL_AI_CHAN_1_RANGE,		(void*) _ai_chan_1_range		},
	{ AISS2AO2A2M_IOCTL_AI_CHAN_SEL,			(void*) _ai_chan_sel			},
	{ AISS2AO2A2M_IOCTL_AI_IO_ABORT,			(void*) _ai_io_abort			},
	{ AISS2AO2A2M_IOCTL_AI_IO_MODE,				(void*) _ai_io_mode				},
	{ AISS2AO2A2M_IOCTL_AI_IO_OVERFLOW,			(void*) _ai_io_overflow			},
	{ AISS2AO2A2M_IOCTL_AI_IO_TIMEOUT,			(void*) _ai_io_timeout			},
	{ AISS2AO2A2M_IOCTL_AI_IO_UNDERFLOW,		(void*) _ai_io_underflow		},
	{ AISS2AO2A2M_IOCTL_AI_MODE,				(void*) _ai_mode				},
	{ AISS2AO2A2M_IOCTL_AI_SW_CLOCK,			(void*) _ai_sw_clock			},
	{ AISS2AO2A2M_IOCTL_AI_SW_TRIGGER,			(void*) _ai_sw_trigger			},
	{ AISS2AO2A2M_IOCTL_AI_THRESH_LVL,			(void*) _ai_thresh_lvl			},
	{ AISS2AO2A2M_IOCTL_AI_THRESH_STS,			(void*) _ai_thresh_sts			},

	{ AISS2AO2A2M_IOCTL_AO_ACCESS_MODE,			(void*) _ao_access_mode			},
	{ AISS2AO2A2M_IOCTL_AO_BUF_CLEAR,			(void*) _ao_buf_clear			},
	{ AISS2AO2A2M_IOCTL_AO_BUF_LEVEL,			(void*) _ao_buf_level			},
	{ AISS2AO2A2M_IOCTL_AO_BUF_LOAD_REQ,		(void*) _ao_buf_load_req		},
	{ AISS2AO2A2M_IOCTL_AO_BUF_LOAD_STS,		(void*) _ao_buf_load_sts		},
	{ AISS2AO2A2M_IOCTL_AO_BUF_MODE,			(void*) _ao_buf_mode			},
	{ AISS2AO2A2M_IOCTL_AO_BUF_OVER_DATA,		(void*) _ao_buf_over_data		},
	{ AISS2AO2A2M_IOCTL_AO_BUF_OVER_FRAME,		(void*) _ao_buf_over_frame		},
	{ AISS2AO2A2M_IOCTL_AO_BUF_STATUS,			(void*) _ao_buf_status			},
	{ AISS2AO2A2M_IOCTL_AO_BURST_ENABLE,		(void*) _ao_burst_enable		},
	{ AISS2AO2A2M_IOCTL_AO_BURST_STATUS,		(void*) _ao_burst_status		},
	{ AISS2AO2A2M_IOCTL_AO_BURST_SYNC_AI,		(void*) _ao_burst_sync_ai		},
	{ AISS2AO2A2M_IOCTL_AO_CHAN_SEL,			(void*) _ao_chan_sel			},
	{ AISS2AO2A2M_IOCTL_AO_CLOCK_ENABLE,		(void*) _ao_clock_enable		},
	{ AISS2AO2A2M_IOCTL_AO_CLOCK_SOURCE,		(void*) _ao_clock_source		},
	{ AISS2AO2A2M_IOCTL_AO_CLOCK_STATUS,		(void*) _ao_clock_status		},
	{ AISS2AO2A2M_IOCTL_AO_ENABLE,				(void*) _ao_enable				},
	{ AISS2AO2A2M_IOCTL_AO_IO_ABORT,			(void*) _ao_io_abort			},
	{ AISS2AO2A2M_IOCTL_AO_IO_MODE,				(void*) _ao_io_mode				},
	{ AISS2AO2A2M_IOCTL_AO_IO_OVERFLOW,			(void*) _ao_io_overflow			},
	{ AISS2AO2A2M_IOCTL_AO_IO_TIMEOUT,			(void*) _ao_io_timeout			},
	{ AISS2AO2A2M_IOCTL_AO_OUTPUT_MODE,			(void*) _ao_output_mode			},
	{ AISS2AO2A2M_IOCTL_AO_RANGE,				(void*) _ao_range				},
	{ AISS2AO2A2M_IOCTL_AO_SW_CLOCK,			(void*) _ao_sw_clock			},
	{ AISS2AO2A2M_IOCTL_AO_SW_TRIGGER,			(void*) _ao_sw_trigger			},
	{ AISS2AO2A2M_IOCTL_AO_THRESH_LVL,			(void*) _ao_thresh_lvl			},
	{ AISS2AO2A2M_IOCTL_AO_THRESH_STS,			(void*) _ao_thresh_sts			},

	{ AISS2AO2A2M_IOCTL_BDO_ACCESS_MODE,		(void*) _bdo_access_mode		},
	{ AISS2AO2A2M_IOCTL_BDO_BUF_CLEAR,			(void*) _bdo_buf_clear			},
	{ AISS2AO2A2M_IOCTL_BDO_BUF_ENABLE,			(void*) _bdo_buf_enable			},
	{ AISS2AO2A2M_IOCTL_BDO_BUF_LEVEL,			(void*) _bdo_buf_level			},
	{ AISS2AO2A2M_IOCTL_BDO_BUF_OVERFLOW,		(void*) _bdo_buf_overflow		},
	{ AISS2AO2A2M_IOCTL_BDO_BUF_UNDERFLOW,		(void*) _bdo_buf_underflow		},
	{ AISS2AO2A2M_IOCTL_BDO_CLOCK_ENABLE,		(void*) _bdo_clock_enable		},
	{ AISS2AO2A2M_IOCTL_BDO_CLOCK_SOURCE,		(void*) _bdo_clock_source		},
	{ AISS2AO2A2M_IOCTL_BDO_IO_ABORT,			(void*) _bdo_io_abort			},
	{ AISS2AO2A2M_IOCTL_BDO_IO_MODE,			(void*) _bdo_io_mode			},
	{ AISS2AO2A2M_IOCTL_BDO_IO_OVERFLOW,		(void*) _bdo_io_overflow		},
	{ AISS2AO2A2M_IOCTL_BDO_IO_TIMEOUT,			(void*) _bdo_io_timeout			},
	{ AISS2AO2A2M_IOCTL_BDO_IO_UNDERFLOW,		(void*) _bdo_io_underflow		},
	{ AISS2AO2A2M_IOCTL_BDO_RATE_GEN_ENABLE,	(void*) _bdo_rate_gen_enable	},
	{ AISS2AO2A2M_IOCTL_BDO_RATE_GEN_NDIV,		(void*) _bdo_rate_gen_ndiv		},
	{ AISS2AO2A2M_IOCTL_BDO_SW_CLOCK,			(void*) _bdo_sw_clock			},
	{ AISS2AO2A2M_IOCTL_BDO_THRESH_LVL,			(void*) _bdo_thresh_lvl			},
	{ AISS2AO2A2M_IOCTL_BDO_THRESH_STS,			(void*) _bdo_thresh_sts			},

	{ AISS2AO2A2M_IOCTL_CBL_IN_CLK_IO,			(void*) _cbl_in_clk_io			},
	{ AISS2AO2A2M_IOCTL_CBL_IO_CLK_DIR,			(void*) _cbl_io_clk_dir			},
	{ AISS2AO2A2M_IOCTL_CBL_OUT_CLK_IO,			(void*) _cbl_out_clk_io			},
	{ AISS2AO2A2M_IOCTL_CBL_TRIG_IO,			(void*) _cbl_trig_io			},
	{ AISS2AO2A2M_IOCTL_DATA_FORMAT,			(void*) _data_format			},
	{ AISS2AO2A2M_IOCTL_DIO_DIR,				(void*) _dio_dir				},
	{ AISS2AO2A2M_IOCTL_DIO_READ,				(void*) _dio_read				},
	{ AISS2AO2A2M_IOCTL_DIO_WRITE,				(void*) _dio_write				},
	{ AISS2AO2A2M_IOCTL_IRQ_ENABLE,				(void*) _irq_enable				},
	{ AISS2AO2A2M_IOCTL_RATE_A_GEN_ENABLE,		(void*) _rate_a_gen_enable		},
	{ AISS2AO2A2M_IOCTL_RATE_A_GEN_NDIV,		(void*) _rate_a_gen_ndiv		},
	{ AISS2AO2A2M_IOCTL_RATE_B_GEN_ENABLE,		(void*) _rate_b_gen_enable		},
	{ AISS2AO2A2M_IOCTL_RATE_B_GEN_NDIV,		(void*) _rate_b_gen_ndiv		},
	{ AISS2AO2A2M_IOCTL_RATE_C_GEN_ENABLE,		(void*) _rate_c_gen_enable		},
	{ AISS2AO2A2M_IOCTL_RATE_C_GEN_NDIV,		(void*) _rate_c_gen_ndiv		},
	{ AISS2AO2A2M_IOCTL_TRIGGER_DIR,			(void*) _trigger_dir			},
	{ AISS2AO2A2M_IOCTL_WAIT_CANCEL,			(void*) gsc_wait_cancel_ioctl	},
	{ AISS2AO2A2M_IOCTL_WAIT_EVENT,				(void*) gsc_wait_event_ioctl	},
	{ AISS2AO2A2M_IOCTL_WAIT_STATUS,			(void*) gsc_wait_status_ioctl	},
	{ -1, NULL }
};


