// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AISS2AO2A2M/driver/io.c $
// $Rev: 45168 $
// $Date: 2019-07-01 13:41:01 -0500 (Mon, 01 Jul 2019) $

// 16AISS2AO2A2M: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	dev_io_close
*
*	Purpose:
*
*		Cleanup the I/O stuff for the device as it is being closed.
*
*	Arguments:
*
*		dev	The data for the device of interest.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void dev_io_close(dev_data_t* dev)
{
	int	i;

	for (i = 0; i < DEV_IO_STREAM_QTY; i++)
	{
		if (dev->io.io_streams[i])
		{
			if (dev->io.io_streams[i]->dev_io_close)
			{
				(dev->io.io_streams[i]->dev_io_close)(dev, dev->io.io_streams[i]);
			}
		}
	}
}



/******************************************************************************
*
*	Function:	dev_io_create
*
*	Purpose:
*
*		Perform I/O based initialization as the driver is being loaded.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*	Returned:
*
*		0		All went well.
*		< 0		The error status for the problem encounterred.
*
******************************************************************************/

int dev_io_create(dev_data_t* dev)
{
	int	ret		= 0;
	int	tmp;

	dev->io.io_streams[0]	= &dev->io.ai_rx;
	dev->io.io_streams[1]	= &dev->io.ao_tx;
	dev->io.io_streams[2]	= &dev->io.bdo_tx;

	tmp	= dev_read_ai_create(dev, &dev->io.ai_rx);
	ret	= ret ? ret : tmp;

	tmp	= dev_write_ao_create(dev, &dev->io.ao_tx);
	ret	= ret ? ret : tmp;

	tmp	= dev_write_bdo_create(dev, &dev->io.bdo_tx);
	ret	= ret ? ret : tmp;

	return(ret);
}



/******************************************************************************
*
*	Function:	dev_io_destroy
*
*	Purpose:
*
*		Perform I/O based cleanup as the driver is being unloaded.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void dev_io_destroy(dev_data_t* dev)
{
	int	i;

	dev_io_close(dev);	// Just in case.

	for (i = 0; i < DEV_IO_STREAM_QTY; i++)
	{
		if (dev->io.io_streams[i])
		{
			gsc_io_destroy(dev, dev->io.io_streams[i]);
		}
	}
}



/******************************************************************************
*
*	Function:	dev_io_open
*
*	Purpose:
*
*		Perform I/O based initialization as the device is being opened.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*	Returned:
*
*		0		All went well.
*		< 0		The code for the error seen.
*
******************************************************************************/

int dev_io_open(dev_data_t* dev)
{
	int	i;

	dev_io_close(dev);	// Just in case.

	for (i = 0; i < DEV_IO_STREAM_QTY; i++)
	{
		if (dev->io.io_streams[i])
		{
			if (dev->io.io_streams[i]->dev_io_open)
				(dev->io.io_streams[i]->dev_io_open)(dev, dev->io.io_streams[i]);
		}
	}

	return(0);
}



/******************************************************************************
*
*	Function:	dev_io_read_select
*
*	Purpose:
*
*		Select the dev_io_t structure for the referenced I/O read stream.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*		count	The number of bytes to transfer, with the I/O selector encoded
*				in the 0xF0000000 nibble.
*
*	Returned:
*
*		0		All went well.
*		< 0		The code for the error seen.
*
******************************************************************************/

dev_io_t* dev_io_read_select(dev_data_t* dev, size_t select)
{
	dev_io_t*	io;

	// We have only one input stream.
	io	= &dev->io.ai_rx;
	return(io);
}



/******************************************************************************
*
*	Function:	dev_io_write_select
*
*	Purpose:
*
*		Select the dev_io_t structure for the referenced I/O write stream.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*		count	The number of bytes to transfer, with the I/O selector encoded
*				in the 0xF0000000 nibble.
*
*	Returned:
*
*		0		All went well.
*		< 0		The code for the error seen.
*
******************************************************************************/

dev_io_t* dev_io_write_select(dev_data_t* dev, size_t select)
{
	dev_io_t*	io;

	if (select & ~GSC_IO_SIZE_QTY_MASK)
	{
		io	= &dev->io.bdo_tx;
	}
	else
	{
		io	= &dev->io.ao_tx;
	}

	return(io);
}



