// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AISS2AO2A2M/driver/device.c $
// $Rev: 45168 $
// $Date: 2019-07-01 13:41:01 -0500 (Mon, 01 Jul 2019) $

// 16AISS2AO2A2M: Device Driver: source file

#include "main.h"



// variables *******************************************************************

const gsc_dev_id_t	dev_id_list[]	=
{
	// model			Vendor	Device	SubVen	SubDev	type
	{ "16AISS2AO2A2M",	0x10B5, 0x9056, 0x10B5, 0x3595,	GSC_DEV_TYPE_16AISS2AO2A2M	},
	{ NULL }
};



//*****************************************************************************
static void _channels_compute(dev_data_t* dev)
{
	if (dev->cache.gsc_acr_32 & D16)
		dev->cache.ai_chans_qty	= 1;
	else
		dev->cache.ai_chans_qty	= 2;

	if (dev->cache.gsc_acr_32 & D17)
		dev->cache.ao_chans_qty	= 0;
	else
		dev->cache.ao_chans_qty	= 2;

	dev->cache.ai_chans_max		= 2;
	dev->cache.ao_chans_max		= 2;
}



//*****************************************************************************
static void _master_clock_compute(dev_data_t* dev)
{
	s32	def	= 40320000;

	switch (dev->cache.gsc_acr_32 & 0xC0000)
	{
		default:
		case 0x40000:
		case 0x80000:
		case 0xC0000:

			printf("Master Clock option not recognized.\n");
			printf("ACR = 0x%08lX\n", (unsigned long) dev->cache.gsc_acr_32);
			printf("Resorting to the default of %ld Hz.\n", (long) def);

		case 0x00000:	dev->cache.master_clock	= def;	break;
	}
}



/******************************************************************************
*
*	Function:	dev_device_create
*
*	Purpose:
*
*		Do everything needed to setup and use the given device.
*
*	Arguments:
*
*		dev		The structure to initialize.
*
*	Returned:
*
*		0		All is well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int dev_device_create(dev_data_t* dev)
{
	static const gsc_bar_maps_t	bar_map	=
	{
		{
			// mem	io	rw
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 0: PLX registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 1: PLX registers, I/O mapped
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RW },	// BAR 2: GSC registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 3: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 4: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO }	// BAR 5: unused
		}
	};

	u32	dma0;
	u32	dma1;
	int	ret;

	for (;;)	// A convenience loop.
	{
		// Verify some macro contents.
		ret	= gsc_macro_test_base_name(AISS2AO2A2M_BASE_NAME);
		if (ret)	break;

		ret	= gsc_macro_test_model();
		if (ret)	break;

		// PCI setup.
		ret	= os_pci_dev_enable(&dev->pci);
		if (ret)	break;

		ret	= os_pci_master_set(&dev->pci);
		if (ret)	break;

		// Control ISR access to the device and data structure.
		ret	= os_spinlock_create(&dev->spinlock);
		if (ret)	break;

		// Control access to the device and data structure.
		ret	= os_sem_create(&dev->sem);
		if (ret)	break;

		// Access the BAR regions.
		ret	= gsc_bar_create(dev, &dev->bar, &bar_map);
		if (ret)	break;

		// Firmware access.
		dev->vaddr.gsc_aobr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_AOBR));
		dev->vaddr.gsc_aoc0r_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_AOC0R));
		dev->vaddr.gsc_aoc1r_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_AOC1R));
		dev->vaddr.gsc_acfgr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_ACFGR));
		dev->vaddr.gsc_acr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_ACR));
		dev->vaddr.gsc_aibr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_AIBR));
		dev->vaddr.gsc_avr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_AVR));
		dev->vaddr.gsc_bcr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_BCR));
		dev->vaddr.gsc_bdobcr_32	= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_BDOBCR));
		dev->vaddr.gsc_bdobr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_BDOBR));
		dev->vaddr.gsc_bdobsr_32	= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_BDOBSR));
		dev->vaddr.gsc_bdorgr_32	= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_BDORGR));
		dev->vaddr.gsc_boor_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_BOOR));
		dev->vaddr.gsc_diopr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_DIOPR));
		dev->vaddr.gsc_ibsr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_IBSR));
		dev->vaddr.gsc_ibtr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_IBTR));
		dev->vaddr.gsc_icr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_ICR));
		dev->vaddr.gsc_obsr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_OBSR));
		dev->vaddr.gsc_obtr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_OBTR));
		dev->vaddr.gsc_psr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_PSR));
		dev->vaddr.gsc_ragr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_RAGR));
		dev->vaddr.gsc_rbgr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_RBGR));
		dev->vaddr.gsc_rcgr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AISS2AO2A2M_GSC_RCGR));

		// Data cache initialization.
		dev->cache.gsc_acr_32		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_acr_32);

		_channels_compute(dev);
		_master_clock_compute(dev);

		#define	AI_FSAMP_MAX		2000000
		#define	AO_FSAMP_MAX		2000000
		#define	BDO_FSAMP_MAX		5000000

		dev->cache.autocal_ms		= 5000;		// BCR.D28
		dev->cache.initialize_ms	= 3;
		dev->cache.rate_gen_qty		= 4;

		dev->cache.ai_fifo_size		= _2M;
		dev->cache.ai_ndiv_min		= dev->cache.master_clock / AI_FSAMP_MAX;
		dev->cache.ai_ndiv_max		= 0xFFFFFF;
		dev->cache.ai_fsamp_max		= dev->cache.master_clock / dev->cache.ai_ndiv_min;
		dev->cache.ai_fsamp_min		= dev->cache.master_clock / dev->cache.ai_ndiv_max;

		dev->cache.ao_fifo_size		= _2M;
		dev->cache.ao_ndiv_min		= dev->cache.master_clock / AO_FSAMP_MAX;
		dev->cache.ao_ndiv_max		= 0xFFFFFF;
		dev->cache.ao_fsamp_max		= dev->cache.master_clock / dev->cache.ao_ndiv_min;
		dev->cache.ao_fsamp_min		= dev->cache.master_clock / dev->cache.ao_ndiv_max;

		dev->cache.bdo_fifo_size	= _256K;
		dev->cache.bdo_ndiv_min		= dev->cache.master_clock / BDO_FSAMP_MAX;
		dev->cache.bdo_ndiv_max		= 0xFFFFFF;
		dev->cache.bdo_fsamp_max	= dev->cache.master_clock / dev->cache.bdo_ndiv_min;
		dev->cache.bdo_fsamp_min	= dev->cache.master_clock / dev->cache.bdo_ndiv_max;

		// Initialize additional resources.
		ret	= dev_irq_create(dev);
		if (ret)	break;

		ret	= dev_io_create(dev);
		if (ret)	break;

		dma0	= GSC_DMA_SEL_STATIC
				| GSC_DMA_CAP_BMDMA_READ
				| GSC_DMA_CAP_DMDMA_READ;
		dma1	= GSC_DMA_SEL_STATIC
				| GSC_DMA_CAP_BMDMA_WRITE
				| GSC_DMA_CAP_DMDMA_WRITE;
		ret		= gsc_dma_create(dev, dma0, dma1);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	dev_device_destroy
*
*	Purpose:
*
*		Do everything needed to release the referenced device.
*
*	Arguments:
*
*		dev		The partial data for the device of interest.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void dev_device_destroy(dev_data_t* dev)
{
	if (dev)
	{
		gsc_dma_destroy(dev);
		dev_io_destroy(dev);
		dev_irq_destroy(dev);
		gsc_bar_destroy(&dev->bar);
		os_sem_destroy(&dev->sem);
		os_spinlock_destroy(&dev->spinlock);
		os_pci_master_clear(&dev->pci);
		os_pci_dev_disable(&dev->pci);
	}
}


