// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AICS32/utils/reg.c $
// $Rev: 56657 $
// $Date: 2025-09-19 11:05:56 -0500 (Fri, 19 Sep 2025) $

// 16AICS32: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	_GSC_REG(a)				"GSC " #a, AICS32_GSC_##a, 0, 0
#define	SHOW_FIELD(b,e,eol,l,d)	gsc_reg_field_show(width+10,21, value,(b),(e),(eol),(l),(d))



// prototypes *****************************************************************

static	int	_bcfgr_decode(int fd, int supported, u32 value, int width);
static	int	_bctlr_decode(int fd, int supported, u32 value, int width);
static	int	_bsizr_decode(int fd, int supported, u32 value, int width);
static	int	_em00r_decode(int fd, int supported, u32 value, int width);
static	int	_em16r_decode(int fd, int supported, u32 value, int width);
static	int	_ibcr_decode(int fd, int supported, u32 value, int width);
static	int	_ibdr_decode(int fd, int supported, u32 value, int width);
static	int	_icr_decode(int fd, int supported, u32 value, int width);
static	int	_rgr_decode(int fd, int supported, u32 value, int width);
static	int	_sscr_decode(int fd, int supported, u32 value, int width);

static const gsc_reg_def_t* _find_reg(u32 reg, const gsc_reg_def_t* list);



// variables ******************************************************************

static gsc_reg_def_t	_gsc[]	=
{
	{ _GSC_REG(BCTLR),		0,	_bctlr_decode,	"Board Control Register"			},
	{ _GSC_REG(ICR),		0,	_icr_decode,	"Interrupt Control Register"		},
	{ _GSC_REG(IBDR),		0,	_ibdr_decode,	"Input Buffer Data Register"		},
	{ _GSC_REG(IBCR),		0,	_ibcr_decode,	"Input Buffer Control Register"		},
	{ _GSC_REG(RAGR),		0,	_rgr_decode,	"Rate-A Generator Register"			},
	{ _GSC_REG(RBGR),		0,	_rgr_decode,	"Rate-B Generator Register"			},
	{ _GSC_REG(BSIZR),		0,	_bsizr_decode,	"Buffer Size Register"				},
	{ _GSC_REG(SSCR),		0,	_sscr_decode,	"Scan & Sync Control Register"		},
	{ _GSC_REG(BCFGR),		0,	_bcfgr_decode,	"Board Configuration Register"		},
	{ _GSC_REG(AVR),		0,	NULL,			"Autocal Values Register"			},
	{ _GSC_REG(EM00R),		0,	_em00r_decode,	"Excitation Mask 00-15 Register"	},
	{ _GSC_REG(EM16R),		0,	_em16r_decode,	"Excitation Mask 16-31 Register"	},

	{ NULL, 0, 0, 0, 0,		NULL	}
};



//*****************************************************************************
static int _bctlr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	aim[]	=
	{
		"Differential",
		"Single-Ended",
		"Zero Test",
		"+Vref Test",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved"
	};

	static const char*	range[]	=
	{
		"+-2.5 Volts",
		"+-5 Volts",
		"+-10 Volts",
		"+-10 Volts"
	};

	static const char*	disabled[]	= { "Disabled",			"Enabled"		};
	static const char*	idle[]		= { "Idle",				"Active"		};
	static const char*	no[]		= { "No",				"Yes"			};
	static const char*	twos[]		= { "Twos Compliment",	"Offset Binary"	};

	SHOW_FIELD( 2,  0, 1, aim,		"Input Mode"		);
	SHOW_FIELD( 3,  3, 1, NULL,		"Reserved"			);
	SHOW_FIELD( 5,  4, 1, range,	"Range"				);
	SHOW_FIELD( 6,  6, 1, twos,		"Data Format"		);
	SHOW_FIELD( 7,  7, 1, disabled,	"External Sync"		);
	SHOW_FIELD( 8,  8, 1, disabled,	"Excitation Test"	);
	SHOW_FIELD(11,  9, 1, NULL,		"Reserved"			);
	SHOW_FIELD(12, 12, 1, idle,		"Input Sync"		);
	SHOW_FIELD(13, 13, 1, idle,		"Autocal"			);
	SHOW_FIELD(14, 14, 1, no,		"Autocal Pass"		);
	SHOW_FIELD(15, 15, 1, idle,		"Initialize"		);
	SHOW_FIELD(31, 16, 1, NULL,		"Reserved"			);
	return(0);
}



//*****************************************************************************
static int _icr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	irq0[]	=
	{
		"Initialization Complete",
		"Autocal Completed",
		"Input Scan Initiated (Sync)",
		"Input Scan Completed",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved"
	};

	static const char*	irq1[]	=
	{
		"Idle/None",
		"Input Buffer Threshold Low->High",
		"Input Buffer Threshold High->Low",
		"Input Buffer Overflow/Underflow",
		"Reserved",
		"Reserved",
		"Reserved",
		"Reserved"
	};

	static const char*	negated[]	= { "Negated",	"Asserted"	};

	SHOW_FIELD( 2,  0, 1, irq0,		"IRQ0 Select"	);
	SHOW_FIELD( 3,  3, 1, negated,	"IRQ0 Request"	);
	SHOW_FIELD( 6,  4, 1, irq1,		"IRQ1 Select"	);
	SHOW_FIELD( 7,  7, 1, negated,	"IRQ1 Request"	);
	SHOW_FIELD(31,  8, 1, NULL,		"Reserved"		);
	return(0);
}



//*****************************************************************************
static int _ibdr_decode(int fd, int supported, u32 value, int width)
{
	SHOW_FIELD(15,  0, 1, NULL,	"Data"			);
	SHOW_FIELD(31, 16, 1, NULL,	"Reserved"		);
	return(0);
}



//*****************************************************************************
static int _ibcr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	clear[]	= { "Clear",	"Set"		};
	static const char*	idle[]	= { "Idle",		"Active"	};

	SHOW_FIELD(15,  0, 1, NULL,		"Threshold"			);
	SHOW_FIELD(16, 16, 1, idle,		"Clear Buffer"		);
	SHOW_FIELD(17, 17, 1, clear,	"Threshold Flag"	);
	SHOW_FIELD(31, 18, 1, NULL,		"Reserved"			);
	return(0);
}



//*****************************************************************************
static int _rgr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	enabled[]	= { "No/Enabled",	"Yes/Disabled"	};

	SHOW_FIELD(15,  0, 1, NULL,		"Nrate"		);
	SHOW_FIELD(16, 16, 1, enabled,	"Disable"	);
	SHOW_FIELD(31, 17, 1, NULL,		"Reserved"	);
	return(0);
}



//*****************************************************************************
static int _bsizr_decode(int fd, int supported, u32 value, int width)
{
	SHOW_FIELD(15,  0, 1, NULL,	"Filled Locations"	);
	SHOW_FIELD(31, 16, 1, NULL,	"Reserved"			);
	return(0);
}



//*****************************************************************************
static int _sscr_decode(int fd, int supported, u32 value, int width)
{
	static const char*	channels[]	=
	{
		"Single Channel Mode",
		"2 Channels (0-1)",
		"4 Channels (0-3)",
		"8 Channels (0-7)",
		"16 Channels (0-15)",
		"32 Channels (0-31)",
		"64 Channels (0-63, Single Ended only)",
		"Channel Range (x-y)"
	};

	static const char*	clk_src[]	=
	{
		"Rate-A Generator",
		"Rate-B Generator",
		"External Sync Input",
		"Software (BCTLR Input Sync Bit)"
	};

	static const char*	b_src[]		= { "Master Clock",	"Rate-A Generator"	};

	SHOW_FIELD( 2,  0, 1, channels,	"Scan Size"				);
	SHOW_FIELD( 4,  3, 1, clk_src,	"Input Clock Source"	);
	SHOW_FIELD( 9,  5, 1, NULL,		"Reserved"				);
	SHOW_FIELD(10, 10, 1, b_src,	"Rate-B Gen. Source"	);
	SHOW_FIELD(11, 11, 1, NULL,		"Reserved"				);
	SHOW_FIELD(17, 12, 1, NULL,		"Single Channel"		);
	SHOW_FIELD(31, 18, 1, NULL,		"Reserved"				);
	return(0);
}



//*****************************************************************************
static int _bcfgr_decode(int fd, int supported, u32 value, int width)
{
//	static const char*	excite[]	=
//	{
//		"1.0 ma",
//		"2.0 ma",
//		"3.0 ma",
//		"5.0 ma",
//		"0.4 ma",
//		"10 ma",
//		"Reserved",
//		"Reserved"
//	};

	SHOW_FIELD(11,  0, 1, NULL,	"Firmware Rev"			);
	SHOW_FIELD(15, 12, 1, NULL,	"Reserved"				);
	SHOW_FIELD(18, 16, 0, NULL,	"Excitation Current"	);
	printf("FIRMWARE FIELD INACTIVE.\n");
	SHOW_FIELD(31, 19, 1, NULL,	"Reserved"				);
	return(0);
}



//*****************************************************************************
static int _emxxr_decode(u32 value, int width, int offset)
{
	char					buf[16];
	int						errs;
	int						i;
	const gsc_reg_def_t*	ptr;

	static const char*	excite[]	= { "Off",	"Excited"	};

	ptr	= _find_reg(AICS32_GSC_BCTLR, _gsc);

	if (ptr == NULL)
	{
		errs	= 1;
	}
	else
	{
		errs	= 0;

		for (i = 0; i <= 15; i++)
		{
			sprintf(buf, "Channel %02d", i + offset);
			SHOW_FIELD(i,  i, 1, excite, buf);
		}

		SHOW_FIELD(31, 16, 1, NULL, "Reserved");
	}

	return(errs);
}



//*****************************************************************************
static int _em00r_decode(int fd, int supported, u32 value, int width)
{
	int	ret;

	ret	= _emxxr_decode(value, width, 0);
	return(ret);
}



//*****************************************************************************
static int _em16r_decode(int fd, int supported, u32 value, int width)
{
	int	ret;

	ret	= _emxxr_decode(value, width, 16);
	return(ret);
}



//*****************************************************************************
static const gsc_reg_def_t* _find_reg(u32 reg, const gsc_reg_def_t* list)
{
	const gsc_reg_def_t*	def	= NULL;
	int						i;

	for (i = 0; list[i].name; i++)
	{
		if (reg == list[i].reg)
		{
			def	= &list[i];
			break;
		}
	}

	return(def);
}



/******************************************************************************
*
*	Function:	aics32_reg_get_def_id
*
*	Purpose:
*
*		Retrieve the register definition structure given the register id.
*
*	Arguments:
*
*		reg		The id of the register to access.
*
*	Returned:
*
*		NULL	The register id wasn't found.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* aics32_reg_get_def_id(u32 reg)
{
	const gsc_reg_def_t*	def;

	def	= _find_reg(reg, _gsc);
	return(def);
}



/******************************************************************************
*
*	Function:	aics32_reg_get_def_index
*
*	Purpose:
*
*		Retrieve the register definition structure based on an index.
*
*	Arguments:
*
*		index	The index of the register to access.
*
*	Returned:
*
*		NULL	The index doesn't correspond to a known register.
*		else	A pointer to the register definition.
*
******************************************************************************/

const gsc_reg_def_t* aics32_reg_get_def_index(int index)
{
	const gsc_reg_def_t*	def;

	if (index < 0)
		def	= NULL;
	else if (index >= (SIZEOF_ARRAY(_gsc) - 1))
		def	= NULL;
	else
		def	= &_gsc[index];

	return(def);
}



/******************************************************************************
*
*	Function:	aics32_reg_get_desc
*
*	Purpose:
*
*		Retrieve the description of the specified register.
*
*	Arguments:
*
*		reg		The register whose description is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* aics32_reg_get_desc(u32 reg)
{
	const gsc_reg_def_t*	def;
	const char*				desc;

	def	= _find_reg(reg, _gsc);

	if (def)
		desc	= def->desc;
	else
		desc	= "UNKNOWN";

	return(desc);
}



/******************************************************************************
*
*	Function:	aics32_reg_get_name
*
*	Purpose:
*
*		Retrieve the name of the specified register.
*
*	Arguments:
*
*		reg		The register whose name is desired.
*
*	Returned:
*
*		!NULL	The register's name.
*
******************************************************************************/

const char* aics32_reg_get_name(u32 reg)
{
	const gsc_reg_def_t*	def;
	const char*				name;

	def	= _find_reg(reg, _gsc);

	if (def)
		name	= def->name;
	else
		name	= "UNKNOWN";

	return(name);
}



/******************************************************************************
*
*	Function:	aics32_reg_list
*
*	Purpose:
*
*		List the GSC registers and their values.
*
*	Arguments:
*
*		fd		The handle to access the device.
*
*		detail	List the register details?
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int aics32_reg_list(int fd, int detail)
{
	int	errs;

	errs	= gsc_reg_list(fd, _gsc, detail, aics32_reg_read);
	return(errs);
}



