// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AICS32/utils/query.c $
// $Rev: 56657 $
// $Date: 2025-09-19 11:05:56 -0500 (Fri, 19 Sep 2025) $

// 16AICS32: Utilities: source file

#include "main.h"



/******************************************************************************
*
*	Function:	aics32_query
*
*	Purpose:
*
*		Provide a visual wrapper for the AICS32_IOCTL_QUERY service.
*
*	Arguments:
*
*		fd		Use this handle to access the device.
*
*		index	The index of the device to access. Ignore if < 0.
*
*		verbose	Work verbosely? 0 = no, !0 = yes
*
*		set		This is the query option to access.
*
*		get		The results are reported here, if not NULL.
*
*	Returned:
*
*		>= 0	The number of errors encountered here.
*
******************************************************************************/

int aics32_query(int fd, int index, int verbose, s32 set, s32* get)
{
	char		buf[128];
	int			errs	= 0;
	const char*	ptr;
	s32			query	= set;
	int			ret;
	s32			tmp;

	switch (query)
	{
		default:

			errs++;
			ptr	= buf;
			sprintf(buf, "Query Error (#%ld)", (long) query);
			break;

		case AICS32_QUERY_AUTOCAL_MS:	ptr	= "Autocal Duration";	break;
		case AICS32_QUERY_CHANNEL_MAX:	ptr	= "Max Channels";		break;
		case AICS32_QUERY_CHANNEL_QTY:	ptr	= "Installed Channels";	break;
		case AICS32_QUERY_DEVICE_TYPE:	ptr	= "Device Type";		break;
		case AICS32_QUERY_EXCITATION:	ptr	= "Excitation Current";	break;
		case AICS32_QUERY_FGEN_MAX:		ptr	= "Fgen Max";			break;
		case AICS32_QUERY_FGEN_MIN:		ptr	= "Fgen Min";			break;
		case AICS32_QUERY_FIFO_SIZE:	ptr	= "FIFO Size";			break;
		case AICS32_QUERY_FSAMP_MAX:	ptr	= "Fsamp Max";			break;
		case AICS32_QUERY_FSAMP_MIN:	ptr	= "Fsamp Min";			break;
		case AICS32_QUERY_INIT_MS:		ptr	= "Init. Duration";		break;
		case AICS32_QUERY_NRATE_MAX:	ptr	= "Nrate Max";			break;
		case AICS32_QUERY_NRATE_MIN:	ptr	= "Nrate Min";			break;
		case AICS32_QUERY_MASTER_CLOCK:	ptr	= "Master Clock";		break;
		case AICS32_QUERY_RATE_GEN_QTY:	ptr	= "Rate Gen Qty";		break;
	}

	if (verbose)
		gsc_label_index(ptr, index);

	ret		= aics32_ioctl(fd, AICS32_IOCTL_QUERY, &set);
	errs	+= ret ? 1 : 0;

	switch (query)
	{
		default:

			errs++;
			gsc_label_long_comma_buf(set, buf);
			break;

		case AICS32_QUERY_CHANNEL_MAX:
		case AICS32_QUERY_CHANNEL_QTY:
		case AICS32_QUERY_NRATE_MAX:
		case AICS32_QUERY_NRATE_MIN:
		case AICS32_QUERY_RATE_GEN_QTY:

			gsc_label_long_comma_buf(set, buf);
			break;

		case AICS32_QUERY_FGEN_MAX:
		case AICS32_QUERY_FGEN_MIN:
		case AICS32_QUERY_MASTER_CLOCK:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " Hz");
			break;

		case AICS32_QUERY_AUTOCAL_MS:
		case AICS32_QUERY_INIT_MS:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " ms");
			break;

		case AICS32_QUERY_FSAMP_MAX:
		case AICS32_QUERY_FSAMP_MIN:

			gsc_label_long_comma_buf(set, buf);
			strcat(buf, " S/S");
			break;

		case AICS32_QUERY_FIFO_SIZE:

			ptr	= "";
			tmp	= set;

			if ((tmp) && ((tmp % 1024) == 0))
			{
				ptr	= "K";
				tmp	/= 1024;
			}

			gsc_label_long_comma_buf(tmp, buf);
			strcat(buf, ptr);
			strcat(buf, " Words");
			break;

		case AICS32_QUERY_DEVICE_TYPE:

			if (set == GSC_DEV_TYPE_16AICS32)
			{
				strcpy(buf, "16AICS32");
			}
			else
			{
				errs++;
				sprintf(buf, "INVALID: %ld", (long) set);
			}

			break;

		case AICS32_QUERY_EXCITATION:

			switch (set)
			{
				default:

					errs++;
					sprintf(buf, "Invalid option: %ld", (long) set);
					break;

				case AICS32_QUERY_EXCITATION_1_0_MA:	//strcpy(buf, "1.0 ma");	break;
				case AICS32_QUERY_EXCITATION_2_0_MA:	//strcpy(buf, "2.0 ma");	break;
				case AICS32_QUERY_EXCITATION_3_0_MA:	//strcpy(buf, "3.0 ma");	break;
				case AICS32_QUERY_EXCITATION_5_0_MA:	//strcpy(buf, "5.0 ma");	break;
				case AICS32_QUERY_EXCITATION_0_4_MA:	//strcpy(buf, "0.4 ma");	break;
				case AICS32_QUERY_EXCITATION_10_0_MA:	//strcpy(buf, "10.0 ma");	break;

					strcpy(buf, "THIS FIRMWARE FIELD IS INACTIVE.");
					break;
			}

			break;
	}

	if (verbose)
	{
		if (errs)
			printf("FAIL <---  (%s)\n", buf);
		else
			printf("%s\n", buf);
	}

	if (get)
		get[0]	= set;

	return(errs);
}


