// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AICS32/utils/fsamp_ai.c $
// $Rev: 56657 $
// $Date: 2025-09-19 11:05:56 -0500 (Fri, 19 Sep 2025) $

// 16AICS32: Utilities: source file

#include "main.h"



// macros *********************************************************************

#define	ABS(v)					(((v) < 0) ? (-(v)) : (v))
#define	LIMIT_MAX(v,max)		(((v) > (max)) ? (max) : (v))
#define	LIMIT_MIN(v,min)		(((v) < (min)) ? (min) : (v))
#define	LIMIT_RANGE(v,min,max)	(LIMIT_MIN((LIMIT_MAX((v),(max))),(min)))



// data types *****************************************************************

typedef struct
{
	s32		fsamp_want;
	double	fsamp_got;

	double	delta;
	s32		enable_a;
	s32		enable_b;
	s32		master;
	s32		fsamp_max;
	s32		fsamp_min;
	s32		nrate_a;
	s32		nrate_b;
	s32		nrate_max;
	s32		nrate_min;
	s32		src;
	s32		src_b;
} fsamp_t;



//*****************************************************************************
static int _channel_count(int fd, int* count)
{
	int	errs	= 0;
	s32	limit;
	s32	mode	= -1;
	s32	qty;
	s32	size	= -1;

	// Compute the number of channels configured for use.

	errs		+= aics32_query(fd, -1, 0, AICS32_QUERY_CHANNEL_QTY, &qty	);
	errs		+= aics32_ioctl_dsl(fd, AICS32_IOCTL_AI_MODE,		&mode	);
	errs		+= aics32_ioctl_dsl(fd, AICS32_IOCTL_SCAN_SIZE,		&size	);
	size		= 0x1 << size;
	limit		= (mode == AICS32_AI_MODE_SE) ? (qty * 2) : qty;
	count[0]	= (size <= limit) ? size : limit;
	return(errs);
}



//*****************************************************************************
static int _fsamp_max(int fd, s32* fsamp)
{
	int	count	= 0;
	int	errs	= 0;

	errs		+= _channel_count(fd, &count);
	errs		+= aics32_query(fd, -1, 0, AICS32_QUERY_FSAMP_MAX, fsamp);
	fsamp[0]	/= count;
	return(errs);
}



//*****************************************************************************
static void _fsamp_compute_a(fsamp_t* data)
{
	double	delta;
	double	fsamp_got;
	double	fsamp_want	= data->fsamp_want;
	int		init		= 0;	// Have we performed the first run initialization?
	s32		master		= data->master;
	s32		nr;
	s32		nr_max;
	s32		nr_min;
	s32		nrate_a;
	s32		nrate_max	= data->nrate_max;
	s32		nrate_min	= data->nrate_min;

	data->src		= AICS32_AI_CLK_SRC_RAG;
	data->src_b		= AICS32_RBG_CLK_SRC_MASTER;
	data->enable_a	= AICS32_GEN_ENABLE_YES;
	data->enable_b	= AICS32_GEN_ENABLE_NO;

	nr		= master / fsamp_want;
	nr		= (nr < nrate_min) ? nrate_min
			: (nr > nrate_max) ? nrate_max : nr;

	nr_min	= nr - 5;
	nr_max	= nr + 5;
	nr_min	= (nr_min < nrate_min) ? nrate_min
			: (nr_min > nrate_max) ? nrate_max : nr_min;
	nr_max	= (nr_max < nrate_min) ? nrate_min
			: (nr_max > nrate_max) ? nrate_max : nr_max;

	for (nrate_a = nr_min; nrate_a <= nr_max; nrate_a++)
	{
		fsamp_got	= ((double) master) / nrate_a;
		delta		= fsamp_got - fsamp_want;

		if ((init == 0) || ((ABS(delta)) < (ABS(data->delta))))
		{
			init			= 1;
			data->delta		= delta;
			data->fsamp_got	= fsamp_got;
			data->nrate_a	= nrate_a;
		}
	}
}



//*****************************************************************************
static void _fsamp_compute_ab(fsamp_t* data)
{
	double	delta;
	double	fsamp_got;
	double	fsamp_want	= data->fsamp_want;
	int		init		= 0;	// Have we performed the first run initialization?
	s32		master		= data->master;
	s32		nr;
	s32		nr_max;
	s32		nr_min;
	s32		nrate_a;
	s32		nrate_b;
	s32		nrate_max	= data->nrate_max;
	s32		nrate_min	= data->nrate_min;

	data->src		= AICS32_AI_CLK_SRC_RBG;
	data->src_b		= AICS32_RBG_CLK_SRC_RAG;
	data->enable_a	= AICS32_GEN_ENABLE_YES;
	data->enable_b	= AICS32_GEN_ENABLE_YES;

	for (nrate_a = nrate_min; nrate_a <= nrate_max; nrate_a++)
	{
		nr		= master / fsamp_want / nrate_a;
		nr		= (nr < nrate_min) ? nrate_min
				: (nr > nrate_max) ? nrate_max : nr;

		nr_min	= nr - 5;
		nr_max	= nr + 5;
		nr_min	= (nr_min < nrate_min) ? nrate_min
				: (nr_min > nrate_max) ? nrate_max : nr_min;
		nr_max	= (nr_max < nrate_min) ? nrate_min
				: (nr_max > nrate_max) ? nrate_max : nr_max;

		for (nrate_b = nr_min; nrate_b <= nr_max; nrate_b++)
		{
			fsamp_got	= (double) master / ((double) nrate_b * nrate_a);
			delta		= fsamp_got - fsamp_want;

			if ((init == 0) || ((ABS(delta)) < (ABS(data->delta))))
			{
				init			= 1;
				data->delta		= delta;
				data->fsamp_got	= fsamp_got;
				data->nrate_a	= nrate_a;
				data->nrate_b	= nrate_b;
			}
		}
	}
}



//*****************************************************************************
static int _report_src_rag(int fd, s32* fsamp, char* buf)
{
	int	errs	= 0;
	s32	master;
	s32	nrate	= -1;

	errs		+= aics32_query(fd, -1, 0, AICS32_QUERY_MASTER_CLOCK, &master);
	errs		+= aics32_ioctl_dsl(fd, AICS32_IOCTL_RAG_NRATE, &nrate);
	fsamp[0]	= master / nrate;
	sprintf(buf,
			"Master %ld Hz, Nrate-A %ld, Fsamp %ld S/S",
			(long) master,
			(long) nrate,
			(long) fsamp[0]);
	return(errs);
}



//*****************************************************************************
static int _report_src_rbg_master(int fd, s32* fsamp, char* buf)
{
	int	errs	= 0;
	s32	master;
	s32	nrate	= -1;

	errs		+= aics32_query(fd, -1, 0, AICS32_QUERY_MASTER_CLOCK, &master);
	errs		+= aics32_ioctl_dsl(fd, AICS32_IOCTL_RBG_NRATE, &nrate);
	fsamp[0]	= master / nrate;
	sprintf(buf,
			"Master %ld Hz, Nrate-B %ld, Fsamp %ld S/S",
			(long) master,
			(long) nrate,
			(long) fsamp[0]);
	return(errs);
}



//*****************************************************************************
static int _report_src_rbg_rag(int fd, s32* fsamp, char* buf)
{
	int	errs	= 0;
	s32	master;
	s32	nrate_a	= -1;
	s32	nrate_b	= -1;

	errs		+= aics32_query(fd, -1, 0, AICS32_QUERY_MASTER_CLOCK, &master);
	errs		+= aics32_ioctl_dsl(fd, AICS32_IOCTL_RAG_NRATE, &nrate_a);
	errs		+= aics32_ioctl_dsl(fd, AICS32_IOCTL_RBG_NRATE, &nrate_b);
	fsamp[0]	= master / nrate_a / nrate_b;
	sprintf(buf,
			"Master %ld Hz, Nrate-A %ld, Nrate-B %ld, Fsamp %ld S/S",
			(long) master,
			(long) nrate_a,
			(long) nrate_b,
			(long) fsamp[0]);
	return(errs);
}



//*****************************************************************************
static int _report_src_rbg(int fd, s32* fsamp, char* buf)
{
	int	errs;
	s32	src		= -1;

	errs	= aics32_ioctl_dsl(fd, AICS32_IOCTL_RBG_CLK_SRC, &src);

	if (errs == 0)
	{
		switch (src)
		{
			default:

				sprintf(buf, "%d. INTERNAL ERROR: %ld", __LINE__, (long) src);
				break;

			case AICS32_RBG_CLK_SRC_MASTER:

				errs	+= _report_src_rbg_master(fd, fsamp, buf);
				break;

			case AICS32_RBG_CLK_SRC_RAG:

				errs	+= _report_src_rbg_rag(fd, fsamp, buf);
				break;
		}
	}

	return(errs);
}



//*****************************************************************************
static int _report_src_ext(s32* fsamp, char* buf)
{
	fsamp[0]	= 0;
	sprintf(buf, "Clock Source = External Clock");
	return(0);
}



//*****************************************************************************
static int _report_src_bcr(s32* fsamp, char* buf)
{
	fsamp[0]	= 0;
	sprintf(buf, "Clock Source = Software Sync");
	return(0);
}



//*****************************************************************************
int aics32_fsamp_ai_compute(int fd, int index, int verbose, s32 fsamp, s32* src, s32* src_b, s32* nrate_a, s32* nrate_b, s32* enable_a, s32* enable_b, double* rate)
{
	fsamp_t	data;
	fsamp_t	data_a;
	fsamp_t	data_ab;
	int		errs	= 0;

	if (verbose)
	{
		gsc_label_index("AI Sample Rate", index);
		gsc_label_long_comma(fsamp);
		printf(" S/S");
	}

	memset(&data, 0, sizeof(data));
	errs	+= aics32_query(fd, -1, 0, AICS32_QUERY_FSAMP_MAX, &data.fsamp_max);
	errs	+= aics32_query(fd, -1, 0, AICS32_QUERY_FSAMP_MIN, &data.fsamp_min);
	errs	+= aics32_query(fd, -1, 0, AICS32_QUERY_MASTER_CLOCK, &data.master);
	errs	+= aics32_query(fd, -1, 0, AICS32_QUERY_NRATE_MAX, &data.nrate_max);
	errs	+= aics32_query(fd, -1, 0, AICS32_QUERY_NRATE_MIN, &data.nrate_min);

	data.delta		= FLT_MAX;
	data.fsamp_want	= LIMIT_RANGE(fsamp, data.fsamp_min, data.fsamp_max);
	data.nrate_a	= data.nrate_max;
	data.nrate_b	= data.nrate_max;

	data_a	= data;
	data_ab	= data;

	_fsamp_compute_a(&data_a);
	_fsamp_compute_ab(&data_ab);

	if ((ABS(data_a.delta)) < (ABS(data_ab.delta)))
	{
		data	= data_a;

		if (verbose)
		{
			printf("%s  (", errs ? " FAIL <---" : "");
			printf("Fref ");
			gsc_label_long_comma((long) data.master);
			printf(", Nrate-A ");
			gsc_label_long_comma((long) data.nrate_a);
			printf(", Fsamp %.3f", data.fsamp_got);
			printf(")\n");
		}
	}
	else
	{
		data	= data_ab;

		if (verbose)
		{
			printf("%s  (", errs ? " FAIL <---" : "");
			printf("Fref ");
			gsc_label_long_comma((long) data.master);
			printf(", Nrate-A ");
			gsc_label_long_comma((long) data.nrate_a);
			printf(", Nrate-B ");
			gsc_label_long_comma((long) data.nrate_b);
			printf(", Fsamp %.3f", data.fsamp_got);
			printf(")\n");
		}
	}

	src[0]		= data.src;
	src_b[0]	= data.src_b;
	nrate_a[0]	= data.nrate_a;
	nrate_b[0]	= data.nrate_b;
	enable_a[0]	= data.enable_a;
	enable_b[0]	= data.enable_b;

	if (rate)
		rate[0]	= data.fsamp_got;

	return(errs);
}



//*****************************************************************************
int aics32_fsamp_ai_report(int fd, int index, int chan, int verbose, s32* sps)
{
	char	buf[1024];
	s32		burst		= -1;
	int		errs		= 0;
	s32		ext			= -1;
	s32		fsamp		= 0;
	s32		max;
	s32		mode		= -1;
	s32		qty;
	s32		single		= -1;
	s32		size		= -1;
	s32		src			= -1;

	if (verbose == 0)
	{
	}
	else if (chan >= 0)
	{
		sprintf(buf, "Channel %d", chan);
		gsc_label_index(buf, index);
	}
	else
	{
		gsc_label_index("Sample Rate", index);
	}

	for (;;)
	{
		errs	+= aics32_query(fd, -1, 0, AICS32_QUERY_FSAMP_MAX, &max);
		errs	+= aics32_ioctl_dsl(fd, AICS32_IOCTL_AI_MODE, &mode);
		errs	+= aics32_query(fd, -1, 0, AICS32_QUERY_CHANNEL_QTY, &qty);
		qty		*= (mode == AICS32_AI_MODE_SE) ? 2 : 1;
		errs	+= aics32_ioctl_dsl(fd, AICS32_IOCTL_SCAN_SIZE, &size);
		size	= 0x1 << size;
		errs	+= aics32_ioctl_dsl(fd, AICS32_IOCTL_SCAN_SINGLE, &single);
		errs	+= aics32_ioctl_dsl(fd, AICS32_IOCTL_AI_CLK_SRC, &src);
		errs	+= aics32_ioctl_dsl(fd, AICS32_IOCTL_EXT_SYNC, &ext);

		if (errs)
			break;

		if (chan >= qty)
		{
			if (verbose)
				printf("FAIL <---  (invalid channel index: %d)\n", chan);

			errs++;
			break;
		}

		if ((size == 1) && (chan != single))
		{
			sps[0]	= 0;
			sprintf(buf, "channel not scanned");
			break;
		}

		if ((size != 1) && (chan >= size))
		{
			sps[0]	= 0;
			sprintf(buf, "channel not scanned");
			break;
		}

		switch (src)
		{
			default:

				sprintf(buf, "%d. INTERNAL ERROR: %ld", __LINE__, (long) burst);
				break;

			case AICS32_AI_CLK_SRC_RAG:

				errs	+= _report_src_rag(fd, &fsamp, buf);
				break;

			case AICS32_AI_CLK_SRC_RBG:

				errs	+= _report_src_rbg(fd, &fsamp, buf);
				break;

			case AICS32_AI_CLK_SRC_EXT:

				errs	+= _report_src_ext(&fsamp, buf);
				break;

			case AICS32_AI_CLK_SRC_BCR:

				errs	+= _report_src_bcr(&fsamp, buf);
				break;
		}

		break;
	}

	if ((errs == 0) && (verbose))
	{
		gsc_label_long_comma((long long) fsamp);
		printf(" S/S  (");

		if (fsamp > max)
			printf("EXCEEDS MAX RATE, ");

		printf("%s)\n", buf);
	}

	if (sps)
		sps[0]	= fsamp;

	return(errs);
}



//*****************************************************************************
int aics32_fsamp_ai_report_all(int fd, int index, int verbose, s32* fsamp)
{
	int			chan;
	int			errs	= 0;
	s32			limit;
	s32			mode	= -1;
	const char*	psz;
	s32			size	= -1;
	s32			sps;
	s32			total	= 0;

	if (verbose)
		gsc_label_index("Sample Rates", index);

	errs	+= aics32_ioctl_dsl(fd, AICS32_IOCTL_AI_MODE, &mode);
	errs	+= aics32_ioctl_dsl(fd, AICS32_IOCTL_SCAN_SIZE, &size);
	size	= 0x1 << size;
	errs	+= _fsamp_max(fd, &limit);

	if (errs == 0)
	{
		psz	= (mode == AICS32_AI_MODE_SE) ? "Single Ended" : "Differential";

		if (verbose)
		{
			printf("(%s)\n", psz);
			gsc_label_level_inc();
		}

		for (chan = 0; chan < size; chan++)
		{
			errs	+= aics32_fsamp_ai_report(fd, -1, chan, 0, &sps);
			total	+= sps;
		}

		if (verbose)
		{
			gsc_label("Overall Rate");
			gsc_label_long_comma((long long) total);
			printf(" S/S\n");
			gsc_label_level_dec();
		}
	}

	if (fsamp)
		fsamp[0]	= total;

	return(errs);
}


