// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AICS32/samples/irq/service.c $
// $Rev: 56661 $
// $Date: 2025-09-19 11:10:25 -0500 (Fri, 19 Sep 2025) $

// 16AICS32: Sample Application: source file

#include "main.h"



// macros *********************************************************************

#define	_15_SEC		(15L * 1000L)



// data types *****************************************************************

struct _wait_src_t
{
	int			(*function)(const args_t* args);
	gsc_wait_t	wait;
};



// variables ******************************************************************

static	sem_t		sem_enter;
static	sem_t		sem_exit;
static	gsc_wait_t	wait_dst;



//*****************************************************************************
static int _wait_thread(void* arg)
{
	const args_t*	args	= arg;
	int				status;

	sem_post(&sem_enter);
	status	= aics32_ioctl(args->fd, AICS32_IOCTL_WAIT_EVENT, &wait_dst);

	if (status)
		wait_dst.flags	= 0xFFFFFFFF;

	sem_post(&sem_exit);
	return(0);
}



//*****************************************************************************
static int _process_thread(const args_t* args, const struct _wait_src_t* src)
{
	char		buf[64];
	int			errs	= 0;
	int			i;
	int			ret;
	int			status;
	os_thread_t	thread;
	gsc_wait_t	wait;

	errs	+= aics32_initialize(args->fd, -1, 0);

	for (;;)	// A convenience loop.
	{
		sem_init(&sem_enter, 0, 0);
		sem_init(&sem_exit, 0, 0);
		memset(&thread, 0, sizeof(thread));
		sprintf(buf, "wait event");
		wait_dst	= src->wait;
		errs		+= os_thread_create(&thread, buf, _wait_thread, (void*) args);

		// Wait for the thread to become waiting.
		sem_wait(&sem_enter);

		if (errs == 0)
		{
			wait	= src->wait;

			for (i = 0; i < 100; i++)
			{
				status	= aics32_ioctl(args->fd, AICS32_IOCTL_WAIT_STATUS, &wait);

				if (status)
				{
					printf(	"FAIL <---  (%d. status request: error %d)\n",
							__LINE__,
							status);
					errs++;
					break;
				}

				if (wait.count == 0)
				{
					os_sleep_ms(100);
					continue;
				}

				if (wait.count == 1)
					break;

				errs++;
				printf(	"FAIL <---  (%d. invalid wait count %ld)\n",
						__LINE__,
						(long) wait.count);
				break;
			}
		}

		// Initiate the respective event.
		errs	+= src->function(args);

		if (errs)
			break;		// There was an error.

		// Wait for the resumed thread to run.
		sem_wait(&sem_exit);

		for (i = 0;; i++)
		{
			if (wait_dst.flags)
				break;

			if (i >= 100)
			{
				printf(	"FAIL <---  (%d. thread failed to resume)\n",
						__LINE__);
				errs++;
				break;
			}

			os_sleep_ms(100);
		}

		// Verify that the wait ended as expected.

		if (src->wait.flags == 0xFFFFFFFF)
			break;		// There was an error.

		if (errs)
			break;		// There was an error.

		if (src->wait.timeout_ms == _15_SEC)
		{
			if (wait_dst.flags != GSC_WAIT_FLAG_DONE)
			{
				printf(	"FAIL <---  (%d. flag: expect 0x%lX, got 0x%lX)\n",
						__LINE__,
						(long) GSC_WAIT_FLAG_DONE,
						(long) wait_dst.flags);
				errs++;
			}

			break;
		}

		printf("FAIL <---  (%d. INTERNAL ERROR)\n", __LINE__);
		errs++;
		break;
	}

	// Force termination, just in case of an error.
	wait.flags			= 0;
	wait.main			= 0xFFFFFFFF;
	wait.gsc			= 0xFFFFFFFF;
	wait.alt			= 0xFFFFFFFF;
	wait.io				= 0xFFFFFFFF;
	wait.timeout_ms		= 0;
	wait.count			= 0;
	ret		= aics32_ioctl(args->fd, AICS32_IOCTL_WAIT_CANCEL, &wait);
	errs	+= ret ? 1 : 0;
	errs	+= os_thread_destroy(&thread);

	errs	+= aics32_initialize(args->fd, -1, 0);

	if (errs == 0)
		printf("PASS\n");

	return(errs);
}



//*****************************************************************************
int irq_wait_test(const args_t* args, u32 gsc, int (*callback)(const args_t* args))
{
	int					errs	= 0;
	struct _wait_src_t	src;

	memset(&wait_dst, 0, sizeof(wait_dst));
	memset(&src, 0, sizeof(src));
	src.function		= callback;
	src.wait.gsc		= gsc;
	src.wait.timeout_ms	= _15_SEC;
	errs	+= _process_thread(args, &src);
	return(errs);
}


