// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AICS32/driver/read.c $
// $Rev: 52042 $
// $Date: 2022-11-22 10:18:31 -0600 (Tue, 22 Nov 2022) $

// 16AICS32: Device Driver: source file

#include "main.h"



//*****************************************************************************
static void _dev_io_sw_init(dev_data_t* dev, dev_io_t* io)
{
	io->io_mode			= AICS32_IO_MODE_DEFAULT;
	io->pio_threshold	= 32;
	io->timeout_s		= AICS32_IO_TIMEOUT_DEFAULT;
}



//*****************************************************************************
static void _dev_io_close(dev_data_t* dev, dev_io_t* io)
{
	io->dma_channel	= NULL;
}



//*****************************************************************************
static void _dev_io_open(dev_data_t* dev, dev_io_t* io)
{
	_dev_io_sw_init(dev, io);
}



//*****************************************************************************
static int _dev_io_startup(dev_data_t* dev, dev_io_t* io)
{
	return(0);
}



//*****************************************************************************
static long _dev_pio_available(dev_data_t* dev, dev_io_t* io, size_t count)
{
	u32	bsr;
	u32	samples;

	bsr		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsizr_32);
	samples	= bsr & 0xFFFF;
	count	= samples * 4;
	return(count);
}



//*****************************************************************************
static long _dev_pio_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			count,
	os_time_tick_t	st_end)
{
	long	qty;

	qty	= gsc_read_pio_work_32_bit(dev, io, mem, count, st_end);
	return(qty);
}



//*****************************************************************************
static long _dev_bmdma_available(dev_data_t* dev, dev_io_t* io, size_t count)
{
	u32	avail;
	u32	bsr;
	u32	ibcr;
	u32	samples;
	u32	thresh;

	bsr		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsizr_32);
	samples	= bsr & 0xFFFF;
	avail	= samples * 4;

	if (avail >= count)
	{
		// The desired amount is available.
	}
	else
	{
		ibcr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_ibcr_32);
		samples	= ibcr & 0xFFFF;
		thresh	= samples * 4;

		if (avail >= thresh)
			count	= avail;
		else
			count	= 0;
	}

	return(count);
}



//*****************************************************************************
static long _dev_bmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			count,
	os_time_tick_t	st_end)
{
	long			qty;
	long			samples	= count / 4;
	gsc_dma_setup_t	setup;

	if (samples < io->pio_threshold)
	{
		qty	= gsc_read_pio_work_32_bit(dev, io, mem, count, st_end);
	}
	else
	{
		memset(&setup, 0, sizeof(gsc_dma_setup_t));
		setup.alt		= dev;
		setup.dev		= dev;
		setup.io		= io;
		setup.mem		= mem;
		setup.st_end	= st_end;
		setup.bytes		= count;
		setup.ability	= GSC_DMA_CAP_BMDMA_READ;

		setup.mode		= GSC_DMA_MODE_BLOCK_DMA
						| GSC_DMA_MODE_SIZE_32_BITS
						| GSC_DMA_MODE_INPUT_ENABLE
						| GSC_DMA_MODE_BURSTING_LOCAL
						| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
						| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
						| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		setup.dpr		= GSC_DMA_DPR_BOARD_TO_HOST
						| GSC_DMA_DPR_END_OF_CHAIN
						| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty	= gsc_dma_perform(&setup);
	}

	return(qty);
}



//*****************************************************************************
static long _dev_dmdma_available(dev_data_t* dev, dev_io_t* io, size_t count)
{
	// DMDMA is unsupported.
	return(-EINVAL);
}



//*****************************************************************************
static long _dev_dmdma_xfer(
	dev_data_t*		dev,
	dev_io_t*		io,
	const os_mem_t*	mem,
	size_t			count,
	os_time_tick_t	st_end)
{
	// DMDMA is unsupported.
	return(-EINVAL);
}



/******************************************************************************
*
*	Function:	dev_read_create
*
*	Purpose:
*
*		Perform a one-tine setup of the Read Analog Input streaming I/O structure.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*		io		The I/O structure for this stream.
*
*	Returned:
*
*		None.
*
******************************************************************************/

int dev_read_create(dev_data_t* dev, dev_io_t* io)
{
	int	ret;

	io->bytes_per_sample	= 4;
	io->io_reg_offset		= GSC_REG_OFFSET(AICS32_GSC_IBDR);
	io->io_reg_vaddr		= dev->vaddr.gsc_ibdr_32;

	io->dev_io_sw_init		= _dev_io_sw_init;
	io->dev_io_close		= _dev_io_close;
	io->dev_io_open			= _dev_io_open;
	io->dev_io_startup		= _dev_io_startup;
	io->dev_pio_available	= _dev_pio_available;
	io->dev_bmdma_available	= _dev_bmdma_available;
	io->dev_dmdma_available	= _dev_dmdma_available;
	io->dev_pio_xfer		= _dev_pio_xfer;
	io->dev_bmdma_xfer		= _dev_bmdma_xfer;
	io->dev_dmdma_xfer		= _dev_dmdma_xfer;

	io->wait.abort			= AICS32_WAIT_IO_RX_ABORT;
	io->wait.done			= AICS32_WAIT_IO_RX_DONE;
	io->wait.error			= AICS32_WAIT_IO_RX_ERROR;
	io->wait.timeout		= AICS32_WAIT_IO_RX_TIMEOUT;

	ret	= gsc_io_create(dev, io, 64L * 1024 * 4);

	return(ret);
}



