// $URL: http://subversion:8080/svn/gsc/trunk/drivers/gsc_common/driver/linux/os_proc.c $
// $Rev: 56590 $
// $Date: 2025-08-20 13:54:56 -0500 (Wed, 20 Aug 2025) $

// Linux: Device Driver: source file: This software is covered by the GNU GENERAL PUBLIC LICENSE (GPL).

#include "main.h"



// macros *********************************************************************

#define	_PROC_MSG_SIZE		1024

#if PAGE_SIZE < _PROC_MSG_SIZE
#error	"/proc file size may be too big."
#endif



//*****************************************************************************
static void _proc_add_version(char* page)
{
	// This is the first part of the proc file.
	// We know in advance that the buffer will accommodate this string.
	strcpy(page, "version: ");
	strcat(page, GSC_DRIVER_VERSION);
	strcat(page, "\n");
}



//*****************************************************************************
static void _proc_add_32_bit_support(char* page)
{
	// This is the second part of the proc file.
	// We know in advance that the buffer will accommodate this addition.

	const char*	support;

	switch (gsc_global.ioctl_32bit)
	{
		case GSC_IOCTL_32BIT_DISABLED:	support	= "disabled";		break;
		case GSC_IOCTL_32BIT_NATIVE:	support	= "yes (native)";	break;
		case GSC_IOCTL_32BIT_COMPAT:
		case GSC_IOCTL_32BIT_TRANSLATE:	support	= "yes";			break;
		case GSC_IOCTL_32BIT_NONE:		support	= "no";				break;
		default:
		case GSC_IOCTL_32BIT_ERROR:		support	= "INTERNAL ERROR";

			printf(	"%s: %d. %s: internal error: 32-bit support\n",
					DEV_NAME,
					__LINE__,
					__FUNCTION__);
			break;

	}

	strcat(page, "32-bit support: ");
	strcat(page, support);
	strcat(page, "\n");
}



//*****************************************************************************
static void _proc_add_board_count(char* page)
{
	char	buf[32];
	// This is the third part of the proc file.
	// We know in advance that the buffer will accommodate this addition.

	strcat(page, "boards: ");
	sprintf(buf, "%d", gsc_global.dev_qty);
	strcat(page, buf);
	strcat(page, "\n");
}



//*****************************************************************************
static void _proc_add_models(char* page)
{
	// This is the forth part of the proc file.
	// We know in advance that this might overflow the buffer.

	const char*	comma;
	const char*	eol	= "\n";
	int			i;
	int			len;
	const char*	str;

	strcat(page, "models: ");
	comma	= "";
	len		= strlen(page);

	for (i = 0; i < (int) SIZEOF_ARRAY(gsc_global.dev_list); i++)
	{
		if (gsc_global.dev_list[i])
		{
			// Append the comma.
			len	+= strlen(comma);

			if (len >= _PROC_MSG_SIZE)
				break;

			strcat(page, comma);

			// Append the model string.
			str	= gsc_global.dev_list[i]->model;
			len	+= strlen(str);

			if (len >= _PROC_MSG_SIZE)
				break;

			strcat(page, str);
			comma	= ",";
		}
	}

	// Append the line ending.
	len	+= strlen(eol);

	if (len < _PROC_MSG_SIZE)
		strcat(page, "\n");
}



//*****************************************************************************
static void _proc_add_id_strs(char* page)
{
#if defined(DEV_SUPPORTS_PROC_ID_STR)
	// This is the fifth part of the proc file.
	// We know in advance that this might overflow the buffer.

	const char*	comma;
	const char*	eol	= "\n";
	int			i;
	int			len;
	const char*	str;

	strcat(page, "ids: ");
	comma	= "";
	len		= strlen(page);

	for (i = 0; i < (int) SIZEOF_ARRAY(gsc_global.dev_list); i++)
	{
		if (gsc_global.dev_list[i])
		{
			// Append the comma.
			len	+= strlen(comma);

			if (len >= _PROC_MSG_SIZE)
				break;

			strcat(page, comma);

			// Append the model string.
			str	= gsc_global.dev_list[i]->proc_id_str;
			len	+= strlen(str);

			if (len >= _PROC_MSG_SIZE)
				break;

			strcat(page, str);
			comma	= ",";
		}
	}

	// Append the line ending.
	len	+= strlen(eol);

	if (len < _PROC_MSG_SIZE)
		strcat(page, "\n");
#endif
}



//*****************************************************************************
static void _proc_add_eof(char* page, int* eof)
{
	// This is the sixth part of the proc file.
	// We know in advance that the buffer might have overrun.

	int	i;

	i	= strlen(page) + 1;

	if (i >= _PROC_MSG_SIZE)
	{
		printf(	"%s: %d. %s: /proc/%s size (%d) is larger than PAGE_SIZE (%d).\n",
				DEV_NAME,
				__LINE__,
				__FUNCTION__,
				DEV_NAME,
				i,
				(int) PAGE_SIZE);
	}

	eof[0]	= 1;
}



/******************************************************************************
*
*	Function:	os_proc_read
*
*	Purpose:
*
*		Implement the read service for the module's /proc file system entry.
*		Read the documentation on this service for details, as we ignore some
*		arguments according to our needs and the documentation.
*
*	Arguments:
*
*		page	The data produced is put here.
*
*		start	Records pointer to where the data in "page" begins.
*
*		offset	The offset into the file where we're to begin.
*
*		count	The limit to the amount of data we can produce.
*
*		eof		Set this flag when we hit EOF.
*
*		data	A private data item that we may use, but don't.
*
*	Returned:
*
*		int		The number of characters written.
*
******************************************************************************/
int os_proc_read(
	char*	page,
	char**	start,
	off_t	offset,
	int		count,
	int*	eof,
	void*	data)
{
	int	i;

	for (;;)	// A convenience loop.
	{
		i	= os_module_count_inc();

		if (i)
		{
			i	= 0;
			break;
		}

		_proc_add_version(page);
		_proc_add_32_bit_support(page);
		_proc_add_board_count(page);
		_proc_add_models(page);
		_proc_add_id_strs(page);
		_proc_add_eof(page, eof);

		i	= strlen(page);

		os_module_count_dec();
		break;
	}

	return(i);
}



/******************************************************************************
*
*	Function:	os_proc_start
*
*	Purpose:
*
*		initialize use of the /proc file system.
*
*	Arguments:
*
*		None.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int os_proc_start(void)
{
	int	ret;

	if (gsc_global.dev_qty)
		ret	= os_proc_start_detail();
	else
		ret	= 0;

	return(ret);
}



/******************************************************************************
*
*	Function:	os_proc_stop
*
*	Purpose:
*
*		Stop use of the /proc file system.
*
*	Arguments:
*
*		None.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void os_proc_stop(void)
{
	if (gsc_global.proc_enabled)
	{
		remove_proc_entry(DEV_NAME, NULL);
		gsc_global.proc_enabled	= 0;
	}
}


