// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AICS32/driver/main.h $
// $Rev: 56668 $
// $Date: 2025-09-19 12:14:55 -0500 (Fri, 19 Sep 2025) $

// 16AICS32: Device Driver: header file

#ifndef	__MAIN_H__
#define	__MAIN_H__

#include "16aics32.h"

#define	DEV_BAR_SHOW						0
#define	DEV_METRICS_SHOW					0
#define	DEV_PCI_ID_SHOW						0
#define	DEV_SUPPORTS_IRQ
#define	DEV_SUPPORTS_READ
#define	DEV_SUPPORTS_WAIT
#define	GSC_ALT_DEV_GET(a)					(a)
#define	GSC_ALT_STRUCT_GET(i,d)				(d)
#define	GSC_ALT_STRUCT_T					dev_data_t
#define	GSC_DEVS_PER_BOARD					1

typedef struct _dev_data_t					dev_data_t;

#include "gsc_main.h"



// macros *********************************************************************

#define	DEV_MODEL							"16AICS32"	// Upper case form of the below.
#define	DEV_NAME							"16aics32"	// MUST AGREE WITH AICS32_BASE_NAME

#define	DEV_VERSION							"2.5"		// FOR DEVICE SPECIFIC CODE ONLY!
// 2.5	Updated to support the 6.x series kernel.
//		Updated for Fedora 38.
//		Renamed all Auto_Calibration content to Autocal.
//		Renamed all Auto_Cal_Sts content to Autocal_Status
//		Renamed all Auto_Cal content to Autocal.
//		Updated the Initialize IOCTL service for consistency.
//		Updated the Autocal IOCTL service for consistency.
//		Updated the make file for consistency.
// 2.4	The AICS32_QUERY_EXCITATION query feature is now inactive.
//		Updated comments for consistency.
//		Updated the Initialize and Autocal IOCTL services for consistency.
//		Removed unused macros.
//		Updated per directory reorganization.
//		Updated per PLX code segregation.
// 2.3	Added support for the 5.x kernel series.
//		Updated for porting to Windows.
//		Standardized various IOCTL system log messages.
//		Standardized processing of the initialization IOCTL service.
//		Standardized processing of the autocalibration IOCTL service.
//		Updated comments for consistency.
// 2.2	Bug fix: was ignoring BAR setup return value.
//		Updated per changes to the OS specific PCI services.
//		Made updates for DEV_SUPPORTS_XXX macro changes.
//		Modified code so open succeeds even if initialization fails.
//		Updated per changes to common I/O code.
// 2.1	The BAR code has been updated to include the data under a single structure.
//		The register definitions have been updated.
//		I/O services now pass around an os_mem_t structure instead of a buffer pointer.
//		White space cleanup.
//		Changed the arbitrary wait event callback argument to a void* type.
//		Modified to allow multiple apps to simultaneously access a single device.
//		Correct access to BAR0 and BAR1 (is RO, was RW).
//		Added an infinite I/O timeout option.
//		Corrected the IOCTL code for the register write and mod services.
// 2.0	Updated to use the newer common driver sources.
//		Removed GNU notice from non-Linux specific files.
//		Removed Linux specific content from non-Linux specific source files.
//		White space cleanup.
//		Now using a spinlock rather than enabling and disabling interrupts.
//		Updated gsc_irq_open() and gsc_irq_close().
//		Updated gsc_dma_open() and gsc_dma_close().
//		Fixed a couple bugs due to missing register declarations.
// 1.4	Reduced #include list in driver interface header.
// 1.3	Updated for the 3.x kernel.
//		Corrected a system time tick count roll over bug in ioctl.c.
// 1.2	BAR0 and BAR2 are now the only BAR regions used.
//		Include all common source, though not all are used.
// 1.1	Added D0-D31 macros to the common source code.
//		Removed warnings for Fedora 15.
//		Eliminated the global dev_check_id() routine.
// 1.0	Initial release.

// I/O services
#define	DEV_IO_STREAM_QTY					(DEV_IO_RX_STREAM_QTY + DEV_IO_TX_STREAM_QTY)
#define	DEV_IO_RX_STREAM_QTY				1
#define	DEV_IO_TX_STREAM_QTY				0

#define	GSC_READ_PIO_WORK_32_BIT

// WAIT services
#define	DEV_WAIT_GSC_ALL					AICS32_WAIT_GSC_ALL
#define	DEV_WAIT_ALT_ALL					AICS32_WAIT_ALT_ALL
#define	DEV_WAIT_IO_ALL						AICS32_WAIT_IO_ALL



// data types *****************************************************************

struct _dev_io_t
{
	// Initialized by open and IOCTL services (initialize and/or service specific).

	s32					io_mode;			// PIO, DMA, DMDMA
	s32					pio_threshold;		// Use PIO if samples <= this.
	s32					timeout_s;			// I/O timeout in seconds.

	// Initialized by I/O service at time of use.

	int					abort;
	gsc_dma_ch_t*		dma_channel;		// Use this channel for DMA.
	int					non_blocking;		// Is this non-blocking I/O?

	// Initialized by device specific I/O create code.

	int					bytes_per_sample;	// Sample size in bytes.
	u32					io_reg_offset;		// Offset of device's I/O FIFO.
	VADDR_T				io_reg_vaddr;		// Address of device's I/O FIFO.

	void				(*dev_io_sw_init)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	void				(*dev_io_close)			(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	void				(*dev_io_open)			(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	int					(*dev_io_startup)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io);
	long				(*dev_pio_available)	(GSC_ALT_STRUCT_T* alt, dev_io_t* io, size_t count);
	long				(*dev_bmdma_available)	(GSC_ALT_STRUCT_T* alt, dev_io_t* io, size_t count);
	long				(*dev_dmdma_available)	(GSC_ALT_STRUCT_T* alt, dev_io_t* io, size_t count);
	long				(*dev_pio_xfer)			(GSC_ALT_STRUCT_T* alt, dev_io_t* io, const os_mem_t* mem, size_t count, os_time_tick_t st_end);
	long				(*dev_bmdma_xfer)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io, const os_mem_t* mem, size_t count, os_time_tick_t st_end);
	long				(*dev_dmdma_xfer)		(GSC_ALT_STRUCT_T* alt, dev_io_t* io, const os_mem_t* mem, size_t count, os_time_tick_t st_end);

	struct
	{
		u32				abort;			// ...WAIT_IO_XXX_ABORT
		u32				done;			// ...WAIT_IO_XXX_DONE
		u32				error;			// ...WAIT_IO_XXX_ERROR
		u32				timeout;		// ...WAIT_IO_XXX_TIMEOUT
	} wait;

	// Initialize by GSC commone create code.

	os_sem_t			sem;			// Only one Tx or Rx at a time.
	os_mem_t			mem;			// I/O buffer.
};

struct _dev_data_t
{
	os_pci_t			pci;			// The kernel PCI device descriptor.
	os_data_t			os;				// OS specific data.
	os_spinlock_t		spinlock;		// Control ISR access.
	os_sem_t			sem;			// Control thread access.
	gsc_dev_type_t		board_type;		// Corresponds to basic device type.
	const char*			model;			// Base model number in upper case.
	int					board_index;	// Index of the device being accessed.
	int					users;			// Number of currently active open() requests.
	int					share;			// Were we opened in shared mode?

	gsc_bar_t			bar;			// device register mappings
	gsc_dma_t			dma;			// For DMA based I/O.
	gsc_irq_t			irq;			// For interrut support.
	gsc_wait_node_t*	wait_list;

	struct
	{					// This is for streaming I/O
		dev_io_t		rx;				// Analog Input read
		dev_io_t*		io_streams[DEV_IO_STREAM_QTY];
	} io;

	struct
	{
		VADDR_T			plx_intcsr_32;	// Interrupt Control/Status Register
		VADDR_T			plx_dmaarb_32;	// DMA Arbitration Register
		VADDR_T			plx_dmathr_32;	// DMA Threshold Register

		VADDR_T			gsc_bctlr_32;	// 0x00 Board Control Register
		VADDR_T			gsc_icr_32;		// 0x04 Interrupt Control Register
		VADDR_T			gsc_ibdr_32;	// 0x08	Input Data Buffer Register
		VADDR_T			gsc_ibcr_32;	// 0x0C Input Buffer Control Register
		VADDR_T			gsc_ragr_32;	// 0x10 Rate-A Generator Register
		VADDR_T			gsc_rbgr_32;	// 0x14 Rate-B Generator Register
		VADDR_T			gsc_bsizr_32;	// 0x18 Buffer Size Register

		VADDR_T			gsc_sscr_32;	// 0x20 Scan & Sync Control Register

		VADDR_T			gsc_bcfgr_32;	// 0x28 Board Configuration Register

		VADDR_T			gsc_em00r_32;	// 0x30 Excitation Mask: bits 0-15
		VADDR_T			gsc_em16r_32;	// 0x34 Excitation Mask: bits 16-31

	} vaddr;

	struct
	{
		s32				autocal_ms;			// Maximum ms for autocalibrate

		s32				channel_qty;		// The number of channels on the device.
		s32				channels_max;		// Maximum channels supported by model.

		s32				excitation_current;	// Excitation Current

		s32				fgen_max;			// Rate Generator maximum output rate.
		s32				fgen_min;			// Rate Generator minimum output rate.
		u32				fifo_size;			// Size of FIFO - not the fill level.
		s32				fsamp_max;			// The maximum Fsamp rate per channel.
		s32				fsamp_min;			// The minimum Fsamp rate per channel.

		u32				gsc_bcfgr_32;		// Board Configuration Register

		s32				initialize_ms;		// Maximum ms for initialize

		s32				master_clock;		// Master clock frequency

		s32				nrate_max;			// Minimum Nrate value.
		s32				nrate_min;			// Maximum Nrate value.

		s32				rate_gen_qty;		// The number of rate generators on device.

	} cache;
};



// prototypes *****************************************************************

void		dev_io_close(dev_data_t* dev);
int			dev_io_create(dev_data_t* dev);
void		dev_io_destroy(dev_data_t* dev);
int			dev_io_open(dev_data_t* dev);
dev_io_t*	dev_io_read_select(dev_data_t* dev, size_t count);
int			dev_irq_create(dev_data_t* dev);
void		dev_irq_destroy(dev_data_t* dev);
int			dev_read_create(dev_data_t* dev, dev_io_t* io);

int			initialize_ioctl(dev_data_t* dev, void* arg);



#endif
