// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AICS32/driver/ioctl.c $
// $Rev: 56668 $
// $Date: 2025-09-19 12:14:55 -0500 (Fri, 19 Sep 2025) $

// 16AICS32: Device Driver: source file

#include "main.h"



// macros *********************************************************************

#define	AUTOCAL_START			D13		// BCTLR
#define	AUTOCAL_PASS			D14		// BCTLR
#define	INIT_START				D15		// BCTLR

#define	IRQ0_INIT_DONE			0x0000	// ICR
#define	IRQ0_AUTOCAL_DONE		0x0001	// ICR
#define	IRQ0_MASK				0x0007	// ICR
#define	IRQ0_ACTIVE				0x0008	// ICR
#define	IRQ1_ACTIVE				0x0080	// ICR



//*****************************************************************************
static int _query(dev_data_t* dev, s32* arg)
{
	switch (arg[0])
	{
		default:						arg[0]	= AICS32_IOCTL_QUERY_ERROR;			break;
		case AICS32_QUERY_AUTOCAL_MS:	arg[0]	= dev->cache.autocal_ms;			break;
		case AICS32_QUERY_CHANNEL_MAX:	arg[0]	= dev->cache.channels_max;			break;
		case AICS32_QUERY_CHANNEL_QTY:	arg[0]	= dev->cache.channel_qty;			break;
		case AICS32_QUERY_COUNT:		arg[0]	= AICS32_QUERY_COUNT + 1;			break;
		case AICS32_QUERY_DEVICE_TYPE:	arg[0]	= dev->board_type;					break;
		case AICS32_QUERY_EXCITATION:	arg[0]	= dev->cache.excitation_current;	break;
		case AICS32_QUERY_FGEN_MAX:		arg[0]	= dev->cache.fgen_max;				break;
		case AICS32_QUERY_FGEN_MIN:		arg[0]	= dev->cache.fgen_min;				break;
		case AICS32_QUERY_FIFO_SIZE:	arg[0]	= dev->cache.fifo_size;				break;
		case AICS32_QUERY_FSAMP_MAX:	arg[0]	= dev->cache.fsamp_max;				break;
		case AICS32_QUERY_FSAMP_MIN:	arg[0]	= dev->cache.fsamp_min;				break;
		case AICS32_QUERY_INIT_MS:		arg[0]	= dev->cache.initialize_ms;			break;
		case AICS32_QUERY_MASTER_CLOCK:	arg[0]	= dev->cache.master_clock;			break;
		case AICS32_QUERY_NRATE_MAX:	arg[0]	= dev->cache.nrate_max;				break;
		case AICS32_QUERY_NRATE_MIN:	arg[0]	= dev->cache.nrate_min;				break;
		case AICS32_QUERY_RATE_GEN_QTY:	arg[0]	= dev->cache.rate_gen_qty;			break;
	}

	return(0);
}



//*****************************************************************************
static int _init_start(dev_data_t* dev, void* arg)
{
	// Initiate initialization.
	os_reg_mem_tx_u32(NULL, dev->vaddr.gsc_bctlr_32, INIT_START);
	return(0);
}



//*****************************************************************************
int initialize_ioctl(dev_data_t* dev, void* arg)
{
	int			i;
	int			mask;
	long		ms			= dev->cache.initialize_ms + 5000;
	long		ms_total	= ms;
	u32			reg;
	int			ret			= 0;
	os_sem_t	sem;
	int			tmp;
	VADDR_T		va			= dev->vaddr.gsc_icr_32;
	int			value;
	gsc_wait_t	wt;

	if ((dev->irq.opened) && (gsc_global.driver_unloading == 0))
	{
		ms_total	*= 2;

		// Safely select the Initialize Done interrupt.
		mask	= IRQ0_MASK | IRQ0_ACTIVE | IRQ1_ACTIVE;
		value	= IRQ0_INIT_DONE;
		os_reg_mem_mx_u32(dev, va, value, mask);

		// Wait for the local interrupt.
		os_sem_create(&sem);	// dummy, required for wait operations.
		memset(&wt, 0, sizeof(wt));
		wt.flags		= GSC_WAIT_FLAG_INTERNAL;
		wt.gsc			= AICS32_WAIT_GSC_INIT_DONE;
		wt.timeout_ms	= ms;
		ret				= gsc_wait_event(dev, &wt, _init_start, NULL, &sem);
		os_sem_destroy(&sem);

		if (wt.flags & GSC_WAIT_FLAG_TIMEOUT)
		{
			ret	= ret ? ret : -ETIMEDOUT;
			printf(	"%s: INITILIZE DONE IRQ TIMED OUT AFTER %ld ms.\n",
					dev->model,
					ms);
		}
	}
	else
	{
		_init_start(dev, NULL);
	}

	// Manually wait for completion as the IRQ wait may have ended early.
	va	= dev->vaddr.gsc_bctlr_32;
	tmp	= gsc_poll_u32(dev, ms, va, INIT_START, 0);

	if (tmp)
	{
		ret	= ret ? ret : tmp;
		printf(	"%s: INITIALIZATION DID NOT COMPLETE WITHIN %ld ms.\n",
				dev->model,
				ms_total);
	}

	// Manually check for completion as the IRQ wait may have ended early.
	reg	= os_reg_mem_rx_u32(dev, va);

	if (reg & INIT_START)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: INITILIZE STILL ACTIVE AFTER %ld ms.\n",
				dev->model,
				ms_total);
	}

	// Initialize the software settings.

	for (i = 0; i < DEV_IO_STREAM_QTY; i++)
	{
		if (dev->io.io_streams[i])
		{
			if (dev->io.io_streams[i]->dev_io_sw_init)
			{
				(dev->io.io_streams[i]->dev_io_sw_init)(dev, dev->io.io_streams[i]);
			}
		}
	}

	return(ret);
}



//*****************************************************************************
static int _autocal_start(dev_data_t* dev, void* arg)
{
	// Initiate autocalibration.
	os_reg_mem_mx_u32(NULL, dev->vaddr.gsc_bctlr_32, AUTOCAL_START, AUTOCAL_START);
	return(0);
}



//*****************************************************************************
static int _autocal(dev_data_t* dev, void* arg)
{
	u32			mask;
	long		ms		= dev->cache.autocal_ms + 5000;
	u32			reg;
	int			ret;
	os_sem_t	sem;
	int			tmp;
	VADDR_T		va		= dev->vaddr.gsc_icr_32;
	u32			value;
	gsc_wait_t	wt;

	// Safely select the Autocal Done interrupt.
	mask	= IRQ0_MASK | IRQ0_ACTIVE | IRQ1_ACTIVE;
	value	= IRQ0_AUTOCAL_DONE;
	os_reg_mem_mx_u32(dev, va, value, mask);

	// Wait for the local interrupt.
	os_sem_create(&sem);	// dummy, required for wait operations.
	memset(&wt, 0, sizeof(wt));
	wt.flags		= GSC_WAIT_FLAG_INTERNAL;
	wt.gsc			= AICS32_WAIT_GSC_AUTOCAL_DONE;
	wt.timeout_ms	= ms;
	ret				= gsc_wait_event(dev, &wt, _autocal_start, NULL, &sem);
	os_sem_destroy(&sem);

	if (wt.flags & GSC_WAIT_FLAG_TIMEOUT)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATE DONE IRQ TIMED OUT AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually wait for completion in case something terminates our wait early.
	va	= dev->vaddr.gsc_bctlr_32;
	tmp	= gsc_poll_u32(dev, ms, va, AUTOCAL_START, 0);

	if (tmp)
	{
		ms	*= 2;
		ret	= ret ? ret : tmp;
		printf(	"%s: AUTOCALIBRATION DID NOT COMPLETE WITHIN %ld ms.\n",
				dev->model,
				ms);
	}

	// Manually check for completion as the IRQ wait may have ended early.
	reg	= os_reg_mem_rx_u32(dev, va);

	if (reg & AUTOCAL_START)
	{
		ret	= ret ? ret : -ETIMEDOUT;
		printf(	"%s: AUTOCALIBRATION STILL ACTIVE AFTER %ld ms.\n",
				dev->model,
				ms);
	}

	// Final results.
	reg	= os_reg_mem_rx_u32(dev, va);

	if ((reg & AUTOCAL_PASS) == 0)
	{
		ret	= ret ? ret : -EIO;
		printf(	"%s: AUTOCALIBRATION FAILED (%ld ms).\n",
				dev->model,
				ms);
	}

	if (ret == 0)
	{
		// Wait for settling.
		os_time_sleep_ms(100);
	}

	return(ret);
}



//*****************************************************************************
static int _autocal_status(dev_data_t* dev, s32* arg)
{
	u32	reg;

	reg	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bctlr_32);

	if (reg & AUTOCAL_START)
		arg[0]	= AICS32_AUTOCAL_STATUS_ACTIVE;
	else if ((reg & AUTOCAL_PASS) == 0)
		arg[0]	= AICS32_AUTOCAL_STATUS_FAIL;
	else
		arg[0]	= AICS32_AUTOCAL_STATUS_PASS;

	return(0);
}



//*****************************************************************************
static int _ai_buf_clear(dev_data_t* dev, void* arg)
{
	int	ret;

	// This action clears the buffer and the over and under run flags.
	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_ibcr_32, D16, D16);

	// The bit should clear within a short period of time.
	ret	= gsc_poll_u32(dev, 2, dev->vaddr.gsc_ibcr_32, D16, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_level(dev_data_t* dev, s32* arg)
{
	arg[0]	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bsizr_32);
	arg[0]	&= 0xFFFF;
	return(0);
}



//*****************************************************************************
static int _ai_buf_thr_lvl(dev_data_t* dev, void* arg)
{
	int	ret;

	ret	= gsc_s32_range_reg(dev, arg, 0, 0xFFFF, dev->vaddr.gsc_ibcr_32, 15, 0);
	return(ret);
}



//*****************************************************************************
static int _ai_buf_thr_sts(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AICS32_AI_BUF_THR_STS_CLEAR,
		AICS32_AI_BUF_THR_STS_SET,
		-1	// terminate list
	};

	int	ret;

	arg[0]	= -1;
	ret		= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_ibcr_32, 17, 17);
	return(ret);
}



//*****************************************************************************
static int _ai_clk_src(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AICS32_AI_CLK_SRC_RAG,
		AICS32_AI_CLK_SRC_RBG,
		AICS32_AI_CLK_SRC_EXT,
		AICS32_AI_CLK_SRC_BCR,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_sscr_32, 4, 3);
	return(ret);
}



//*****************************************************************************
static int _ai_mode(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AICS32_AI_MODE_DIFF,
		AICS32_AI_MODE_SE,
		AICS32_AI_MODE_ZERO,
		AICS32_AI_MODE_VREF,
		-1	// terminate list
	};

	int	ret;
	s32	v	= arg[0];

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 2, 0);

	if ((v != -1) && (ret == 0))
		os_time_sleep_ms(100);

	return(ret);
}



//*****************************************************************************
static int _ai_range(dev_data_t* dev, s32* arg)
{
	static const s32	options[]	=
	{
		AICS32_AI_RANGE_2_5V,
		AICS32_AI_RANGE_5V,
		AICS32_AI_RANGE_10V,
		-1	// terminate list
	};

	int	ret;
	s32	v	= arg[0];

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 5, 4);

	if ((v != -1) && (ret == 0))
		os_time_sleep_ms(100);

	return(ret);
}



//*****************************************************************************
static int _data_format(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AICS32_DATA_FORMAT_2S_COMP,
		AICS32_DATA_FORMAT_OFF_BIN,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 6, 6);
	return(ret);
}



//*****************************************************************************
static int _excitation_mask_get(dev_data_t* dev, u32* arg)
{
	u32	high;
	u32	low;

	low		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_em00r_32);
	high	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_em16r_32);
	arg[0]	= (low & 0xFFFF) | ((high & 0xFFFF) << 16);
	return(0);
}



//*****************************************************************************
static int _excitation_mask_set(dev_data_t* dev, s32* arg)
{
	u32	high	= 0xFFFF & (arg[0] >> 16);
	u32	low		= 0xFFFF & arg[0];

	os_reg_mem_tx_u32(dev, dev->vaddr.gsc_em00r_32, low);
	os_reg_mem_tx_u32(dev, dev->vaddr.gsc_em16r_32, high);
	return(0);
}



//*****************************************************************************
static int _excitation_test(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AICS32_EXCITATION_TEST_OFF,
		AICS32_EXCITATION_TEST_ON,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 8, 8);
	return(ret);
}



//*****************************************************************************
static int _ext_sync(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AICS32_EXT_SYNC_DISABLE,
		AICS32_EXT_SYNC_ENABLE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_bctlr_32, 7, 7);
	return(ret);
}



//*****************************************************************************
static int _input_sync(dev_data_t* dev, void* arg)
{
	unsigned long	ms_limit;
	int				ret;

	os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bctlr_32, D12, D12);

	// Wait for the operation to complete.

	if (dev->io.rx.timeout_s)
		ms_limit	= dev->io.rx.timeout_s * 1000;
	else
		ms_limit	= 1000;

	ret	= gsc_poll_u32(dev, ms_limit, dev->vaddr.gsc_bctlr_32, D12, 0);
	return(ret);
}



//*****************************************************************************
static int _irq0_sel(dev_data_t* dev, s32* arg)
{
	u32	reg;
	int	ret	= 0;

	if (arg[0] == -1)
	{
		// Retrieve the current setting.
		reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_icr_32);
		arg[0]	= GSC_FIELD_DECODE(reg, 2, 0);
	}
	else
	{
		// Validate the option value passed in.

		switch (arg[0])
		{
			default:

				ret	= -EINVAL;
				break;

			case AICS32_IRQ0_INIT_DONE:
			case AICS32_IRQ0_AUTOCAL_DONE:
			case AICS32_IRQ0_SCAN_START:
			case AICS32_IRQ0_SCAN_DONE:

				break;
		}

		if (ret == 0)
		{
			// Clear the status bit and apply the new setting.
			// Do not clear the IRQ1 status bit.
			os_reg_mem_mx_u32(dev, dev->vaddr.gsc_icr_32, arg[0], 0xF);
		}
	}

	return(ret);
}



//*****************************************************************************
static int _irq1_sel(dev_data_t* dev, s32* arg)
{
	u32	reg;
	int	ret	= 0;

	if (arg[0] == -1)
	{
		// Retrieve the current setting.
		reg		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_icr_32);
		arg[0]	= GSC_FIELD_DECODE(reg, 6, 4);
	}
	else
	{
		// Validate the option value passed in.

		switch (arg[0])
		{
			default:

				ret	= -EINVAL;
				break;

			case AICS32_IRQ1_NONE:
			case AICS32_IRQ1_IN_BUF_THR_L2H:
			case AICS32_IRQ1_IN_BUF_THR_H2L:

				break;
		}

		if (ret == 0)
		{
			// Clear the status bit and apply the new setting.
			// Do not clear the IRQ0 status bit.
			os_reg_mem_mx_u32(dev, dev->vaddr.gsc_icr_32, arg[0] << 4, 0xF0);
		}
	}

	return(ret);
}



//*****************************************************************************
static int _rag_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AICS32_GEN_ENABLE_NO,
		AICS32_GEN_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_ragr_32, 16, 16);
	return(ret);
}



//*****************************************************************************
static int _rag_nrate(dev_data_t* dev, void* arg)
{
	int	ret;
	s32	v	= ((s32*) arg)[0];

	ret	= gsc_s32_range_reg(dev, arg, 250, 0xFFFF, dev->vaddr.gsc_ragr_32, 15, 0);

	if ((v != -1) && (ret == 0))
		os_time_sleep_ms(100);

	return(ret);
}



//*****************************************************************************
static int _rbg_clk_src(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AICS32_RBG_CLK_SRC_MASTER,
		AICS32_RBG_CLK_SRC_RAG,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_sscr_32, 10, 10);
	return(ret);
}



//*****************************************************************************
static int _rbg_enable(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AICS32_GEN_ENABLE_NO,
		AICS32_GEN_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_rbgr_32, 16, 16);
	return(ret);
}



//*****************************************************************************
static int _rbg_nrate(dev_data_t* dev, void* arg)
{
	int	ret;
	s32	v	= ((s32*) arg)[0];

	ret	= gsc_s32_range_reg(dev, arg, 250, 0xFFFF, dev->vaddr.gsc_rbgr_32, 15, 0);

	if ((v != -1) && (ret == 0))
		os_time_sleep_ms(100);

	return(ret);
}



//*****************************************************************************
static int _rx_io_abort(dev_data_t* dev, s32* arg)
{
	arg[0]	= gsc_read_abort_active_xfer(dev, &dev->io.rx);
	return(0);
}



//*****************************************************************************
static int _rx_io_mode(dev_data_t* dev, s32* arg)
{
	static const s32	list[]	=
	{
		GSC_IO_MODE_PIO,
		GSC_IO_MODE_BMDMA,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &dev->io.rx.io_mode);
	return(ret);
}



//*****************************************************************************
static int _rx_io_timeout(dev_data_t* dev, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_var(
			arg,
			AICS32_IO_TIMEOUT_MIN,
			AICS32_IO_TIMEOUT_INFINITE,
			&dev->io.rx.timeout_s);
	return(ret);
}



//*****************************************************************************
static int _scan_single(dev_data_t* dev, void* arg)
{
	u32	last;
	int	ret;

	last	= dev->cache.channel_qty - 1;
	ret		= gsc_s32_range_reg(dev, arg, 0, last, dev->vaddr.gsc_sscr_32, 17, 12);
	return(ret);
}



//*****************************************************************************
static int _scan_size(dev_data_t* dev, void* arg)
{
	static const s32	options[]	=
	{
		AICS32_SCAN_SIZE_SINGLE,
		AICS32_SCAN_SIZE_0_1,
		AICS32_SCAN_SIZE_0_3,
		AICS32_SCAN_SIZE_0_7,
		AICS32_SCAN_SIZE_0_15,
		AICS32_SCAN_SIZE_0_31,
		AICS32_SCAN_SIZE_0_63,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_reg(dev, arg, options, dev->vaddr.gsc_sscr_32, 2, 0);
	return(ret);
}



// variables ******************************************************************

const gsc_ioctl_t	dev_ioctl_list[]	=
{
	{ AICS32_IOCTL_REG_READ,			(void*) gsc_reg_read_ioctl		},
	{ AICS32_IOCTL_REG_WRITE,			(void*) gsc_reg_write_ioctl		},
	{ AICS32_IOCTL_REG_MOD,				(void*) gsc_reg_mod_ioctl		},
	{ AICS32_IOCTL_QUERY,				(void*) _query					},
	{ AICS32_IOCTL_INITIALIZE,			(void*) initialize_ioctl		},
	{ AICS32_IOCTL_AUTOCAL,				(void*) _autocal				},
	{ AICS32_IOCTL_AUTOCAL_STATUS,		(void*) _autocal_status			},
	{ AICS32_IOCTL_AI_BUF_CLEAR,		(void*) _ai_buf_clear			},
	{ AICS32_IOCTL_AI_BUF_LEVEL,		(void*) _ai_buf_level			},
	{ AICS32_IOCTL_AI_BUF_THR_LVL,		(void*) _ai_buf_thr_lvl			},
	{ AICS32_IOCTL_AI_BUF_THR_STS,		(void*) _ai_buf_thr_sts			},
	{ AICS32_IOCTL_AI_CLK_SRC,			(void*) _ai_clk_src				},
	{ AICS32_IOCTL_AI_MODE,				(void*) _ai_mode				},
	{ AICS32_IOCTL_AI_RANGE,			(void*) _ai_range				},
	{ AICS32_IOCTL_DATA_FORMAT,			(void*) _data_format			},
	{ AICS32_IOCTL_EXCITATION_MASK_GET,	(void*) _excitation_mask_get	},
	{ AICS32_IOCTL_EXCITATION_MASK_SET,	(void*) _excitation_mask_set	},
	{ AICS32_IOCTL_EXCITATION_TEST,		(void*) _excitation_test		},
	{ AICS32_IOCTL_EXT_SYNC,			(void*) _ext_sync				},
	{ AICS32_IOCTL_INPUT_SYNC,			(void*) _input_sync				},
	{ AICS32_IOCTL_IRQ0_SEL,			(void*) _irq0_sel				},
	{ AICS32_IOCTL_IRQ1_SEL,			(void*) _irq1_sel				},
	{ AICS32_IOCTL_RAG_ENABLE,			(void*) _rag_enable				},
	{ AICS32_IOCTL_RAG_NRATE,			(void*) _rag_nrate				},
	{ AICS32_IOCTL_RBG_CLK_SRC,			(void*) _rbg_clk_src			},
	{ AICS32_IOCTL_RBG_ENABLE,			(void*) _rbg_enable				},
	{ AICS32_IOCTL_RBG_NRATE,			(void*) _rbg_nrate				},
	{ AICS32_IOCTL_RX_IO_ABORT,			(void*) _rx_io_abort			},
	{ AICS32_IOCTL_RX_IO_MODE,			(void*) _rx_io_mode				},
	{ AICS32_IOCTL_RX_IO_TIMEOUT,		(void*) _rx_io_timeout			},
	{ AICS32_IOCTL_SCAN_SINGLE,			(void*) _scan_single			},
	{ AICS32_IOCTL_SCAN_SIZE,			(void*) _scan_size				},
	{ AICS32_IOCTL_WAIT_CANCEL,			(void*) gsc_wait_cancel_ioctl	},
	{ AICS32_IOCTL_WAIT_EVENT,			(void*) gsc_wait_event_ioctl	},
	{ AICS32_IOCTL_WAIT_STATUS,			(void*) gsc_wait_status_ioctl	},
	{ -1, NULL }
};


