// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/16AICS32/driver/device.c $
// $Rev: 56668 $
// $Date: 2025-09-19 12:14:55 -0500 (Fri, 19 Sep 2025) $

// 16AICS32: Device Driver: source file

#include "main.h"



// variables ******************************************************************

const gsc_dev_id_t	dev_id_list[]	=
{
	// model		Vendor	Device	SubVen	SubDev	type

	{ "16AICS32",	0x10B5, 0x9080, 0x10B5, 0x3010,	GSC_DEV_TYPE_16AICS32	},
	{ NULL }
};



//*****************************************************************************
static void _excitation_compute(dev_data_t* dev)
{
	static const s32	list[8]	=
	{
		AICS32_QUERY_EXCITATION_1_0_MA,		// 0
		AICS32_QUERY_EXCITATION_2_0_MA,		// 1
		AICS32_QUERY_EXCITATION_3_0_MA,		// 2
		AICS32_QUERY_EXCITATION_5_0_MA,		// 3
		AICS32_QUERY_EXCITATION_0_4_MA,		// 4
		AICS32_QUERY_EXCITATION_10_0_MA,	// 5
		-1,
		-1
	};

	u32	val	= GSC_FIELD_DECODE(dev->cache.gsc_bcfgr_32, 18, 16);

	dev->cache.excitation_current	= list[val];
}



/******************************************************************************
*
*	Function:	dev_device_create
*
*	Purpose:
*
*		Do everything needed to setup and use the given device.
*
*	Arguments:
*
*		dev		The structure to initialize.
*
*	Returned:
*
*		0		All is well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int dev_device_create(dev_data_t* dev)
{
	static const gsc_bar_maps_t	bar_map	=
	{
		{
			// mem	io	rw
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 0: PLX registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 1: PLX registers, I/O mapped
			{ 1,	0,	GSC_REG_TYPE_ACCESS_RW },	// BAR 2: GSC registers, memory mapped
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 3: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO },	// BAR 4: unused
			{ 0,	0,	GSC_REG_TYPE_ACCESS_RO }	// BAR 5: unused
		}
	};

	u32	dma;
	int	ret;

	for (;;)	// A convenience loop.
	{
		// Verify some macro contents.
		ret	= gsc_macro_test_base_name(AICS32_BASE_NAME);
		if (ret)	break;

		ret	= gsc_macro_test_model();
		if (ret)	break;

		// PCI setup.
		ret	= os_pci_dev_enable(&dev->pci);
		if (ret)	break;

		ret	= os_pci_master_set(&dev->pci);
		if (ret)	break;

		// Control ISR access to the device and data structure.
		ret	= os_spinlock_create(&dev->spinlock);
		if (ret)	break;

		// Control access to the device and data structure.
		ret	= os_sem_create(&dev->sem);
		if (ret)	break;

		// Access the BAR regions.
		ret	= gsc_bar_create(dev, &dev->bar, &bar_map);
		if (ret)	break;

		// Firmware access.
		dev->vaddr.gsc_bcfgr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AICS32_GSC_BCFGR));
		dev->vaddr.gsc_bctlr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AICS32_GSC_BCTLR));
		dev->vaddr.gsc_bsizr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AICS32_GSC_BSIZR));
		dev->vaddr.gsc_em00r_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AICS32_GSC_EM00R));
		dev->vaddr.gsc_em16r_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AICS32_GSC_EM16R));
		dev->vaddr.gsc_ibcr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AICS32_GSC_IBCR));
		dev->vaddr.gsc_ibdr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AICS32_GSC_IBDR));
		dev->vaddr.gsc_icr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AICS32_GSC_ICR));
		dev->vaddr.gsc_ragr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AICS32_GSC_RAGR));
		dev->vaddr.gsc_rbgr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AICS32_GSC_RBGR));
		dev->vaddr.gsc_sscr_32		= GSC_VADDR(dev, GSC_REG_OFFSET(AICS32_GSC_SSCR));

		// Data cache initialization.
		dev->cache.gsc_bcfgr_32		= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_bcfgr_32);

		dev->cache.autocal_ms		= 4000;
		dev->cache.channel_qty		= 32;
		dev->cache.channels_max		= 32;
		dev->cache.fgen_max			= 100000L;
		dev->cache.fgen_min			= 1;
		dev->cache.fifo_size		= _64K;
		dev->cache.fsamp_max		= 100000L;
		dev->cache.fsamp_min		= 1;
		dev->cache.initialize_ms	= 3;
		dev->cache.master_clock		= 24000000L;
		dev->cache.nrate_max		= 0xFFFF;
		dev->cache.nrate_min		= 240;
		dev->cache.rate_gen_qty		= 2;

		_excitation_compute(dev);

		// Initialize additional resources.
		ret	= dev_irq_create(dev);
		if (ret)	break;

		ret	= dev_io_create(dev);
		if (ret)	break;

		dma	= GSC_DMA_SEL_STATIC
			| GSC_DMA_CAP_BMDMA_READ;
		ret	= gsc_dma_create(dev, dma, dma);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	dev_device_destroy
*
*	Purpose:
*
*		Do everything needed to release the referenced device.
*
*	Arguments:
*
*		dev		The partial data for the device of interest.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void dev_device_destroy(dev_data_t* dev)
{
	if (dev)
	{
		gsc_dma_destroy(dev);
		dev_io_destroy(dev);
		dev_irq_destroy(dev);
		gsc_bar_destroy(&dev->bar);
		os_sem_destroy(&dev->sem);
		os_spinlock_destroy(&dev->spinlock);
		os_pci_master_clear(&dev->pci);
		os_pci_dev_disable(&dev->pci);
	}
}


